package net.plsar;

import net.plsar.environments.Environments;
import net.plsar.model.*;
import net.plsar.schemes.RenderingScheme;
import net.plsar.resources.*;
import net.plsar.security.SecurityManager;
import net.plsar.security.SecurityAccess;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Logger;

public class PLSAR {

    static Logger Log = Logger.getLogger(PLSAR.class.getName());

    Integer port;
    String PROPERTIES;
    String RENDERING_SCHEME;
    Integer STARTUP_EXECUTORS;

    ViewConfig viewConfig;
    SchemaConfig schemaConfig;
    PropertiesConfig propertiesConfig;
    Integer numberOfPartitions = 3;
    Integer numberOfRequestExecutors = 7;
    PersistenceConfig persistenceConfig;
    Class<?> securityAccessKlass;
    List<Class<?>> viewRenderers;


    public PLSAR(int port){
        this.port = port;
        this.viewConfig = new ViewConfig();
        this.viewRenderers = new ArrayList<>();
        this.PROPERTIES = "system.properties";
        this.STARTUP_EXECUTORS = 19;
        this.RENDERING_SCHEME = RenderingScheme.CACHE_REQUESTS;
    }

    public void start(){
        try {
            Integer TOTAL_NUMBER_EXECUTORS = numberOfPartitions * numberOfRequestExecutors;

            if(schemaConfig != null &&
                    schemaConfig.getEnvironment().equals(Environments.DEVELOPMENT)) {
                DatabaseEnvironmentManager databaseEnvironmentManager = new DatabaseEnvironmentManager();
                databaseEnvironmentManager.configure(schemaConfig, persistenceConfig);
            }

            ServerResources serverResources = new ServerResources();
            StartupAnnotationInspector startupAnnotationInspector = new StartupAnnotationInspector(new ComponentsHolder());
            startupAnnotationInspector.inspect();
            ComponentsHolder componentsHolder = startupAnnotationInspector.getComponentsHolder();

            if(propertiesConfig == null){
                propertiesConfig = new PropertiesConfig();
                propertiesConfig.setPropertiesFile(PROPERTIES);
            }

            String propertiesFile = propertiesConfig.getPropertiesFile();
            RouteAttributesResolver routeAttributesResolver = new RouteAttributesResolver(propertiesFile);
            RouteAttributes routeAttributes = routeAttributesResolver.resolve();
            AnnotationComponent serverStartup = componentsHolder.getServerStartup();

            String resourcesDirectory = viewConfig.getResourcesPath();
            ConcurrentMap<String, byte[]> viewBytesMap = serverResources.getViewBytesMap(viewConfig);

            Log.info("Running startup routine, please wait...");
            if(serverStartup != null) {
                Method startupMethod = serverStartup.getKlass().getMethod("startup");
                startupMethod.invoke(serverResources.getInstance(serverStartup.getKlass()));
            }

            Log.info("Registering route negotiators, please wait...\n");

            BigDecimal ROUTE_NEGOTIATOR_INTERRUPT_DECIMAL = new BigDecimal(TOTAL_NUMBER_EXECUTORS).divide(new BigDecimal(STARTUP_EXECUTORS), 0, RoundingMode.HALF_UP);
            Integer ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT = ROUTE_NEGOTIATOR_INTERRUPT_DECIMAL.intValue();
            BigDecimal ITERATION_STUFF = new BigDecimal(TOTAL_NUMBER_EXECUTORS).divide(ROUTE_NEGOTIATOR_INTERRUPT_DECIMAL, 0, RoundingMode.HALF_UP);
            Integer ITERATION_COMPONENT = ITERATION_STUFF.intValue();
            List<RouteNegotiator> routeNegotiators = new ArrayList<>();
            
            Integer REGISTRY_INCREMENT = 0;
            NegotiatorRegistryExecutor negotiatorRegistryExecutor = null;
            for(int abc = 0; abc < ITERATION_COMPONENT; abc++){
                REGISTRY_INCREMENT+=ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT;
                negotiatorRegistryExecutor = new NegotiatorRegistryExecutor(REGISTRY_INCREMENT, ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT, TOTAL_NUMBER_EXECUTORS, serverResources, persistenceConfig, routeAttributes, securityAccessKlass);
                negotiatorRegistryExecutor.run();
                List<RouteNegotiator> routeNegotiatorsPartial = negotiatorRegistryExecutor.getRouteNegotiators();
                routeNegotiators.addAll(routeNegotiatorsPartial);
;           }

            ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry = registerRouteDirectors(routeNegotiators);
            RedirectRegistry redirectRegistry = new RedirectRegistry();

            ServerSocket serverSocket = new ServerSocket(port);
            serverSocket.setPerformancePreferences(0, 1, 2);
            ExecutorService executors = Executors.newFixedThreadPool(numberOfPartitions);
            executors.execute(new PartitionedExecutor(RENDERING_SCHEME, numberOfRequestExecutors, resourcesDirectory, viewBytesMap, serverSocket, redirectRegistry, routeDirectorRegistry, viewRenderers));

            negotiatorRegistryExecutor.join();

            System.out.println("\n");
            Log.info("Ready!");

        }catch(IOException | NoSuchMethodException | InvocationTargetException | IllegalAccessException | InstantiationException | PlsarException | InterruptedException ex){
            ex.printStackTrace();
        }
    }

    public class NegotiatorRegistryExecutor extends Thread {

        Integer REGISTRY_INCREMENT;
        Integer TOTAL_NUMBER_EXECUTORS;
        Integer ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT;
        Class<?> securityAccessKlass;
        PersistenceConfig persistenceConfig;
        ServerResources serverResources;
        RouteAttributes routeAttributes;
        List<RouteNegotiator> routeNegotiators;

        public NegotiatorRegistryExecutor(Integer REGISTRY_INCREMENT, Integer ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT, Integer TOTAL_NUMBER_EXECUTORS, ServerResources serverResources, PersistenceConfig persistenceConfig, RouteAttributes routeAttributes, Class<?> securityAccessKlass){
            this.TOTAL_NUMBER_EXECUTORS = TOTAL_NUMBER_EXECUTORS;
            this.REGISTRY_INCREMENT = REGISTRY_INCREMENT;
            this.ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT = ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT;
            this.serverResources = serverResources;
            this.persistenceConfig = persistenceConfig;
            this.routeAttributes = routeAttributes;
            this.securityAccessKlass = securityAccessKlass;
        }

        @Override
        public void run() {
            try {
                routeNegotiators = new ArrayList<>();
                for(int xyz = 0; xyz < ROUTE_NEGOTIATORS_ITERABLE_INTERRUPT; xyz++) {

                    RouteEndpointsResolver routeEndpointsResolver = new RouteEndpointsResolver(serverResources);
                    RouteEndpointHolder routeEndpointHolder = routeEndpointsResolver.resolve();
                    routeAttributes.setRouteEndpointHolder(routeEndpointHolder);

                    if (persistenceConfig != null) {
                        PersistenceConfig persistenceConfig = new PersistenceConfig();
                        persistenceConfig.setDriver(this.persistenceConfig.getDriver());
                        persistenceConfig.setUrl(this.persistenceConfig.getUrl());
                        persistenceConfig.setUser(this.persistenceConfig.getUser());
                        persistenceConfig.setConnections(this.persistenceConfig.getConnections());
                        persistenceConfig.setPassword(this.persistenceConfig.getPassword());
                        routeAttributes.setPersistenceConfig(this.persistenceConfig);
                    }

                    if (securityAccessKlass != null) {
                        Dao dao = new Dao(persistenceConfig);
                        SecurityAccess securityAccessInstance = (SecurityAccess) securityAccessKlass.getConstructor().newInstance();
                        Method setPersistence = securityAccessInstance.getClass().getMethod("setDao", Dao.class);
                        setPersistence.invoke(securityAccessInstance, dao);
                        SecurityManager securityManager = new SecurityManager(securityAccessInstance);
                        routeAttributes.setSecurityManager(securityManager);
                        routeAttributes.setSecurityAccess(securityAccessKlass);
                    }

                    ComponentAnnotationInspector componentAnnotationInspector = new ComponentAnnotationInspector(new ComponentsHolder());
                    componentAnnotationInspector.inspect();
                    ComponentsHolder componentsHolder = componentAnnotationInspector.getComponentsHolder();

                    RouteNegotiator routeNegotiator = new RouteNegotiator();
                    routeNegotiator.setRouteAttributes(routeAttributes);
                    routeNegotiator.setComponentsHolder(componentsHolder);
                    routeNegotiators.add(routeNegotiator);
                }

                BigDecimal percentDecimal = new BigDecimal(REGISTRY_INCREMENT).divide(new BigDecimal(TOTAL_NUMBER_EXECUTORS), 2, RoundingMode.HALF_UP);
                Long percent = percentDecimal.multiply(new BigDecimal(100)).longValue();
                if(REGISTRY_INCREMENT >= TOTAL_NUMBER_EXECUTORS)REGISTRY_INCREMENT = TOTAL_NUMBER_EXECUTORS;
                System.out.print(REGISTRY_INCREMENT + "/" + TOTAL_NUMBER_EXECUTORS + " [ " + percent + "% ] network executors registered.\r");

            } catch (InvocationTargetException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            }
        }

        public List<RouteNegotiator> getRouteNegotiators(){
            return this.routeNegotiators;
        }

    }

    ConcurrentMap<String, RouteNegotiator> registerRouteDirectors(List<RouteNegotiator> routeNegotiators) {
        ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry = new ConcurrentHashMap<>(0, 3, 63010);
        for(RouteNegotiator routeNegotiator : routeNegotiators){
            routeDirectorRegistry.put(routeNegotiator.getGuid(), routeNegotiator);
        }
        return routeDirectorRegistry;
    }

    public static class PartitionedExecutor implements Runnable {
        String guid;
        String RENDERER;
        String resourcesDirectory;
        Integer numberOfExecutors;
        ServerSocket serverSocket;
        ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry;
        RedirectRegistry redirectRegistry;
        List<Class<?>> viewRenderers;
        ConcurrentMap<String, String> sessionRouteRegistry;
        ConcurrentMap<String, byte[]> viewBytesMap;
        ConcurrentMap<String, RedirectInfo> initialsRegistry;

        public PartitionedExecutor(String RENDERER, Integer numberOfExecutors, String resourcesDirectory, ConcurrentMap<String, byte[]> viewBytesMap, ServerSocket serverSocket, RedirectRegistry redirectRegistry, ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry, List<Class<?>> viewRenderers) {
            Random random = new Random();
            this.RENDERER = RENDERER;
            this.viewBytesMap = viewBytesMap;
            this.numberOfExecutors = numberOfExecutors;
            this.serverSocket = serverSocket;
            this.redirectRegistry = redirectRegistry;;
            this.routeDirectorRegistry = routeDirectorRegistry;
            this.viewRenderers = viewRenderers;
            this.guid = String.valueOf(random.nextFloat());
            this.resourcesDirectory = resourcesDirectory;
            this.sessionRouteRegistry = new ConcurrentHashMap<>();
            this.initialsRegistry = new ConcurrentHashMap<>();
        }

        @Override
        public void run() {
            ExecutorService executors = Executors.newFixedThreadPool(numberOfExecutors);
            executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));
        }
    }

    public static class NetworkRequestExecutor implements Runnable {

        String IGNORE_CHROME = "/favicon.ico";
        String BREAK = "\r\n";
        String SPACE = " ";
        String DOUBLEBREAK = "\r\n\r\n";

        Integer REQUEST_METHOD = 0;
        Integer REQUEST_PATH = 1;
        Integer REQUEST_VERSION = 2;

        String RENDERER;

        static String HTTPREQUEST = "http-request";
        static String HTTPRESPONSE = "http-response";
        static String CACHE = "cache";

        String resourcesDirectory;
        Socket socketClient;
        ExecutorService executors;
        ServerSocket serverSocket;
        ConcurrentMap<String, String> sessionRouteRegistry;
        ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry;
        RedirectRegistry redirectRegistry;
        List<Class<?>> viewRenderers;
        ConcurrentMap<String, byte[]> viewBytesMap;
        ConcurrentMap<String, RedirectInfo> initialsRegistry;

        public NetworkRequestExecutor(String RENDERER, String resourcesDirectory, ConcurrentMap<String, byte[]> viewBytesMap, ExecutorService executors, ServerSocket serverSocket, RedirectRegistry redirectRegistry, ConcurrentMap<String, String> sessionRouteRegistry, ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry, List<Class<?>> viewRenderers, ConcurrentMap<String, RedirectInfo> initialsRegistry){
            this.RENDERER = RENDERER;
            this.resourcesDirectory = resourcesDirectory;
            this.viewBytesMap = viewBytesMap;
            this.executors = executors;
            this.serverSocket = serverSocket;
            this.redirectRegistry = redirectRegistry;
            this.sessionRouteRegistry = sessionRouteRegistry;
            this.routeDirectorRegistry = routeDirectorRegistry;
            this.viewRenderers = viewRenderers;
            this.initialsRegistry = initialsRegistry;
        }

        @Override
        public void run() {
            try {

                ServerResources serverResources = new ServerResources();

                socketClient = serverSocket.accept();
                Thread.sleep(19);
                InputStream requestInputStream = socketClient.getInputStream();

                OutputStream clientOutput = socketClient.getOutputStream();

                if(requestInputStream.available() == 0) {
                    requestInputStream.close();
                    clientOutput.flush();
                    clientOutput.close();
                    executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));
                    return;
                }

                ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
                ByteBuffer byteBuffer = ByteBuffer.allocate(1024);
                int bytesRead;
                while((bytesRead = requestInputStream.read(byteBuffer.array())) != -1){
                    byteArrayOutputStream.write(byteBuffer.array(), 0, bytesRead);
                    if(requestInputStream.available() == 0)break;
                }

                String completeRequestContent = byteArrayOutputStream.toString();
                String[] requestBlocks = completeRequestContent.split(DOUBLEBREAK, 2);

                String headerComponent = requestBlocks[0];
                String[] methodPathComponentsLookup = headerComponent.split(BREAK);
                String methodPathComponent = methodPathComponentsLookup[0];

                String[] methodPathVersionComponents = methodPathComponent.split("\\s");

                String requestVerb = methodPathVersionComponents[REQUEST_METHOD];
                String requestPath = methodPathVersionComponents[REQUEST_PATH];
                String requestVersion = methodPathVersionComponents[REQUEST_VERSION];

                NetworkRequest networkRequest = new NetworkRequest(requestVerb, requestPath, serverResources);

                Integer attributesIdx = requestPath.indexOf("?");
                if(attributesIdx != -1) {
                    String attributesElement = requestPath.substring(attributesIdx + 1);
                    requestPath = requestPath.substring(0, attributesIdx);
                    networkRequest.setValues(attributesElement);
                    networkRequest.setUriPath(requestPath);
                }

                if(networkRequest.getUriPath().equals(IGNORE_CHROME)){
                    requestInputStream.close();
                    clientOutput.flush();
                    clientOutput.close();
                    executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));
                    return;
                }

                NetworkResponse networkResponse = new NetworkResponse();
                networkResponse.setResponseStream(clientOutput);

                String[] headerComponents = headerComponent.split(BREAK);
                for(String headerLine : headerComponents){
                    String[] headerLineComponents = headerLine.split(":");
                    if(headerLineComponents.length == 2) {
                        String fieldKey = headerLineComponents[0].trim();
                        String content = headerLineComponents[1].trim();
                        networkRequest.getHeaders().put(fieldKey.toLowerCase(), content);
                    }
                }

                ComponentCompiler requestComponentCompiler = new ComponentCompiler(byteArrayOutputStream.toByteArray(), networkRequest);
                requestComponentCompiler.ingestRequest();

                Long time = serverResources.getTime(0);

                String sessionGuid = serverResources.getCookie(networkRequest.getHeaders());
                if(sessionGuid == null) sessionGuid = serverResources.getGuid(24);

                String routeDirectorGuid = sessionRouteRegistry.get(sessionGuid);
                RouteNegotiator routeNegotiator = null;
                if(routeDirectorGuid != null) {
                    routeNegotiator = routeDirectorRegistry.get(routeDirectorGuid);
                }
                if(routeNegotiator == null){
                    routeNegotiator = getRouteDirector(routeDirectorRegistry);
                }

                NetworkSession activeNetworkSession = routeNegotiator.getRouteAttributes().getSessions().get(sessionGuid);
                if(activeNetworkSession == null) activeNetworkSession = new NetworkSession(time, sessionGuid);
                networkRequest.setSession(activeNetworkSession);

                PageCache pageCache = new PageCache();
                routeDirectorGuid = routeNegotiator.getGuid();
                if(redirectRegistry.getRegistry().containsKey(routeDirectorGuid) &&
                        redirectRegistry.getRegistry().get(routeDirectorGuid).containsKey(HTTPREQUEST)) {
                    NetworkRequest storedNetworkRequest = (NetworkRequest) redirectRegistry.getRegistry().get(routeDirectorGuid).get(HTTPREQUEST);
                    networkResponse = (NetworkResponse) redirectRegistry.getRegistry().get(routeDirectorGuid).get(HTTPRESPONSE);
                    pageCache = (PageCache) redirectRegistry.getRegistry().get(routeDirectorGuid).get(CACHE);
                    activeNetworkSession = storedNetworkRequest.getSession(true);
                    networkRequest.setSession(activeNetworkSession);
                    networkRequest.setVerb("get");
                }


                setSessionAttributesCache(pageCache, activeNetworkSession);

                RouteAttributes routeAttributes = routeNegotiator.getRouteAttributes();
                networkRequest.setRouteAttributes(routeAttributes);
                SecurityManager securityManager = routeAttributes.getSecurityManager();
                RouteResponse routeResponse = routeNegotiator.negotiate(RENDERER, resourcesDirectory, pageCache, networkRequest, networkResponse, securityManager, viewRenderers, viewBytesMap, initialsRegistry);

                if(routeResponse.getCompleteRequest() != null &&
                        routeResponse.getCompleteRequest()){
                    initialsRegistry.remove(requestPath);
                    requestInputStream.close();
                    clientOutput.flush();
                    clientOutput.close();
                    executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));
                    return;
                }

                sessionGuid = networkRequest.getSession(true).getGuid();
                if(!routeNegotiator.getRouteAttributes().getSessions().containsKey(sessionGuid)){
                    routeNegotiator.getRouteAttributes().getSessions().put(sessionGuid, activeNetworkSession);
                }else{
                    routeNegotiator.getRouteAttributes().getSessions().replace(sessionGuid, activeNetworkSession);
                }

                sessionRouteRegistry.put(sessionGuid, routeNegotiator.getGuid());
                routeDirectorRegistry.replace(routeNegotiator.getGuid(), routeNegotiator);

                StringBuilder sessionValues = new StringBuilder();
                sessionValues.append(serverResources.getSessionId()).append("=").append(networkRequest.getSession(true).getGuid() + "; path=/;");
                for(SecurityAttribute securityAttribute : networkResponse.getSecurityAttributes()){
                    sessionValues.append(securityAttribute.getName()).append("=").append(securityAttribute.getValue());
                }

                String redirectLocation = networkResponse.getRedirectLocation();
                if(redirectLocation == null || redirectLocation.equals("")){
                    redirectRegistry.getRegistry().remove(routeDirectorGuid);

                    try {

                        clientOutput.write("HTTP/1.1 ".getBytes());
                        clientOutput.write(routeResponse.getResponseCode().getBytes());
                        clientOutput.write(BREAK.getBytes());

                        Integer bytesLength = routeResponse.getResponseBytes().length;
                        byte[] contentLengthBytes = ("Content-Length:" + bytesLength + BREAK).getBytes();
                        clientOutput.write(contentLengthBytes);

                        clientOutput.write("Server: PLSA.R  [ www.plsar.net ]".getBytes());
                        clientOutput.write(BREAK.getBytes());

                        clientOutput.write("Content-Type:".getBytes());
                        clientOutput.write(routeResponse.getContentType().getBytes());
                        clientOutput.write(BREAK.getBytes());

                        clientOutput.write("Set-Cookie:".getBytes());
                        clientOutput.write(sessionValues.toString().getBytes());

                        clientOutput.write(DOUBLEBREAK.getBytes());
                        clientOutput.write(routeResponse.getResponseBytes());
                    }catch(SocketException sxe){
                        sxe.printStackTrace();
                        try{
                            clientOutput.write("HTTP/1.1 ".getBytes());
                            clientOutput.write(routeResponse.getResponseCode().getBytes());
                            clientOutput.write(BREAK.getBytes());

                            Integer bytesLength = routeResponse.getResponseBytes().length;
                            byte[] contentLengthBytes = ("Content-Length:" + bytesLength + BREAK).getBytes();
                            clientOutput.write(contentLengthBytes);

                            clientOutput.write("Content-Type:".getBytes());
                            clientOutput.write(routeResponse.getContentType().getBytes());
                            clientOutput.write(BREAK.getBytes());

                            clientOutput.write("Set-Cookie:".getBytes());
                            clientOutput.write(sessionValues.toString().getBytes());

                            clientOutput.write(DOUBLEBREAK.getBytes());
                            clientOutput.write(routeResponse.getResponseBytes());
                        }catch(SocketException sxedeux){
                            sxedeux.printStackTrace();
                            requestInputStream.close();
                            clientOutput.flush();
                            clientOutput.close();
                            executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));
                            return;
                        }
                    }
                }else{
                    Map<String, Object> redirectAttributes = new HashMap<>();
                    redirectAttributes.put(HTTPREQUEST, networkRequest);
                    redirectAttributes.put(HTTPRESPONSE, networkResponse);
                    redirectAttributes.put(CACHE, pageCache);
                    redirectRegistry.getRegistry().put(routeDirectorGuid, redirectAttributes);
                    StringBuilder response = new StringBuilder();
                    response.append("HTTP/1.1 307\r\n");
                    response.append("Server: PLSA.R [ www.plsar.net ]" + BREAK);
                    response.append("Content-Type:text/html" + BREAK);
                    response.append("Location:" + redirectLocation + SPACE + BREAK);
                    response.append("Content-Length: -1" + BREAK);
                    networkResponse.removeRedirectLocation();
                    clientOutput.write(response.toString().getBytes());
                }


                clientOutput.close();
                socketClient.close();

                executors.execute(new NetworkRequestExecutor(RENDERER, resourcesDirectory, viewBytesMap, executors, serverSocket, redirectRegistry, sessionRouteRegistry, routeDirectorRegistry, viewRenderers, initialsRegistry));

            }catch(IOException ex){
                ex.printStackTrace();
            }catch(InterruptedException ioException){
                ioException.printStackTrace();
            }
        }

        void setSessionAttributesCache(PageCache pageCache, NetworkSession networkSession) {
            for(Map.Entry<String, Object> sessionEntry : networkSession.getAttributes().entrySet()){
                pageCache.set(sessionEntry.getKey(), sessionEntry.getValue());
            }
        }

        void clearRedirectRegistry(String routeDirectorGuid) {
            List<String> redirectKeys = new ArrayList<>();
            if(redirectRegistry.getRegistry().containsKey(routeDirectorGuid)) {
                for (Map.Entry<String, Object> redirectEntry : redirectRegistry.getRegistry().get(routeDirectorGuid).entrySet()) {
                    redirectKeys.add(redirectEntry.getKey());
                }
                for (String redirectKey : redirectKeys) {
                    redirectRegistry.getRegistry().get(routeDirectorGuid).remove(redirectKey);
                }//start doing the right thing and life will be easier.
            }
        }

        RouteNegotiator getRouteDirector(ConcurrentMap<String, RouteNegotiator> routeDirectorRegistry){
            for(Map.Entry<String, RouteNegotiator> routeDirectorEntry : routeDirectorRegistry.entrySet()){
                return routeDirectorEntry.getValue();
            }
            return null;
        }

    }

    public void setPropertiesConfig(PropertiesConfig propertiesConfig){
        this.propertiesConfig = propertiesConfig;
    }

    public void setPageRenderingScheme(String RENDERING_SCHEME) {
        this.RENDERING_SCHEME = RENDERING_SCHEME;
    }

    public void setViewConfig(ViewConfig viewConfig) {
        this.viewConfig = viewConfig;
    }

    public void setSecurityAccess(Class<?> securityAccessKlass) {
        this.securityAccessKlass = securityAccessKlass;
    }

    public void setSchemaConfig(SchemaConfig schemaConfig) {
        this.schemaConfig = schemaConfig;
    }

    public void setNumberOfPartitions(int numberOfPartitions){
        this.numberOfPartitions = numberOfPartitions;
    }

    public void setNumberOfRequestExecutors(int numberOfRequestExecutors){
        this.numberOfRequestExecutors = numberOfRequestExecutors;
    }

    public PLSAR setStartupExecutors(Integer STARTUP_EXECUTORS){
        this.STARTUP_EXECUTORS = STARTUP_EXECUTORS;
        return this;
    }

    public PLSAR addViewRenderer(Class<?> viewRenderer){
        this.viewRenderers.add(viewRenderer);
        return this;
    }

    public PLSAR setPersistenceConfig(PersistenceConfig persistenceConfig) {
        this.persistenceConfig = persistenceConfig;
        return this;
    }

}