/*
 * Decompiled with CFR 0.152.
 */
package org.knowm.xchange.simulated;

import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.knowm.xchange.currency.CurrencyPair;
import org.knowm.xchange.dto.Order;
import org.knowm.xchange.dto.marketdata.OrderBook;
import org.knowm.xchange.dto.marketdata.Ticker;
import org.knowm.xchange.dto.marketdata.Trade;
import org.knowm.xchange.dto.trade.LimitOrder;
import org.knowm.xchange.dto.trade.MarketOrder;
import org.knowm.xchange.dto.trade.UserTrade;
import org.knowm.xchange.exceptions.ExchangeException;
import org.knowm.xchange.instrument.Instrument;
import org.knowm.xchange.simulated.Account;
import org.knowm.xchange.simulated.AccountFactory;
import org.knowm.xchange.simulated.BookLevel;
import org.knowm.xchange.simulated.BookOrder;
import org.knowm.xchange.simulated.Fill;
import org.knowm.xchange.simulated.Level3OrderBook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class MatchingEngine {
    private static final Logger LOGGER = LoggerFactory.getLogger(MatchingEngine.class);
    private static final BigDecimal FEE_RATE = new BigDecimal("0.001");
    private static final int TRADE_HISTORY_SIZE = 50;
    private final AccountFactory accountFactory;
    private final CurrencyPair currencyPair;
    private final int priceScale;
    private final BigDecimal minimumAmount;
    private final Consumer<Fill> onFill;
    private final List<BookLevel> asks = new LinkedList<BookLevel>();
    private final List<BookLevel> bids = new LinkedList<BookLevel>();
    private final Deque<Trade> publicTrades = new ConcurrentLinkedDeque<Trade>();
    private final Multimap<String, UserTrade> userTrades = LinkedListMultimap.create();
    private volatile Ticker ticker = new Ticker.Builder().build();

    MatchingEngine(AccountFactory accountFactory, CurrencyPair currencyPair, int priceScale, BigDecimal minimumAmount) {
        this(accountFactory, currencyPair, priceScale, minimumAmount, f -> {});
    }

    MatchingEngine(AccountFactory accountFactory, CurrencyPair currencyPair, int priceScale, BigDecimal minimumAmount, Consumer<Fill> onFill) {
        this.accountFactory = accountFactory;
        this.currencyPair = currencyPair;
        this.priceScale = priceScale;
        this.minimumAmount = minimumAmount;
        this.onFill = onFill;
    }

    public synchronized LimitOrder postOrder(String apiKey, Order original) {
        LOGGER.debug("User {} posting order: {}", (Object)apiKey, (Object)original);
        this.validate(original);
        Account account = this.accountFactory.get(apiKey);
        this.checkBalance(original, account);
        BookOrder takerOrder = BookOrder.fromOrder(original, apiKey);
        switch (takerOrder.getType()) {
            case ASK: {
                LOGGER.debug("Matching against bids");
                this.chewBook(this.bids, takerOrder);
                if (takerOrder.isDone()) break;
                if (original instanceof MarketOrder) {
                    throw new ExchangeException("Cannot fulfil order. No buyers.");
                }
                this.insertIntoBook(this.asks, takerOrder, Order.OrderType.ASK, account);
                break;
            }
            case BID: {
                LOGGER.debug("Matching against asks");
                this.chewBook(this.asks, takerOrder);
                if (takerOrder.isDone()) break;
                if (original instanceof MarketOrder) {
                    throw new ExchangeException("Cannot fulfil order. No sellers.");
                }
                this.insertIntoBook(this.bids, takerOrder, Order.OrderType.BID, account);
                break;
            }
            default: {
                throw new ExchangeException("Unsupported order type: " + takerOrder.getType());
            }
        }
        return takerOrder.toOrder(this.currencyPair);
    }

    private void validate(Order order) {
        if (order.getOriginalAmount().compareTo(this.minimumAmount) < 0) {
            throw new ExchangeException("Trade amount is " + order.getOriginalAmount() + ", minimum is " + this.minimumAmount);
        }
        if (order instanceof LimitOrder) {
            LimitOrder limitOrder = (LimitOrder)order;
            if (limitOrder.getLimitPrice() == null) {
                throw new ExchangeException("No price");
            }
            if (limitOrder.getLimitPrice().compareTo(BigDecimal.ZERO) <= 0) {
                throw new ExchangeException("Limit price is " + limitOrder.getLimitPrice() + ", must be positive");
            }
            int scale = limitOrder.getLimitPrice().stripTrailingZeros().scale();
            if (scale > this.priceScale) {
                throw new ExchangeException("Price scale is " + scale + ", maximum is " + this.priceScale);
            }
        }
    }

    private void checkBalance(Order order, Account account) {
        if (order instanceof LimitOrder) {
            account.checkBalance((LimitOrder)order);
        } else {
            BigDecimal marketCostOrProceeds = this.marketCostOrProceeds(order.getType(), order.getOriginalAmount());
            BigDecimal marketAmount = order.getType().equals((Object)Order.OrderType.BID) ? marketCostOrProceeds : order.getOriginalAmount();
            account.checkBalance(order, marketAmount);
        }
    }

    private void insertIntoBook(List<BookLevel> book, BookOrder order, Order.OrderType type, Account account) {
        int i = 0;
        boolean insert = false;
        for (BookLevel level : book) {
            int signum = level.getPrice().compareTo(order.getLimitPrice());
            if (signum == 0) {
                level.getOrders().add(order);
                return;
            }
            if (signum < 0 && type == Order.OrderType.BID || signum > 0 && type == Order.OrderType.ASK) {
                insert = true;
                break;
            }
            ++i;
        }
        account.reserve(order.toOrder(this.currencyPair));
        BookLevel newLevel = new BookLevel(order.getLimitPrice());
        newLevel.getOrders().add(order);
        if (insert) {
            book.add(i, newLevel);
        } else {
            book.add(newLevel);
        }
        this.ticker = this.newTickerFromBook().last(this.ticker.getLast()).build();
    }

    private Ticker.Builder newTickerFromBook() {
        return new Ticker.Builder().ask(this.asks.isEmpty() ? null : this.asks.get(0).getPrice()).bid(this.bids.isEmpty() ? null : this.bids.get(0).getPrice());
    }

    public BigDecimal marketCostOrProceeds(Order.OrderType orderType, BigDecimal amount) {
        BigDecimal remaining = amount;
        BigDecimal cost = BigDecimal.ZERO;
        List<BookLevel> orderbookSide = orderType.equals((Object)Order.OrderType.BID) ? this.asks : this.bids;
        for (BookOrder order : FluentIterable.from(orderbookSide).transformAndConcat(BookLevel::getOrders)) {
            BigDecimal available = order.getRemainingAmount();
            BigDecimal tradeAmount = remaining.compareTo(available) >= 0 ? available : remaining;
            BigDecimal tradeCost = tradeAmount.multiply(order.getLimitPrice());
            cost = cost.add(tradeCost);
            if ((remaining = remaining.subtract(tradeAmount)).compareTo(BigDecimal.ZERO) != 0) continue;
            return cost;
        }
        throw new ExchangeException("Insufficient liquidity in book");
    }

    public synchronized Level3OrderBook book() {
        return new Level3OrderBook((List<LimitOrder>)FluentIterable.from(this.asks).transformAndConcat(BookLevel::getOrders).transform(o -> o.toOrder(this.currencyPair)).toList(), (List<LimitOrder>)FluentIterable.from(this.bids).transformAndConcat(BookLevel::getOrders).transform(o -> o.toOrder(this.currencyPair)).toList());
    }

    public Ticker ticker() {
        return this.ticker;
    }

    public List<Trade> publicTrades() {
        return FluentIterable.from(this.publicTrades).transform(t -> Trade.Builder.from((Trade)t).build()).toList();
    }

    public synchronized List<UserTrade> tradeHistory(String apiKey) {
        return ImmutableList.copyOf((Collection)this.userTrades.get((Object)apiKey));
    }

    private void chewBook(Iterable<BookLevel> makerOrders, BookOrder takerOrder) {
        Iterator<BookLevel> levelIter = makerOrders.iterator();
        while (levelIter.hasNext()) {
            BookLevel level = levelIter.next();
            Iterator<BookOrder> orderIter = level.getOrders().iterator();
            while (orderIter.hasNext() && !takerOrder.isDone()) {
                BookOrder makerOrder = orderIter.next();
                LOGGER.debug("Matching against maker order {}", (Object)makerOrder);
                if (!makerOrder.matches(takerOrder)) {
                    LOGGER.debug("Ran out of maker orders at this price");
                    return;
                }
                BigDecimal tradeAmount = takerOrder.getRemainingAmount().compareTo(makerOrder.getRemainingAmount()) > 0 ? makerOrder.getRemainingAmount() : takerOrder.getRemainingAmount();
                LOGGER.debug("Matches for {}", (Object)tradeAmount);
                this.matchOff(takerOrder, makerOrder, tradeAmount);
                if (!makerOrder.isDone()) continue;
                LOGGER.debug("Maker order removed from book");
                orderIter.remove();
                if (!level.getOrders().isEmpty()) continue;
                levelIter.remove();
            }
        }
    }

    private void matchOff(BookOrder takerOrder, BookOrder makerOrder, BigDecimal tradeAmount) {
        Date timestamp = new Date();
        UserTrade takerTrade = UserTrade.builder().currencyPair(this.currencyPair).id(UUID.randomUUID().toString()).originalAmount(tradeAmount).price(makerOrder.getLimitPrice()).timestamp(timestamp).type(takerOrder.getType()).orderId(takerOrder.getId()).feeAmount(takerOrder.getType() == Order.OrderType.ASK ? tradeAmount.multiply(makerOrder.getLimitPrice()).multiply(FEE_RATE) : tradeAmount.multiply(FEE_RATE)).feeCurrency(takerOrder.getType() == Order.OrderType.ASK ? this.currencyPair.counter : this.currencyPair.base).build();
        LOGGER.debug("Created taker trade: {}", (Object)takerTrade);
        this.accumulate(takerOrder, takerTrade);
        Order.OrderType makerType = takerOrder.getType() == Order.OrderType.ASK ? Order.OrderType.BID : Order.OrderType.ASK;
        UserTrade makerTrade = UserTrade.builder().currencyPair(this.currencyPair).id(UUID.randomUUID().toString()).originalAmount(tradeAmount).price(makerOrder.getLimitPrice()).timestamp(timestamp).type(makerType).orderId(makerOrder.getId()).feeAmount(makerType == Order.OrderType.ASK ? tradeAmount.multiply(makerOrder.getLimitPrice()).multiply(FEE_RATE) : tradeAmount.multiply(FEE_RATE)).feeCurrency(makerType == Order.OrderType.ASK ? this.currencyPair.counter : this.currencyPair.base).build();
        LOGGER.debug("Created maker trade: {}", (Object)makerOrder);
        this.accumulate(makerOrder, makerTrade);
        this.recordFill(new Fill(takerOrder.getApiKey(), takerTrade, true));
        this.recordFill(new Fill(makerOrder.getApiKey(), makerTrade, false));
        this.ticker = this.newTickerFromBook().last(makerOrder.getLimitPrice()).build();
    }

    private void accumulate(BookOrder bookOrder, UserTrade trade) {
        BigDecimal amount = trade.getOriginalAmount();
        BigDecimal price = trade.getPrice();
        BigDecimal newTotal = bookOrder.getCumulativeAmount().add(amount);
        if (bookOrder.getCumulativeAmount().compareTo(BigDecimal.ZERO) == 0) {
            bookOrder.setAveragePrice(price);
        } else {
            bookOrder.setAveragePrice(bookOrder.getAveragePrice().multiply(bookOrder.getCumulativeAmount()).add(price.multiply(amount)).divide(newTotal, this.priceScale, RoundingMode.HALF_UP));
        }
        bookOrder.setCumulativeAmount(newTotal);
        bookOrder.setFee(bookOrder.getFee().add(trade.getFeeAmount()));
    }

    public synchronized List<LimitOrder> openOrders(String apiKey) {
        return Stream.concat(this.asks.stream(), this.bids.stream()).flatMap(v -> v.getOrders().stream()).filter(o -> o.getApiKey().equals(apiKey)).sorted(Ordering.natural().onResultOf(BookOrder::getTimestamp).reversed()).map(o -> o.toOrder(this.currencyPair)).collect(Collectors.toList());
    }

    public synchronized OrderBook getLevel2OrderBook() {
        return new OrderBook(new Date(), this.accumulateBookSide(Order.OrderType.ASK, this.asks), this.accumulateBookSide(Order.OrderType.BID, this.bids));
    }

    private List<LimitOrder> accumulateBookSide(Order.OrderType orderType, List<BookLevel> book) {
        BigDecimal price = null;
        BigDecimal amount = BigDecimal.ZERO;
        ArrayList<LimitOrder> result = new ArrayList<LimitOrder>();
        Iterator iter = book.stream().flatMap(v -> v.getOrders().stream()).iterator();
        while (iter.hasNext()) {
            BookOrder bookOrder = (BookOrder)iter.next();
            if (price != null && bookOrder.getLimitPrice().compareTo(price) != 0) {
                result.add(new LimitOrder.Builder(orderType, (Instrument)this.currencyPair).originalAmount(amount).limitPrice(price).build());
                amount = BigDecimal.ZERO;
            }
            amount = amount.add(bookOrder.getRemainingAmount());
            price = bookOrder.getLimitPrice();
        }
        if (price != null) {
            result.add(new LimitOrder.Builder(orderType, (Instrument)this.currencyPair).originalAmount(amount).limitPrice(price).build());
        }
        return result;
    }

    private void recordFill(Fill fill) {
        if (!fill.isTaker()) {
            this.publicTrades.push((Trade)fill.getTrade());
            if (this.publicTrades.size() > 50) {
                this.publicTrades.removeLast();
            }
        }
        this.userTrades.put((Object)fill.getApiKey(), (Object)fill.getTrade());
        this.accountFactory.get(fill.getApiKey()).fill(fill.getTrade(), !fill.isTaker());
        this.onFill.accept(fill);
    }

    public void cancelOrder(String apiKey, String orderId) {
        this.cancelOrder(apiKey, orderId, Order.OrderType.BID);
        this.cancelOrder(apiKey, orderId, Order.OrderType.ASK);
    }

    public void cancelOrder(String apiKey, String orderId, Order.OrderType type) {
        Stream bookLevelStream;
        Account account = this.accountFactory.get(apiKey);
        switch (type) {
            case ASK: {
                bookLevelStream = this.asks.stream();
                break;
            }
            case BID: {
                bookLevelStream = this.bids.stream();
                break;
            }
            default: {
                throw new ExchangeException("Unsupported order type: " + type);
            }
        }
        bookLevelStream.forEach(bookLevel -> bookLevel.getOrders().removeIf(bookOrder -> {
            boolean remove = bookOrder.getId().equals(orderId);
            if (remove) {
                account.release(bookOrder.toOrder(this.currencyPair));
            }
            return remove;
        }));
    }
}

