/**
 *	Copyright (c) 2015 Vr Security Inc.
 *	All rights reserved.
 *	
 *	Redistribution and use in source and binary forms, with or without
 *	modification, are permitted provided that the following conditions are met:
 *	    * Redistributions of source code must retain the above copyright
 *	      notice, this list of conditions and the following disclaimer.
 *	    * Redistributions in binary form must reproduce the above copyright
 *	      notice, this list of conditions and the following disclaimer in the
 *	      documentation and/or other materials provided with the distribution.
 *	    * Neither the name of the <organization> nor the
 *	      names of its contributors may be used to endorse or promote products
 *	      derived from this software without specific prior written permission.
 *	
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *	DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 *	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package net.ossindex.common.resource;

import java.net.URI;
import java.net.URISyntaxException;

import net.ossindex.version.IVersion;
import net.ossindex.version.VersionFactory;
import net.ossindex.version.VersionRange;


/** Representation of a Vulnerability, backed by the OSS Index REST API.
 * 
 * Vulnerabilities come in 3 flavours
 * 
 *   1. As a search result -- external link (/v1.0/scm/299024085/vulnerabilities).
 *      An 'external link' vulnerability is a vulnerability that has been added,
 *      likely manually, which is comprised simply of a link to an external site
 *      describing a vulnerability, and the versions affected. The versions are
 *      related to the unique SCM/artifact that references this vulnerability
 *      instance. Note that a vulnerability *may* affect multiple products, but
 *      the version field is related to the SCM/Aftifact being queries for.
 *       [
 *         {
 *           "uri": "https://nodesecurity.io/advisories/uglifyjs_incorrectly_handles_non-boolean_comparisons",
 *           "versions": [
 *             "&lt;= 2.4.23"
 *           ]
 *          }
 *       ]
 *       
 *   2. As a search result -- CVE (/v1.0/scm/296375846/vulnerabilities).
 *      These are vulnerabilities found in the National Vulnerability Database(NVD).
 *      The versions are
 *      related to the unique SCM/artifact that references this vulnerability
 *      instance. Note that a vulnerability *may* affect multiple products, but
 *      the version field is related to the SCM/Aftifact being queries for.
 *      
 *       [
 *         {
 *          "uri": "cve:/CVE-2011-4969",
 *          "id": "348528",
 *          "cve-id": "CVE-2011-4969",
 *          "summary": "Cross-site scripting (XSS) vulnerability in jQuery before 1.6.3, when using location.hash to select elements, allows remote attackers to inject arbitrary web script or HTML via a crafted tag.",
 *          "details": "http://localhost:8080/v1.0/cve/348528",
 *          "versions": [
 *            "1.6",
 *            "1.6.1",
 *            "1.6.2"
 *          ]
 *        },
 *        ...
 *       ]
 *
 *   3. Full CVE details (/v1.0/cve/323345)
 *      Note that in this case there is no version field in the same sense as the
 *      ones as a result of search, since when searching there is a single product
 *      for which we are returning versions, whereas when CVE details are retrieved
 *      they may be affecting multiple versions.
 *   
 * @author Ken Duck
 *
 */
public class VulnerabilityResource extends AbstractRemoteResource
{
	//--------------------------------------------------------------------
	// External link vulnerability result
	//--------------------------------------------------------------------
	//   Nothing unique here. Everything is represented in 'common' fields, below.

	//--------------------------------------------------------------------
	// CVE details only. Full CVE details include this, CVE Common, and Common fields
	//--------------------------------------------------------------------
	private String complexity;
	private String vector;
	private String authentication;
	private String availability;
	private String confidentiality;
	private String integrity;
	private String source;
	private String security_protection;
	private Long discovered;
	private Long generated;
	private Long published;
	private Long modified;
	// FIXME: Not handled: CPEs
	// FIXME: Not handled: CWEs

	//--------------------------------------------------------------------
	// CVE Common (Search and Details)
	//--------------------------------------------------------------------
	//  When doing a search that returns CVEs, these and common are returned
	private String cveId;

	//--------------------------------------------------------------------
	// Common to every vulnerability type
	//--------------------------------------------------------------------
	private String uri;
	private String title;
	private String summary;
	private String details;
	// Note that in this case there is no version field in the same sense as the
	// ones as a result of search, since when searching there is a single product
	// for which we are returning versions, whereas when CVE details are retrieved
	// they may be affecting multiple versions.
	private String[] versions;



	/*
	 * (non-Javadoc)
	 * @see net.ossindex.common.resource.AbstractRemoteResource#getResourceType()
	 */
	@Override
	protected String getResourceType()
	{
		return "vulnerability";
	}

	/** Return true if this vulnerability represents a known CVE. Otherwise this
	 * is likely an 'external link'.
	 * 
	 * @return true if this is a CVE
	 */
	public boolean isCve()
	{
		return getId() > 0;
	}

	/** Get the unique URI for a vulnerability
	 * 
	 * @return The vulnerability URI
	 */
	public URI getUri()
	{
		if(uri != null)
		{
			try
			{
				return new URI(uri);
			}
			catch (URISyntaxException e)
			{
				e.printStackTrace();
			}
		}
		return null;
	}

	/** Get the product version ranges affected by this vulnerability. This will only
	 * be filled in if this vulnerability is the result of a search which allows us
	 * to identify the explicit project we are interested in.
	 * 
	 * @return The list of version ranges that the vulnerability affects
	 */
	public String[] getVersions()
	{
		return versions;
	}

	/** Get a reasonable description of the vulnerability
	 * 
	 * @return A description
	 */
	public String getDescription()
	{
		if(summary != null) return summary;
		return uri.toString();
	}

	/** Get a reasonable title of the vulnerability
	 * 
	 * @return A title
	 */
	public String getTitle()
	{
		if(title != null) return title;
		return uri.toString();
	}

	/** Return true if this vulnerability affects anything within the provided range
	 * 
	 * @param name product/package name
	 * @param range range of versions that the passed in product/package represents
	 * @return True if the vulnerability affects the specified product/package
	 */
	public boolean affects(String name, String range)
	{
		if(versions != null)
		{
			VersionRange myRange = new VersionRange(versions);
			VersionRange yourRange = new VersionRange(range);
			return myRange.intersects(yourRange);
		}
		return false;
	}

	/** Return true if this vulnerability applies to the specified version.
	 * 
	 * @param version Version we are checking
	 * @return true if the vulnerability affects the resource
	 */
	public boolean appliesTo(String version)
	{
		for(String v: versions)
		{
			IVersion hv = VersionFactory.getVersionFactory().getVersion(v);
			if(hv.satisfies(version)) return true;
		}
		return false;
	}

}
