/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set.hash;

import net.openhft.koloboke.collect.set.ObjSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.*;


/**
 * This class consists only of static factory methods to construct {@code HashObjSet}s, and
 * the default {@link HashObjSetFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashObjSet
 */
public final class HashObjSets {
    private static final ServiceLoader<HashObjSetFactory> LOADER =
            ServiceLoader.load(HashObjSetFactory.class);
    private static HashObjSetFactory defaultFactory = null;

    /**
     * Returns the default implementation of {@link HashObjSetFactory}, to which all static methods
     * in this class delegate.
     *
     * @return the default implementation of {@link HashObjSetFactory}
     * @param <E> the most general element type of the sets that could
                               constructed by the returned factory 
     * @throws RuntimeException if no implementations of {@link HashObjSetFactory} are provided
     */
    @Nonnull
    public static <E> HashObjSetFactory<E> getDefaultFactory() {
        if (defaultFactory != null) {
            return (HashObjSetFactory<E>) defaultFactory;
        } else {
            // synchronization?
            return (HashObjSetFactory<E>)
                    (defaultFactory = LOADER.iterator().next());
        }
    }
    
    

    


    /**
     * Constructs a new empty mutable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet() newMutableSet()}.
     *
      * @param <E> the element type of the returned set 
     * @return a new empty mutable set
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet() {
        return getDefaultFactory().newMutableSet();
    }

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(int) newMutableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(int expectedSize) {
        return getDefaultFactory().newMutableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterator, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterator<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Consumer, int) newMutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Object[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(@Nonnull E[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }
    

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable) newMutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable, Iterable) newMutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2) {
        return getDefaultFactory().newMutableSet(elems1, elems2);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Iterator) newMutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(
            @Nonnull Iterator<? extends E> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Consumer) newMutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            ) {
        return getDefaultFactory().newMutableSet(elementsSupplier);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSet(
     * Object[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSet(@Nonnull E[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }
    


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSetOf(
     * Object) newMutableSetOf(e1)}.
     *
     * @param e1 the sole element
     * @param <E> the element type of the returned set 
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSetOf(E e1) {
        return getDefaultFactory().newMutableSetOf(e1);
    }

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSetOf(
     * Object, Object) newMutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSetOf(E e1, E e2) {
        return getDefaultFactory().newMutableSetOf(e1, e2);
    }

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSetOf(
     * Object, Object, Object) newMutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSetOf(E e1, E e2, E e3) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSetOf(
     * Object, Object, Object, Object) newMutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSetOf(E e1, E e2, E e3, E e4) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newMutableSetOf(Object, Object, Object,
     * Object, Object, Object...) newMutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E> the element type of the returned set 
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newMutableSetOf(E e1, E e2, E e3, E e4,
            E e5, E... restElements) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    /**
     * Constructs a new empty updatable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet() newUpdatableSet()}.
     *
      * @param <E> the element type of the returned set 
     * @return a new empty updatable set
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet() {
        return getDefaultFactory().newUpdatableSet();
    }

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(int) newUpdatableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(int expectedSize) {
        return getDefaultFactory().newUpdatableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterator, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterator<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Consumer, int) newUpdatableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Object[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(@Nonnull E[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }
    

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable) newUpdatableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable, Iterable) newUpdatableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Iterator) newUpdatableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(
            @Nonnull Iterator<? extends E> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Consumer) newUpdatableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            ) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSet(
     * Object[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSet(@Nonnull E[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }
    


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSetOf(
     * Object) newUpdatableSetOf(e1)}.
     *
     * @param e1 the sole element
     * @param <E> the element type of the returned set 
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSetOf(E e1) {
        return getDefaultFactory().newUpdatableSetOf(e1);
    }

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSetOf(
     * Object, Object) newUpdatableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSetOf(E e1, E e2) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2);
    }

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSetOf(
     * Object, Object, Object) newUpdatableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSetOf(E e1, E e2, E e3) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSetOf(
     * Object, Object, Object, Object) newUpdatableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSetOf(E e1, E e2, E e3, E e4) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newUpdatableSetOf(Object, Object, Object,
     * Object, Object, Object...) newUpdatableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E> the element type of the returned set 
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newUpdatableSetOf(E e1, E e2, E e3, E e4,
            E e5, E... restElements) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4, e5, restElements);
    }


    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterator, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterator<? extends E> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Consumer, int) newImmutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Object[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E> the element type of the returned set 
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(@Nonnull E[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }
    

    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable) newImmutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable, Iterable) newImmutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2) {
        return getDefaultFactory().newImmutableSet(elems1, elems2);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterable<? extends E> elems1,
            @Nonnull Iterable<? extends E> elems2,
            @Nonnull Iterable<? extends E> elems3,
            @Nonnull Iterable<? extends E> elems4,
            @Nonnull Iterable<? extends E> elems5) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Iterator) newImmutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(
            @Nonnull Iterator<? extends E> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Consumer) newImmutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E>> elementsSupplier
            ) {
        return getDefaultFactory().newImmutableSet(elementsSupplier);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSet(
     * Object[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E> the element type of the returned set 
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSet(@Nonnull E[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }
    


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSetOf(
     * Object) newImmutableSetOf(e1)}.
     *
     * @param e1 the sole element
     * @param <E> the element type of the returned set 
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSetOf(E e1) {
        return getDefaultFactory().newImmutableSetOf(e1);
    }

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSetOf(
     * Object, Object) newImmutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSetOf(E e1, E e2) {
        return getDefaultFactory().newImmutableSetOf(e1, e2);
    }

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSetOf(
     * Object, Object, Object) newImmutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSetOf(E e1, E e2, E e3) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSetOf(
     * Object, Object, Object, Object) newImmutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSetOf(E e1, E e2, E e3, E e4) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjSetFactory#newImmutableSetOf(Object, Object, Object,
     * Object, Object, Object...) newImmutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E> the element type of the returned set 
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    public static <E> HashObjSet<E> newImmutableSetOf(E e1, E e2, E e3, E e4,
            E e5, E... restElements) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    private HashObjSets() {}
}

