/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set.hash;

import net.openhft.koloboke.collect.set.FloatSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.*;


/**
 * This class consists only of static factory methods to construct {@code HashFloatSet}s, and
 * the default {@link HashFloatSetFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashFloatSet
 */
public final class HashFloatSets {
    private static final ServiceLoader<HashFloatSetFactory> LOADER =
            ServiceLoader.load(HashFloatSetFactory.class);
    private static HashFloatSetFactory defaultFactory = null;

    /**
     * Returns the default implementation of {@link HashFloatSetFactory}, to which all static methods
     * in this class delegate.
     *
     * @return the default implementation of {@link HashFloatSetFactory}
    
     * @throws RuntimeException if no implementations of {@link HashFloatSetFactory} are provided
     */
    @Nonnull
    public static  HashFloatSetFactory getDefaultFactory() {
        if (defaultFactory != null) {
            return (HashFloatSetFactory) defaultFactory;
        } else {
            // synchronization?
            return (HashFloatSetFactory)
                    (defaultFactory = LOADER.iterator().next());
        }
    }
    
    

    


    /**
     * Constructs a new empty mutable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet() newMutableSet()}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
    public static  HashFloatSet newMutableSet() {
        return getDefaultFactory().newMutableSet();
    }

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(int) newMutableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(int expectedSize) {
        return getDefaultFactory().newMutableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterator, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterator<Float> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Consumer, int) newMutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * float[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(@Nonnull float[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Float[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newMutableSet(@Nonnull Float[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable) newMutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable, Iterable) newMutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2) {
        return getDefaultFactory().newMutableSet(elems1, elems2);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Iterator) newMutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(
            @Nonnull Iterator<Float> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Consumer) newMutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newMutableSet(elementsSupplier);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * float[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newMutableSet(@Nonnull float[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSet(
     * Float[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newMutableSet(@Nonnull Float[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSetOf(
     * float) newMutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    public static  HashFloatSet newMutableSetOf(float e1) {
        return getDefaultFactory().newMutableSetOf(e1);
    }

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSetOf(
     * float, float) newMutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    public static  HashFloatSet newMutableSetOf(float e1, float e2) {
        return getDefaultFactory().newMutableSetOf(e1, e2);
    }

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSetOf(
     * float, float, float) newMutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    public static  HashFloatSet newMutableSetOf(float e1, float e2, float e3) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSetOf(
     * float, float, float, float) newMutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    public static  HashFloatSet newMutableSetOf(float e1, float e2, float e3, float e4) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newMutableSetOf(float, float, float,
     * float, float, float...) newMutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    public static  HashFloatSet newMutableSetOf(float e1, float e2, float e3, float e4,
            float e5, float... restElements) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    /**
     * Constructs a new empty updatable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet() newUpdatableSet()}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet() {
        return getDefaultFactory().newUpdatableSet();
    }

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(int) newUpdatableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(int expectedSize) {
        return getDefaultFactory().newUpdatableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterator, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterator<Float> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Consumer, int) newUpdatableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * float[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(@Nonnull float[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Float[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newUpdatableSet(@Nonnull Float[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable) newUpdatableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable, Iterable) newUpdatableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Iterator) newUpdatableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(
            @Nonnull Iterator<Float> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Consumer) newUpdatableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * float[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSet(@Nonnull float[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSet(
     * Float[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newUpdatableSet(@Nonnull Float[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSetOf(
     * float) newUpdatableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSetOf(float e1) {
        return getDefaultFactory().newUpdatableSetOf(e1);
    }

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSetOf(
     * float, float) newUpdatableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSetOf(float e1, float e2) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2);
    }

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSetOf(
     * float, float, float) newUpdatableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSetOf(float e1, float e2, float e3) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSetOf(
     * float, float, float, float) newUpdatableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSetOf(float e1, float e2, float e3, float e4) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newUpdatableSetOf(float, float, float,
     * float, float, float...) newUpdatableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    public static  HashFloatSet newUpdatableSetOf(float e1, float e2, float e3, float e4,
            float e5, float... restElements) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4, e5, restElements);
    }


    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterator, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterator<Float> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Consumer, int) newImmutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * float[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(@Nonnull float[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Float[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newImmutableSet(@Nonnull Float[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable) newImmutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable, Iterable) newImmutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2) {
        return getDefaultFactory().newImmutableSet(elems1, elems2);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Iterator) newImmutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(
            @Nonnull Iterator<Float> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Consumer) newImmutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newImmutableSet(elementsSupplier);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * float[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashFloatSet newImmutableSet(@Nonnull float[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSet(
     * Float[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashFloatSet newImmutableSet(@Nonnull Float[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSetOf(
     * float) newImmutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    public static  HashFloatSet newImmutableSetOf(float e1) {
        return getDefaultFactory().newImmutableSetOf(e1);
    }

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSetOf(
     * float, float) newImmutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    public static  HashFloatSet newImmutableSetOf(float e1, float e2) {
        return getDefaultFactory().newImmutableSetOf(e1, e2);
    }

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSetOf(
     * float, float, float) newImmutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    public static  HashFloatSet newImmutableSetOf(float e1, float e2, float e3) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSetOf(
     * float, float, float, float) newImmutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    public static  HashFloatSet newImmutableSetOf(float e1, float e2, float e3, float e4) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashFloatSetFactory#newImmutableSetOf(float, float, float,
     * float, float, float...) newImmutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    public static  HashFloatSet newImmutableSetOf(float e1, float e2, float e3, float e4,
            float e5, float... restElements) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    private HashFloatSets() {}
}

