/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set;

import net.openhft.koloboke.collect.ContainerFactory;
import net.openhft.koloboke.collect.Equivalence;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Iterator;


/**
 * An immutable factory of {@code LongSet}s.
 *
 * 
 * @param <F> the concrete factory type which extends this interface
 * @see LongSet
 */
public interface LongSetFactory<
        F extends LongSetFactory<F>> extends ContainerFactory<F> {


    

    

    

    


    /**
     * Constructs a new empty mutable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
     LongSet newMutableSet();

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
     LongSet newMutableSet(int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elements, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterator<Long> elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull long[] elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Long[] elements, int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elements);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     LongSet newMutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Iterator<Long> elements);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull long[] elements);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newMutableSet(@Nonnull Long[] elements);


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
     LongSet newMutableSetOf(long e1);

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
     LongSet newMutableSetOf(long e1, long e2);

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
     LongSet newMutableSetOf(long e1, long e2, long e3);

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
     LongSet newMutableSetOf(long e1, long e2, long e3,
            long e4);

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
     LongSet newMutableSetOf(long e1, long e2, long e3,
            long e4, long e5, long... restElements);

    /**
     * Constructs a new empty updatable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
     LongSet newUpdatableSet();

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
     LongSet newUpdatableSet(int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elements, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterator<Long> elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull long[] elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Long[] elements, int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elements);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     LongSet newUpdatableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Iterator<Long> elements);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull long[] elements);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newUpdatableSet(@Nonnull Long[] elements);


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
     LongSet newUpdatableSetOf(long e1);

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
     LongSet newUpdatableSetOf(long e1, long e2);

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
     LongSet newUpdatableSetOf(long e1, long e2, long e3);

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
     LongSet newUpdatableSetOf(long e1, long e2, long e3,
            long e4);

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
     LongSet newUpdatableSetOf(long e1, long e2, long e3,
            long e4, long e5, long... restElements);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elements, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5, int expectedSize);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterator<Long> elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull long[] elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Long[] elements, int expectedSize);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elements);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     LongSet newImmutableSet(@Nonnull Iterable<Long> elems1,
            @Nonnull Iterable<Long> elems2,
            @Nonnull Iterable<Long> elems3,
            @Nonnull Iterable<Long> elems4,
            @Nonnull Iterable<Long> elems5);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Iterator<Long> elements);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.LongConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull long[] elements);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     LongSet newImmutableSet(@Nonnull Long[] elements);


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
     LongSet newImmutableSetOf(long e1);

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
     LongSet newImmutableSetOf(long e1, long e2);

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
     LongSet newImmutableSetOf(long e1, long e2, long e3);

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
     LongSet newImmutableSetOf(long e1, long e2, long e3,
            long e4);

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
     LongSet newImmutableSetOf(long e1, long e2, long e3,
            long e4, long e5, long... restElements);
}

