/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set;

import net.openhft.koloboke.collect.ContainerFactory;
import net.openhft.koloboke.collect.Equivalence;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Iterator;


/**
 * An immutable factory of {@code FloatSet}s.
 *
 * 
 * @param <F> the concrete factory type which extends this interface
 * @see FloatSet
 */
public interface FloatSetFactory<
        F extends FloatSetFactory<F>> extends ContainerFactory<F> {


    

    

    

    


    /**
     * Constructs a new empty mutable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
     FloatSet newMutableSet();

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
     FloatSet newMutableSet(int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elements, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterator<Float> elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull float[] elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Float[] elements, int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elements);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     FloatSet newMutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Iterator<Float> elements);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull float[] elements);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newMutableSet(@Nonnull Float[] elements);


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
     FloatSet newMutableSetOf(float e1);

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
     FloatSet newMutableSetOf(float e1, float e2);

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
     FloatSet newMutableSetOf(float e1, float e2, float e3);

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
     FloatSet newMutableSetOf(float e1, float e2, float e3,
            float e4);

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
     FloatSet newMutableSetOf(float e1, float e2, float e3,
            float e4, float e5, float... restElements);

    /**
     * Constructs a new empty updatable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
     FloatSet newUpdatableSet();

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
     FloatSet newUpdatableSet(int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elements, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterator<Float> elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull float[] elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Float[] elements, int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elements);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     FloatSet newUpdatableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Iterator<Float> elements);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull float[] elements);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newUpdatableSet(@Nonnull Float[] elements);


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
     FloatSet newUpdatableSetOf(float e1);

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
     FloatSet newUpdatableSetOf(float e1, float e2);

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
     FloatSet newUpdatableSetOf(float e1, float e2, float e3);

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
     FloatSet newUpdatableSetOf(float e1, float e2, float e3,
            float e4);

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
     FloatSet newUpdatableSetOf(float e1, float e2, float e3,
            float e4, float e5, float... restElements);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elements, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5, int expectedSize);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterator<Float> elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull float[] elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Float[] elements, int expectedSize);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elements);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     FloatSet newImmutableSet(@Nonnull Iterable<Float> elems1,
            @Nonnull Iterable<Float> elems2,
            @Nonnull Iterable<Float> elems3,
            @Nonnull Iterable<Float> elems4,
            @Nonnull Iterable<Float> elems5);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Iterator<Float> elements);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.FloatConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull float[] elements);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     FloatSet newImmutableSet(@Nonnull Float[] elements);


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
     FloatSet newImmutableSetOf(float e1);

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
     FloatSet newImmutableSetOf(float e1, float e2);

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
     FloatSet newImmutableSetOf(float e1, float e2, float e3);

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
     FloatSet newImmutableSetOf(float e1, float e2, float e3,
            float e4);

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
     FloatSet newImmutableSetOf(float e1, float e2, float e3,
            float e4, float e5, float... restElements);
}

