/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.map.hash;

import net.openhft.koloboke.collect.map.LongShortMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashLongShortMap}s, and
 * the default {@link HashLongShortMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashLongShortMap
 */
public final class HashLongShortMaps {

    private static final ServiceLoader<HashLongShortMapFactory> LOADER =
            ServiceLoader.load(HashLongShortMapFactory.class);
    private static HashLongShortMapFactory defaultFactory = null;

    /**
     * Returns the default {@link HashLongShortMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
     
     
     * @return the default {@link HashLongShortMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashLongShortMapFactory} are provided
     */
    @Nonnull
    public static  HashLongShortMapFactory getDefaultFactory() {
        if (defaultFactory != null) {
            return (HashLongShortMapFactory) defaultFactory;
        } else {
            // synchronization?
            return (HashLongShortMapFactory)
                    (defaultFactory = LOADER.iterator().next());
        }
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap() newMutableMap()}.
     *
     
     
     * @return a new empty mutable map
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap() {
        return getDefaultFactory().newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(int expectedSize) {
        return getDefaultFactory().newMutableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(long[], short[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull long[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Long[],
     * Short[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map) {
        return getDefaultFactory().newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2) {
        return getDefaultFactory().newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3) {
        return getDefaultFactory().newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(long[], short[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull long[] keys, @Nonnull short[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Long[],
     * Short[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newMutableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMapOf(long, short
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static  HashLongShortMap newMutableMapOf(
            long k1, short v1) {
        return getDefaultFactory().newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMapOf(long, short,
     * long, short) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newMutableMapOf(
            long k1, short v1, long k2, short v2) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMapOf(long, short,
     * long, short, long, short) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newMutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMapOf(long, short,
     * long, short, long, short, long, short
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newMutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newMutableMapOf(long, short,
     * long, short, long, short, long, short,
     * long, short) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newMutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4,
            long k5, short v5) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
     
     
     * @return a new empty updatable map
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap() {
        return getDefaultFactory().newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(int expectedSize) {
        return getDefaultFactory().newUpdatableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(long[], short[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull long[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Long[],
     * Short[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map) {
        return getDefaultFactory().newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2) {
        return getDefaultFactory().newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(long[], short[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull long[] keys, @Nonnull short[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Long[],
     * Short[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMapOf(long, short
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMapOf(
            long k1, short v1) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMapOf(long, short,
     * long, short) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMapOf(
            long k1, short v1, long k2, short v2) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMapOf(long, short,
     * long, short, long, short) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMapOf(long, short,
     * long, short, long, short, long, short
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newUpdatableMapOf(long, short,
     * long, short, long, short, long, short,
     * long, short) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newUpdatableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4,
            long k5, short v5) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(long[], short[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull long[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Long[],
     * Short[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map) {
        return getDefaultFactory().newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2) {
        return getDefaultFactory().newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Map<Long, Short> map1,
            @Nonnull Map<Long, Short> map2,
            @Nonnull Map<Long, Short> map3,
            @Nonnull Map<Long, Short> map4,
            @Nonnull Map<Long, Short> map5) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongShortConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(long[], short[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull long[] keys, @Nonnull short[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Long[],
     * Short[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Long[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMap(
            @Nonnull Iterable<Long> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMapOf(long, short
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMapOf(
            long k1, short v1) {
        return getDefaultFactory().newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMapOf(long, short,
     * long, short) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMapOf(
            long k1, short v1, long k2, short v2) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMapOf(long, short,
     * long, short, long, short) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMapOf(long, short,
     * long, short, long, short, long, short
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashLongShortMapFactory#newImmutableMapOf(long, short,
     * long, short, long, short, long, short,
     * long, short) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static  HashLongShortMap newImmutableMapOf(
            long k1, short v1, long k2, short v2,
            long k3, short v3, long k4, short v4,
            long k5, short v5) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashLongShortMaps() {}
}

