/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.map.hash;

import net.openhft.koloboke.collect.map.IntByteMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashIntByteMap}s, and
 * the default {@link HashIntByteMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashIntByteMap
 */
public final class HashIntByteMaps {

    private static final ServiceLoader<HashIntByteMapFactory> LOADER =
            ServiceLoader.load(HashIntByteMapFactory.class);
    private static HashIntByteMapFactory defaultFactory = null;

    /**
     * Returns the default {@link HashIntByteMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
     
     
     * @return the default {@link HashIntByteMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashIntByteMapFactory} are provided
     */
    @Nonnull
    public static  HashIntByteMapFactory getDefaultFactory() {
        if (defaultFactory != null) {
            return (HashIntByteMapFactory) defaultFactory;
        } else {
            // synchronization?
            return (HashIntByteMapFactory)
                    (defaultFactory = LOADER.iterator().next());
        }
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap() newMutableMap()}.
     *
     
     
     * @return a new empty mutable map
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap() {
        return getDefaultFactory().newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(int expectedSize) {
        return getDefaultFactory().newMutableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(int[], byte[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull int[] keys, @Nonnull byte[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Integer[],
     * Byte[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map) {
        return getDefaultFactory().newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2) {
        return getDefaultFactory().newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3) {
        return getDefaultFactory().newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(int[], byte[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull int[] keys, @Nonnull byte[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Integer[],
     * Byte[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newMutableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            ) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMapOf(int, byte
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static  HashIntByteMap newMutableMapOf(
            int k1, byte v1) {
        return getDefaultFactory().newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMapOf(int, byte,
     * int, byte) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newMutableMapOf(
            int k1, byte v1, int k2, byte v2) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMapOf(int, byte,
     * int, byte, int, byte) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newMutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMapOf(int, byte,
     * int, byte, int, byte, int, byte
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newMutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newMutableMapOf(int, byte,
     * int, byte, int, byte, int, byte,
     * int, byte) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newMutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4,
            int k5, byte v5) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
     
     
     * @return a new empty updatable map
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap() {
        return getDefaultFactory().newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(int expectedSize) {
        return getDefaultFactory().newUpdatableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(int[], byte[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull int[] keys, @Nonnull byte[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Integer[],
     * Byte[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map) {
        return getDefaultFactory().newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2) {
        return getDefaultFactory().newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(int[], byte[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull int[] keys, @Nonnull byte[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Integer[],
     * Byte[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            ) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMapOf(int, byte
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMapOf(
            int k1, byte v1) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMapOf(int, byte,
     * int, byte) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMapOf(int, byte,
     * int, byte, int, byte) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMapOf(int, byte,
     * int, byte, int, byte, int, byte
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newUpdatableMapOf(int, byte,
     * int, byte, int, byte, int, byte,
     * int, byte) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newUpdatableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4,
            int k5, byte v5) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(int[], byte[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull int[] keys, @Nonnull byte[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Integer[],
     * Byte[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map) {
        return getDefaultFactory().newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2) {
        return getDefaultFactory().newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Map<Integer, Byte> map1,
            @Nonnull Map<Integer, Byte> map2,
            @Nonnull Map<Integer, Byte> map3,
            @Nonnull Map<Integer, Byte> map4,
            @Nonnull Map<Integer, Byte> map5) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.IntByteConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(int[], byte[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull int[] keys, @Nonnull byte[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Integer[],
     * Byte[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Integer[] keys, @Nonnull Byte[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMap(
            @Nonnull Iterable<Integer> keys, @Nonnull Iterable<Byte> values
            ) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMapOf(int, byte
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMapOf(
            int k1, byte v1) {
        return getDefaultFactory().newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMapOf(int, byte,
     * int, byte) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMapOf(
            int k1, byte v1, int k2, byte v2) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMapOf(int, byte,
     * int, byte, int, byte) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMapOf(int, byte,
     * int, byte, int, byte, int, byte
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntByteMapFactory#newImmutableMapOf(int, byte,
     * int, byte, int, byte, int, byte,
     * int, byte) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static  HashIntByteMap newImmutableMapOf(
            int k1, byte v1, int k2, byte v2,
            int k3, byte v3, int k4, byte v4,
            int k5, byte v5) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashIntByteMaps() {}
}

