/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashObjByteMap;
import javax.annotation.Nonnull;


public abstract class UpdatableQHashSeparateKVObjByteMapSO<K>
        extends UpdatableQHashSeparateKVObjKeyMap<K>
        implements HashObjByteMap<K>, InternalObjByteMapOps<K>, SeparateKVObjByteQHash {

    byte[] values;

    void copy(SeparateKVObjByteQHash hash) {
        super.copy(hash);
        values = hash.valueArray().clone();
    }

    void move(SeparateKVObjByteQHash hash) {
        super.move(hash);
        values = hash.valueArray();
    }

    @Override
    @Nonnull
    public byte[] valueArray() {
        return values;
    }

    
    int valueIndex(byte value) {
        if (isEmpty())
            return -1;
        int index = -1;
        int mc = modCount();
        Object[] keys = set;
        byte[] vals = values;
        for (int i = keys.length - 1; i >= 0; i--) {
            if (keys[i] != FREE) {
                if (value == vals[i]) {
                    index = i;
                    break;
                }
            }
        }
        if (mc != modCount())
            throw new java.util.ConcurrentModificationException();
        return index;
    }

    @Override public
    boolean containsValue(byte value) {
        return valueIndex(value) >= 0;
    }

    boolean removeValue(byte value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsValue(Object value) {
        return containsValue(((Byte) value).byteValue());
    }

    int insert(K key, byte value) {
        if (key != null) {
            // noinspection unchecked
            K[] keys = (K[]) set;
            int capacity, index;
            K cur;
            if ((cur = keys[index = SeparateKVObjKeyMixing.mix(keyHashCode(key)) % (capacity = keys.length)]) == key) {
                // key is present
                return index;
            } else {
                keyAbsent:
                if (cur != FREE) {
                    if (keyEquals(key, cur)) {
                        // key is present
                        return index;
                    } else {
                        int bIndex = index, fIndex = index, step = 1;
                        while (true) {
                            if ((bIndex -= step) < 0) bIndex += capacity;
                            if ((cur = keys[bIndex]) == key) {
                                // key is present
                                return bIndex;
                            } else if (cur == FREE) {
                                index = bIndex;
                                break keyAbsent;
                            }
                            else if (keyEquals(key, cur)) {
                                // key is present
                                return bIndex;
                            }
                            int t;
                            if ((t = (fIndex += step) - capacity) >= 0) fIndex = t;
                            if ((cur = keys[fIndex]) == key) {
                                // key is present
                                return fIndex;
                            } else if (cur == FREE) {
                                index = fIndex;
                                break keyAbsent;
                            }
                            else if (keyEquals(key, cur)) {
                                // key is present
                                return fIndex;
                            }
                            step += 2;
                        }
                    }
                }
                // key is absent
                incrementModCount();
                keys[index] = key;
                values[index] = value;
                postInsertHook();
                return -1;
            }
        } else {
            return insertNullKey(value);
        }
    }

    int insertNullKey(byte value) {
        // noinspection unchecked
        K[] keys = (K[]) set;
        int capacity = keys.length;
        int index;
        K cur;
        if ((cur = keys[index = 0]) == null) {
            // key is present
            return index;
        } else {
            keyAbsent:
            if (cur != FREE) {
                int bIndex = index, fIndex = index, step = 1;
                while (true) {
                    if ((bIndex -= step) < 0) bIndex += capacity;
                    if ((cur = keys[bIndex]) == null) {
                        // key is present
                        return bIndex;
                    } else if (cur == FREE) {
                        index = bIndex;
                        break keyAbsent;
                    }
                    int t;
                    if ((t = (fIndex += step) - capacity) >= 0) fIndex = t;
                    if ((cur = keys[fIndex]) == null) {
                        // key is present
                        return fIndex;
                    } else if (cur == FREE) {
                        index = fIndex;
                        break keyAbsent;
                    }
                    step += 2;
                }
            }
            // key is absent
            incrementModCount();
            keys[index] = null;
            values[index] = value;
            postInsertHook();
            return -1;
        }
    }

    @Override
    void allocateArrays(int capacity) {
        super.allocateArrays(capacity);
        values = new byte[capacity];
    }
}

