/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.*;
import net.openhft.collect.hash.*;
import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashCharShortMapFactory;
import net.openhft.function.CharShortConsumer;
import java.util.function.Predicate;
import net.openhft.collect.map.hash.HashCharShortMap;

import javax.annotation.Nonnull;
import java.util.*;

import static net.openhft.collect.impl.Containers.sizeAsInt;
import static net.openhft.collect.impl.hash.LHashCapacities.configIsSuitableForMutableLHash;


public abstract class QHashParallelKVCharShortMapFactoryGO
        extends QHashParallelKVCharShortMapFactorySO {

    QHashParallelKVCharShortMapFactoryGO(HashConfig hashConf, int defaultExpectedSize
            , char lower, char upper) {
        super(hashConf, defaultExpectedSize, lower, upper);
    }

    

    abstract HashCharShortMapFactory thisWith(HashConfig hashConf, int defaultExpectedSize, char lower, char upper);

    abstract HashCharShortMapFactory lHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, char lower, char upper);

    abstract HashCharShortMapFactory qHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, char lower, char upper);

    @Override
    public final HashCharShortMapFactory withHashConfig(@Nonnull HashConfig hashConf) {
        if (configIsSuitableForMutableLHash(hashConf))
            return lHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                    , getLowerKeyDomainBound(), getUpperKeyDomainBound());
        return qHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    @Override
    public final HashCharShortMapFactory withDefaultExpectedSize(int defaultExpectedSize) {
        if (defaultExpectedSize == getDefaultExpectedSize())
            return this;
        return thisWith(getHashConfig(), defaultExpectedSize
                
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    final HashCharShortMapFactory withDomain(char lower, char upper) {
        if (lower == getLowerKeyDomainBound() && upper == getUpperKeyDomainBound())
            return this;
        return thisWith(getHashConfig(), getDefaultExpectedSize(), lower, upper);
    }

    @Override
    public final HashCharShortMapFactory withKeysDomain(char lower, char upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minPossibleKey shouldn't be greater " +
                    "than maxPossibleKey");
        return withDomain(lower, upper);
    }

    @Override
    public final HashCharShortMapFactory withKeysDomainComplement(char lower, char upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minImpossibleKey shouldn't be greater " +
                    "than maxImpossibleKey");
        return withDomain((char) (upper + 1), (char) (lower - 1));
    }

    @Override
    public String toString() {
        return "HashCharShortMapFactory[" + commonString() + keySpecialString() +
                ",defaultValue=" + getDefaultValue() +
        "]";
    }

    @Override
    public int hashCode() {
        int hashCode = keySpecialHashCode(commonHashCode());
        hashCode = hashCode * 31 + Primitives.hashCode(getDefaultValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj instanceof HashCharShortMapFactory) {
            HashCharShortMapFactory factory = (HashCharShortMapFactory) obj;
            return commonEquals(factory) && keySpecialEquals(factory) &&
                    // boxing to treat NaNs correctly
                   ((Short) getDefaultValue()).equals(factory.getDefaultValue())
                    ;
        } else {
            return false;
        }
    }

    @Override
    public short getDefaultValue() {
        return (short) 0;
    }

    

    

    

    

    

    

    
    

    
    

    private  UpdatableQHashParallelKVCharShortMapGO shrunk(
            UpdatableQHashParallelKVCharShortMapGO map) {
        Predicate<HashContainer> shrinkCondition;
        if ((shrinkCondition = hashConf.getShrinkCondition()) != null) {
            if (shrinkCondition.test(map))
                map.shrink();
        }
        return map;
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap() {
        return newUpdatableMap(getDefaultExpectedSize());
    }
    @Override
    public  MutableQHashParallelKVCharShortMapGO newMutableMap() {
        return newMutableMap(getDefaultExpectedSize());
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map) {
        return shrunk(super.newUpdatableMap(map));
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        return newUpdatableMap(map1, map2, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        return newUpdatableMap(map1, map2, map3, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3, Map<Character, Short> map4) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        return newUpdatableMap(map1, map2, map3, map4, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3, Map<Character, Short> map4,
            Map<Character, Short> map5) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        expectedSize += (long) map5.size();
        return newUpdatableMap(map1, map2, map3, map4, map5, sizeAsInt(expectedSize));
    }


    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3, int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3, Map<Character, Short> map4,
            int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Map<Character, Short> map1, Map<Character, Short> map2,
            Map<Character, Short> map3, Map<Character, Short> map4,
            Map<Character, Short> map5, int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        map.putAll(map5);
        return shrunk(map);
    }


    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier) {
        return newUpdatableMap(entriesSupplier, getDefaultExpectedSize());
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier,
            int expectedSize) {
        final UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        entriesSupplier.accept(new CharShortConsumer() {
             @Override
             public void accept(char k, short v) {
                 map.put(k, v);
             }
         });
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(char[] keys,
            short[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(char[] keys,
            short[] values, int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Character[] keys, Short[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(
            Character[] keys, Short[] values, int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(Iterable<Character> keys,
            Iterable<Short> values) {
        int expectedSize = keys instanceof Collection ? ((Collection) keys).size() :
                getDefaultExpectedSize();
        return newUpdatableMap(keys, values, expectedSize);
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMap(Iterable<Character> keys,
            Iterable<Short> values, int expectedSize) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(expectedSize);
        Iterator<Character> keysIt = keys.iterator();
        Iterator<Short> valuesIt = values.iterator();
        try {
            while (keysIt.hasNext()) {
                map.put(keysIt.next(), valuesIt.next());
            }
            return shrunk(map);
        } catch (NoSuchElementException e) {
            throw new IllegalArgumentException(
                    "keys and values iterables must have the same size", e);
        }
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMapOf(
            char k1, short v1) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(1);
        map.put(k1, v1);
        return map;
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMapOf(
            char k1, short v1, char k2, short v2) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(2);
        map.put(k1, v1);
        map.put(k2, v2);
        return map;
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMapOf(
            char k1, short v1, char k2, short v2,
            char k3, short v3) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(3);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        return map;
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMapOf(
            char k1, short v1, char k2, short v2,
            char k3, short v3, char k4, short v4) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(4);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        return map;
    }

    @Override
    public  UpdatableQHashParallelKVCharShortMapGO newUpdatableMapOf(
            char k1, short v1, char k2, short v2,
            char k3, short v3, char k4, short v4,
            char k5, short v5) {
        UpdatableQHashParallelKVCharShortMapGO map = newUpdatableMap(5);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        map.put(k5, v5);
        return map;
    }

    
    


    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, Map<Character, Short> map5, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashCharShortMap newMutableMap(
            Map<Character, Short> map) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashCharShortMap newMutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, Map<Character, Short> map5) {
        MutableQHashParallelKVCharShortMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashCharShortMap newMutableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier,
            int expectedSize) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(char[] keys,
            short[] values) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(char[] keys,
            short[] values, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(
            Character[] keys, Short[] values) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(
            Character[] keys, Short[] values, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(Iterable<Character> keys,
            Iterable<Short> values) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMap(Iterable<Character> keys,
            Iterable<Short> values, int expectedSize) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMapOf(char k1, short v1) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMapOf(char k1, short v1,
             char k2, short v2) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3,
             char k4, short v4) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashCharShortMap newMutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3,
             char k4, short v4, char k5, short v5) {
        MutableQHashParallelKVCharShortMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
    
    


    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, Map<Character, Short> map5, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashCharShortMap newImmutableMap(
            Map<Character, Short> map) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Map<Character, Short> map1,
            Map<Character, Short> map2, Map<Character, Short> map3,
            Map<Character, Short> map4, Map<Character, Short> map5) {
        ImmutableQHashParallelKVCharShortMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashCharShortMap newImmutableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(
            net.openhft.function.Consumer<CharShortConsumer> entriesSupplier,
            int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(char[] keys,
            short[] values) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(char[] keys,
            short[] values, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(
            Character[] keys, Short[] values) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(
            Character[] keys, Short[] values, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Iterable<Character> keys,
            Iterable<Short> values) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMap(Iterable<Character> keys,
            Iterable<Short> values, int expectedSize) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMapOf(char k1, short v1) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMapOf(char k1, short v1,
             char k2, short v2) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3,
             char k4, short v4) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashCharShortMap newImmutableMapOf(char k1, short v1,
             char k2, short v2, char k3, short v3,
             char k4, short v4, char k5, short v5) {
        ImmutableQHashParallelKVCharShortMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
}

