/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.*;
import net.openhft.collect.impl.*;
import net.openhft.collect.set.hash.HashDoubleSet;
import javax.annotation.Nonnull;

import java.util.*;


public class MutableLHashDoubleSetGO extends MutableDoubleLHashSetSO
        implements HashDoubleSet, InternalDoubleCollectionOps {

    @Override
    final void copy(SeparateKVDoubleLHash hash) {
        int myMC = modCount(), hashMC = hash.modCount();
        super.copy(hash);
        if (myMC != modCount() || hashMC != hash.modCount())
            throw new ConcurrentModificationException();
    }

    @Override
    final void move(SeparateKVDoubleLHash hash) {
        int myMC = modCount(), hashMC = hash.modCount();
        super.move(hash);
        if (myMC != modCount() || hashMC != hash.modCount())
            throw new ConcurrentModificationException();
    }

    public int hashCode() {
        return setHashCode();
    }

    @Override
    public String toString() {
        return setToString();
    }

    @Override
    public boolean equals(Object obj) {
        return CommonSetOps.equals(this, obj);
    }

    @Override
    public boolean containsAll(@Nonnull Collection<?> c) {
        return CommonDoubleCollectionOps.containsAll(this, c);
    }

    @Nonnull
    @Override
    public DoubleCursor cursor() {
        return setCursor();
    }


    @Override
    public boolean add(Double e) {
        return add(e.doubleValue());
    }

    @Override
    public boolean add(long key) {
        long[] keys = set;
        int capacityMask, index;
        long cur;
        keyAbsent:
        if ((cur = keys[index = SeparateKVDoubleKeyMixing.mix(key) & (capacityMask = keys.length - 1)]) != FREE_BITS) {
            if (cur == key) {
                // key is present
                return false;
            } else {
                while (true) {
                    index = (index - 1) & capacityMask;
                    if ((cur = keys[index]) == FREE_BITS) {
                        break keyAbsent;
                    } else if (cur == key) {
                        // key is present
                        return false;
                    }
                }
            }
        }
        // key is absent
        incrementModCount();
        keys[index] = key;
        postInsertHook();
        return true;
    }

    @Override
    public boolean add(double key) {
        return add(Double.doubleToLongBits(key));
    }

    @Override
    public boolean addAll(@Nonnull Collection<? extends Double> c) {
        return CommonDoubleCollectionOps.addAll(this, c);
    }

    @Override
    public boolean remove(Object key) {
        return removeDouble(((Double) key).doubleValue());
    }


    @Override
    boolean justRemove(long key) {
        return removeDouble(key);
    }

    @Override
    public boolean removeDouble(long key) {
        long[] keys = set;
        int capacityMask = keys.length - 1;
        int index;
        long cur;
        keyPresent:
        if ((cur = keys[index = SeparateKVDoubleKeyMixing.mix(key) & capacityMask]) != key) {
            if (cur == FREE_BITS) {
                // key is absent
                return false;
            } else {
                while (true) {
                    index = (index - 1) & capacityMask;
                    if ((cur = keys[index]) == key) {
                        break keyPresent;
                    } else if (cur == FREE_BITS) {
                        // key is absent
                        return false;
                    }
                }
            }
        }
        // key is present
        incrementModCount();
        int indexToRemove = index;
        int indexToShift = indexToRemove;
        int shiftDistance = 1;
        while (true) {
            indexToShift = (indexToShift - 1) & capacityMask;
            long keyToShift;
            if ((keyToShift = keys[indexToShift]) == FREE_BITS) {
                break;
            }
            if (((SeparateKVDoubleKeyMixing.mix(keyToShift) - indexToShift) & capacityMask) >= shiftDistance) {
                keys[indexToRemove] = keyToShift;
                indexToRemove = indexToShift;
                shiftDistance = 1;
            } else {
                shiftDistance++;
                if (indexToShift == 1 + index) {
                    throw new java.util.ConcurrentModificationException();
                }
            }
        }
        keys[indexToRemove] = FREE_BITS;
        postRemoveHook();
        return true;
    }

    @Override
    public boolean removeDouble(double key) {
        return removeDouble(Double.doubleToLongBits(key));
    }


    @Override
    public boolean removeAll(@Nonnull Collection<?> c) {
        if (c instanceof DoubleCollection) {
            if (c instanceof InternalDoubleCollectionOps) {
                InternalDoubleCollectionOps c2 = (InternalDoubleCollectionOps) c;
                if (c2.size() < this.size()) {
                    
                    return c2.reverseRemoveAllFrom(this);
                }
            }
            return removeAll(this, (DoubleCollection) c);
        }
        return removeAll(this, c);
    }

    @Override
    public boolean retainAll(@Nonnull Collection<?> c) {
        return retainAll(this, c);
    }
}

