/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.*;
import net.openhft.collect.hash.*;
import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashByteIntMapFactory;
import net.openhft.function.ByteIntConsumer;
import java.util.function.Predicate;
import net.openhft.collect.map.hash.HashByteIntMap;

import javax.annotation.Nonnull;
import java.util.*;

import static net.openhft.collect.impl.Containers.sizeAsInt;
import static net.openhft.collect.impl.hash.LHashCapacities.configIsSuitableForMutableLHash;


public abstract class LHashSeparateKVByteIntMapFactoryGO
        extends LHashSeparateKVByteIntMapFactorySO {

    LHashSeparateKVByteIntMapFactoryGO(HashConfig hashConf, int defaultExpectedSize
            , byte lower, byte upper) {
        super(hashConf, defaultExpectedSize, lower, upper);
    }

    

    abstract HashByteIntMapFactory thisWith(HashConfig hashConf, int defaultExpectedSize, byte lower, byte upper);

    abstract HashByteIntMapFactory lHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, byte lower, byte upper);

    abstract HashByteIntMapFactory qHashLikeThisWith(HashConfig hashConf, int defaultExpectedSize, byte lower, byte upper);

    @Override
    public final HashByteIntMapFactory withHashConfig(@Nonnull HashConfig hashConf) {
        if (configIsSuitableForMutableLHash(hashConf))
            return lHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                    , getLowerKeyDomainBound(), getUpperKeyDomainBound());
        return qHashLikeThisWith(hashConf, getDefaultExpectedSize()
            
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    @Override
    public final HashByteIntMapFactory withDefaultExpectedSize(int defaultExpectedSize) {
        if (defaultExpectedSize == getDefaultExpectedSize())
            return this;
        return thisWith(getHashConfig(), defaultExpectedSize
                
                , getLowerKeyDomainBound(), getUpperKeyDomainBound());
    }

    final HashByteIntMapFactory withDomain(byte lower, byte upper) {
        if (lower == getLowerKeyDomainBound() && upper == getUpperKeyDomainBound())
            return this;
        return thisWith(getHashConfig(), getDefaultExpectedSize(), lower, upper);
    }

    @Override
    public final HashByteIntMapFactory withKeysDomain(byte lower, byte upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minPossibleKey shouldn't be greater " +
                    "than maxPossibleKey");
        return withDomain(lower, upper);
    }

    @Override
    public final HashByteIntMapFactory withKeysDomainComplement(byte lower, byte upper) {
        if (lower > upper)
            throw new IllegalArgumentException("minImpossibleKey shouldn't be greater " +
                    "than maxImpossibleKey");
        return withDomain((byte) (upper + 1), (byte) (lower - 1));
    }

    @Override
    public String toString() {
        return "HashByteIntMapFactory[" + commonString() + keySpecialString() +
                ",defaultValue=" + getDefaultValue() +
        "]";
    }

    @Override
    public int hashCode() {
        int hashCode = keySpecialHashCode(commonHashCode());
        hashCode = hashCode * 31 + Primitives.hashCode(getDefaultValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj instanceof HashByteIntMapFactory) {
            HashByteIntMapFactory factory = (HashByteIntMapFactory) obj;
            return commonEquals(factory) && keySpecialEquals(factory) &&
                    // boxing to treat NaNs correctly
                   ((Integer) getDefaultValue()).equals(factory.getDefaultValue())
                    ;
        } else {
            return false;
        }
    }

    @Override
    public int getDefaultValue() {
        return 0;
    }

    

    

    

    

    

    

    
    

    
    

    private  UpdatableLHashSeparateKVByteIntMapGO shrunk(
            UpdatableLHashSeparateKVByteIntMapGO map) {
        Predicate<HashContainer> shrinkCondition;
        if ((shrinkCondition = hashConf.getShrinkCondition()) != null) {
            if (shrinkCondition.test(map))
                map.shrink();
        }
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap() {
        return newUpdatableMap(getDefaultExpectedSize());
    }
    @Override
    public  MutableLHashSeparateKVByteIntMapGO newMutableMap() {
        return newMutableMap(getDefaultExpectedSize());
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map) {
        return shrunk(super.newUpdatableMap(map));
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        return newUpdatableMap(map1, map2, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        return newUpdatableMap(map1, map2, map3, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3, Map<Byte, Integer> map4) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        return newUpdatableMap(map1, map2, map3, map4, sizeAsInt(expectedSize));
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3, Map<Byte, Integer> map4,
            Map<Byte, Integer> map5) {
        long expectedSize = (long) map1.size();
        expectedSize += (long) map2.size();
        expectedSize += (long) map3.size();
        expectedSize += (long) map4.size();
        expectedSize += (long) map5.size();
        return newUpdatableMap(map1, map2, map3, map4, map5, sizeAsInt(expectedSize));
    }


    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3, int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3, Map<Byte, Integer> map4,
            int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Map<Byte, Integer> map1, Map<Byte, Integer> map2,
            Map<Byte, Integer> map3, Map<Byte, Integer> map4,
            Map<Byte, Integer> map5, int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        map.putAll(map1);
        map.putAll(map2);
        map.putAll(map3);
        map.putAll(map4);
        map.putAll(map5);
        return shrunk(map);
    }


    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier) {
        return newUpdatableMap(entriesSupplier, getDefaultExpectedSize());
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier,
            int expectedSize) {
        final UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        entriesSupplier.accept(new ByteIntConsumer() {
             @Override
             public void accept(byte k, int v) {
                 map.put(k, v);
             }
         });
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(byte[] keys,
            int[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(byte[] keys,
            int[] values, int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Byte[] keys, Integer[] values) {
        return newUpdatableMap(keys, values, keys.length);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(
            Byte[] keys, Integer[] values, int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        int keysLen = keys.length;
        if (keysLen != values.length)
            throw new IllegalArgumentException("keys and values arrays must have the same size");
        for (int i = 0; i < keys.length; i++) {
            map.put(keys[i], values[i]);
        }
        return shrunk(map);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(Iterable<Byte> keys,
            Iterable<Integer> values) {
        int expectedSize = keys instanceof Collection ? ((Collection) keys).size() :
                getDefaultExpectedSize();
        return newUpdatableMap(keys, values, expectedSize);
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMap(Iterable<Byte> keys,
            Iterable<Integer> values, int expectedSize) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(expectedSize);
        Iterator<Byte> keysIt = keys.iterator();
        Iterator<Integer> valuesIt = values.iterator();
        try {
            while (keysIt.hasNext()) {
                map.put(keysIt.next(), valuesIt.next());
            }
            return shrunk(map);
        } catch (NoSuchElementException e) {
            throw new IllegalArgumentException(
                    "keys and values iterables must have the same size", e);
        }
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMapOf(
            byte k1, int v1) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(1);
        map.put(k1, v1);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMapOf(
            byte k1, int v1, byte k2, int v2) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(2);
        map.put(k1, v1);
        map.put(k2, v2);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMapOf(
            byte k1, int v1, byte k2, int v2,
            byte k3, int v3) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(3);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMapOf(
            byte k1, int v1, byte k2, int v2,
            byte k3, int v3, byte k4, int v4) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(4);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        return map;
    }

    @Override
    public  UpdatableLHashSeparateKVByteIntMapGO newUpdatableMapOf(
            byte k1, int v1, byte k2, int v2,
            byte k3, int v3, byte k4, int v4,
            byte k5, int v5) {
        UpdatableLHashSeparateKVByteIntMapGO map = newUpdatableMap(5);
        map.put(k1, v1);
        map.put(k2, v2);
        map.put(k3, v3);
        map.put(k4, v4);
        map.put(k5, v5);
        return map;
    }

    
    


    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, Map<Byte, Integer> map5, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashByteIntMap newMutableMap(
            Map<Byte, Integer> map) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashByteIntMap newMutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, Map<Byte, Integer> map5) {
        MutableLHashSeparateKVByteIntMapGO res = uninitializedMutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashByteIntMap newMutableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier,
            int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(byte[] keys,
            int[] values) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(byte[] keys,
            int[] values, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(
            Byte[] keys, Integer[] values) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(
            Byte[] keys, Integer[] values, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(Iterable<Byte> keys,
            Iterable<Integer> values) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMap(Iterable<Byte> keys,
            Iterable<Integer> values, int expectedSize) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMapOf(byte k1, int v1) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMapOf(byte k1, int v1,
             byte k2, int v2) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3,
             byte k4, int v4) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashByteIntMap newMutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3,
             byte k4, int v4, byte k5, int v5) {
        MutableLHashSeparateKVByteIntMapGO map = uninitializedMutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
    
    


    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, expectedSize));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, Map<Byte, Integer> map5, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5, expectedSize));
        return res;
    }

    
    

    @Override
    public  HashByteIntMap newImmutableMap(
            Map<Byte, Integer> map) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4));
        return res;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Map<Byte, Integer> map1,
            Map<Byte, Integer> map2, Map<Byte, Integer> map3,
            Map<Byte, Integer> map4, Map<Byte, Integer> map5) {
        ImmutableLHashSeparateKVByteIntMapGO res = uninitializedImmutableMap();
        res.move(newUpdatableMap(map1, map2, map3, map4, map5));
        return res;
    }


    @Override
    public  HashByteIntMap newImmutableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(
            net.openhft.function.Consumer<ByteIntConsumer> entriesSupplier,
            int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(entriesSupplier, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(byte[] keys,
            int[] values) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(byte[] keys,
            int[] values, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(
            Byte[] keys, Integer[] values) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(
            Byte[] keys, Integer[] values, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Iterable<Byte> keys,
            Iterable<Integer> values) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMap(Iterable<Byte> keys,
            Iterable<Integer> values, int expectedSize) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMap(keys, values, expectedSize));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMapOf(byte k1, int v1) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMapOf(byte k1, int v1,
             byte k2, int v2) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3,
             byte k4, int v4) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4));
        return map;
    }

    @Override
    public  HashByteIntMap newImmutableMapOf(byte k1, int v1,
             byte k2, int v2, byte k3, int v3,
             byte k4, int v4, byte k5, int v5) {
        ImmutableLHashSeparateKVByteIntMapGO map = uninitializedImmutableMap();
        map.move(newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5));
        return map;
    }
}

