/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.impl.hash;

import net.openhft.collect.impl.*;
import net.openhft.collect.map.hash.HashShortDoubleMap;
import javax.annotation.Nonnull;


public abstract class UpdatableLHashSeparateKVShortDoubleMapSO
        extends UpdatableLHashSeparateKVShortKeyMap
        implements HashShortDoubleMap, InternalShortDoubleMapOps, SeparateKVShortDoubleLHash {

    long[] values;

    void copy(SeparateKVShortDoubleLHash hash) {
        super.copy(hash);
        values = hash.valueArray().clone();
    }

    void move(SeparateKVShortDoubleLHash hash) {
        super.move(hash);
        values = hash.valueArray();
    }

    @Override
    @Nonnull
    public long[] valueArray() {
        return values;
    }

    
    int valueIndex(long value) {
        if (isEmpty())
            return -1;
        int index = -1;
        int mc = modCount();
        short free = freeValue;
        short[] keys = set;
        long[] vals = values;
        for (int i = keys.length - 1; i >= 0; i--) {
            if (keys[i] != free) {
                if (value == vals[i]) {
                    index = i;
                    break;
                }
            }
        }
        if (mc != modCount())
            throw new java.util.ConcurrentModificationException();
        return index;
    }

    
    boolean containsValue(long value) {
        return valueIndex(value) >= 0;
    }

    boolean removeValue(long value) {
        throw new UnsupportedOperationException();
    }
    
    int valueIndex(double value) {
        if (isEmpty())
            return -1;
        long val = Double.doubleToLongBits(value);
        int index = -1;
        int mc = modCount();
        short free = freeValue;
        short[] keys = set;
        long[] vals = values;
        for (int i = keys.length - 1; i >= 0; i--) {
            if (keys[i] != free) {
                if (val == vals[i]) {
                    index = i;
                    break;
                }
            }
        }
        if (mc != modCount())
            throw new java.util.ConcurrentModificationException();
        return index;
    }

    @Override public
    boolean containsValue(double value) {
        return valueIndex(value) >= 0;
    }

    boolean removeValue(double value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsValue(Object value) {
        return containsValue(((Double) value).doubleValue());
    }

    int insert(short key, long value) {
        short free;
        if (key == (free = freeValue)) {
            free = changeFree();
        }
        short[] keys = set;
        int capacityMask, index;
        short cur;
        if ((cur = keys[index = SeparateKVShortKeyMixing.mix(key) & (capacityMask = keys.length - 1)]) == key) {
            // key is present
            return index;
        } else {
            keyAbsent:
            if (cur != free) {
                while (true) {
                    index = (index - 1) & capacityMask;
                    if ((cur = keys[index]) == key) {
                        // key is present
                        return index;
                    } else if (cur == free) {
                        break keyAbsent;
                    }
                }
            }
            // key is absent
            incrementModCount();
            keys[index] = key;
            values[index] = value;
            postInsertHook();
            return -1;
        }
    }


    @Override
    void allocateArrays(int capacity) {
        super.allocateArrays(capacity);
        values = new long[capacity];
    }
}

