/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.map;

import net.openhft.collect.Container;
import net.openhft.collect.Equivalence;
import net.openhft.function.BiConsumer;
import net.openhft.function.BiFunction;
import net.openhft.function.BiConsumer;
import net.openhft.function.BiPredicate;
import net.openhft.function.BiFunction;
import net.openhft.function.Function;
import net.openhft.function.Function;
import net.openhft.function.BiFunction;
import net.openhft.collect.ObjCollection;
import net.openhft.collect.set.ObjSet;
import net.openhft.collect.set.ObjSet;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import java.util.Map;


/**
 * 
 * The library's extension of the classic {@link Map} interface.
 * 
 *
 * @see ObjObjMapFactory
 */
public interface ObjObjMap<K, V> extends Map<K, V>, Container {

    /**
     * Returns the equivalence strategy for keys in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of key objects
     * (almost all methods, actually), are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for keys in this map,
     *         or {@code null} if it is the default {@link Object} equality
     * @see net.openhft.collect.map.hash.HashObjObjMapFactory#withKeyEquivalence
     */
    @Nullable
    Equivalence<K> keyEquivalence();

    /**
     * Returns the equivalence strategy for values in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of value objects,
     * for example, {@link #containsValue(Object)} and {@link #remove(Object, Object)},
     * are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for values in this map,
     *         or {@code null} if it is the default {@link Object} equality
     * @see ObjObjMapFactory#withValueEquivalence
     */
    @Nullable
    Equivalence<V> valueEquivalence();




    




    /**
     * Returns the value to which the specified key is mapped, or {@code defaultValue} if this map
     * contains no mapping for the key.
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped, or
     *         {@code defaultValue} if this map contains no mapping for the key
     * @throws ClassCastException if the key is of an inappropriate type for
     *         this map (optional restriction)
     * @throws NullPointerException if the specified key is null and this map
     *         does not permit null keys (optional restriction)
     */
    V getOrDefault(Object key, V defaultValue);



    /**
     * Performs the given {@code action} on each entry in this map until all entries
     * have been processed or the action throws an {@code Exception}.
     * Exceptions thrown by the action are relayed to the caller. The entries
     * will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which
     * differs from the entry set iterator.
     *
     * @param action The action to be performed for each entry
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    void forEach(@Nonnull BiConsumer<? super K, ? super V> action);


    /**
     * Checks the given {@code predicate} on each entry in this map until all entries
     * have been processed or the predicate returns {@code false} for some entry,
     * or throws an {@code Exception}. Exceptions thrown by the predicate are relayed to the caller.
     *
     * <p>The entries will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which differs from
     * the entry set iterator.
     *
     * <p>If the map is empty, this method returns {@code true} immediately.
     *
     * @return {@code true} if the predicate returned {@code true} for all entries of the map,
     *         {@code false} if it returned {@code false} for the entry
     * @param predicate the predicate to be checked for each entry
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean forEachWhile(@Nonnull BiPredicate<? super K, ? super V> predicate);

    /**
     * Returns a new cursor over the entries of this map. It's order is always correspond to the
     * entry set iterator order.
     *
     * @return a new cursor over the entries of this map
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    @Nonnull
    ObjObjCursor<K, V> cursor();


    /** {@inheritDoc} */
    @Override
    @Nonnull
    ObjSet<K> keySet();

    /** {@inheritDoc} */
    @Override
    @Nonnull
    ObjCollection<V> values();

    /** {@inheritDoc} */
    @Override
    @Nonnull
    ObjSet<Entry<K, V>> entrySet();

    

    

    

    

    


    

    


    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), associates
     * it with the given value and returns {@code null}, else returns the current value.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or {@code null}
     *         if there was no mapping for the key. (A {@code null} return
     *         can also indicate that the map previously associated {@code null}
     *         with the key, if the implementation supports such values.)
     * @throws UnsupportedOperationException if the {@code put} operation
    *         is not supported by this map
     * @throws ClassCastException if the class of the specified key or value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key or value is null,
     *         and this map does not permit null keys or values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     */
    @Nullable
    V putIfAbsent(K key, V value);


    

    

    

    


    /**
     * Attempts to compute a mapping for the specified key and its current mapped value
     * (or {@code null} if there is no current mapping).
     *
     * <p>If the function returns {@code null}, the mapping is removed (or
     * remains absent if initially absent).
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key or computed value
     *         
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V compute(K key, @Nonnull BiFunction<? super K, ? super V, ? extends V> remappingFunction);



    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), attempts
     * to compute its value using the given mapping function and enters it into this map
     *  unless {@code null}. The most common usage is to construct
     * a new object serving as an initial mapped value or memoized result.
     *
     * <p>If the function returns {@code null} no mapping is recorded.
     *
     * <p>If the function itself throws an (unchecked) exception, the exception is rethrown,
     * and no mapping is recorded.
     *
     * @param key key with which the specified value is to be associated
     * @param mappingFunction the function to compute a value
     * @return the current (existing or computed) value associated with
     *         the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key or computed value
     *         
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V computeIfAbsent(K key, @Nonnull Function<? super K, ? extends V> mappingFunction);



    /**
     * If the value for the specified key is present and non-null,
     * attempts to compute a new mapping given the key and its current mapped value.
     *
     * <p>If the function returns {@code null}, the mapping is removed.
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key,
     *         or "no entry" value
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key or computed value
     *         
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V computeIfPresent(K key, @Nonnull BiFunction<? super K, ? super V, ? extends V> remappingFunction);



    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), associates
     * it with the given value, otherwise, replaces the value with the results of the given
     * remapping function.
     *
     * This method may be of use when combining multiple mapped values for a key.
     *
     * <p>If the function returns {@code null}, the mapping is removed.
     *
     * <p>If the remappingFunction itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param value the value to use if absent
     * @param remappingFunction the function to recompute a value if present
     * @return the new value associated with the specified key
     * @throws NullPointerException if the {@code remappingFunction} is null
     *         or if the specified key is null and this map does not support null keys
     *         or if the specified value is null and this map does not support null values
     * @throws ClassCastException if the class of the
     *         specified key or computed value
     *         
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V merge(K key, V value, @Nonnull BiFunction<? super V, ? super V, ? extends V> remappingFunction);




    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param key key with which the specified value is associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key,
     *         or {@code null} if there was no mapping for the key.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with the key,
     *         if the implementation supports such values.)
     * @throws ClassCastException if the class of the specified key or value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key or value is null,
     *         and this map does not permit null keys or values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    @Nullable
    V replace(K key, V value);



    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param key key with which the specified value is associated
     * @param oldValue value expected to be associated with the specified key
     * @param newValue value to be associated with the specified key
     * @return {@code true} if the value was replaced
     * @throws ClassCastException if the class of the specified key or value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key or value is null,
     *         and this map does not permit null keys or values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    boolean replace(K key, V oldValue, V newValue);



    /**
     * Replaces each entry's value with the result of invoking the given function on that entry,
     * in the order entries are returned by an entry set iterator, until all entries have been
     * processed or the function throws an exception.
     *
     * @param function the function to apply to each entry
     * @throws UnsupportedOperationException if the {@code set} operation
     *         is not supported by this map's entry set iterator
     *         or the specified replacement value is null, and this map does not permit
     *         null values (optional restriction)
     * @throws ClassCastException if the class of a replacement value
     *         prevents it from being stored in this map
     * @throws IllegalArgumentException if some property of a replacement value
     *         prevents it from being stored in this map (optional restriction)
     */
    void replaceAll(@Nonnull BiFunction<? super K, ? super V, ? extends V> function);


    




    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified
     * value.
     *
     * @param key key with which the specified value is associated
     * @param value value expected to be associated with the specified key
     * @return {@code true} if the value was removed
     * @throws ClassCastException if the class of the specified key or value
     *         prevents it from being stored in this map (optional restriction)
     * @throws NullPointerException if the specified key or value is null,
     *         and this map does not permit null key or values (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     */
    boolean remove(Object key, Object value);

    /**
     * Removes all of the entries of this collection that satisfy the given predicate.
     * Errors or runtime exceptions thrown during iteration or by the predicate are relayed
     * to the caller.
     *
     * @param filter a predicate which returns {@code true} for elements to be removed
     * @return {@code true} if any elements were removed
     * @throws NullPointerException if the specified filter is null
     * @throws UnsupportedOperationException if elements cannot be removed from this collection.
     *         Implementations may throw this exception if a matching element cannot be removed
     *         or if, in general, removal is not supported.
     * @see <a href="package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean removeIf(@Nonnull BiPredicate<? super K, ? super V> filter);
}

