/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.collect.map;

import net.openhft.collect.Container;
import net.openhft.collect.Equivalence;
import net.openhft.function.BiConsumer;
import net.openhft.function.BiFunction;
import net.openhft.function.IntObjConsumer;
import net.openhft.function.IntObjPredicate;
import net.openhft.function.IntObjFunction;
import net.openhft.function.IntFunction;
import net.openhft.function.Function;
import net.openhft.function.BiFunction;
import net.openhft.collect.ObjCollection;
import net.openhft.collect.set.IntSet;
import net.openhft.collect.set.ObjSet;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import java.util.Map;


/**
 * 
 * A {@link Map} specialization with {@code int} keys and {@code Object} values.
 * 
 *
 * @see IntObjMapFactory
 */
public interface IntObjMap<V> extends Map<Integer, V>, Container {


    /**
     * Returns the equivalence strategy for values in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of value objects,
     * for example, {@link #containsValue(Object)} and {@link #remove(int, Object)},
     * are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for values in this map,
     *         or {@code null} if it is the default {@link Object} equality
     * @see IntObjMapFactory#withValueEquivalence
     */
    @Nullable
    Equivalence<V> valueEquivalence();


    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #containsKey(int)} instead
     */
    @Override
    @Deprecated
    boolean containsKey(Object key);

    /**
     * Returns {@code true} if this map contains a mapping for the specified key.
     *
     * @param key the {@code int} key whose presence in this map is to be tested
     * @return {@code true} if this map contains a mapping for the specified key
     */
    boolean containsKey(int key);


    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #get(int)} instead
     */
    @Nullable
    @Override
    @Deprecated
    V get(Object key);

    

    /**
     * Returns the value to which the specified key is mapped, or {@code null} if this map
     * contains no mapping for the key.
     *
     * 
     *
     * 
     * <p>If this map permits null values, then a return value of {@code null} does not
     * <i>necessarily</i> indicate that the map contains no mapping for the key; it's also possible
     * that the map explicitly maps the key to {@code null}. The {@link #containsKey containsKey}
     * operation may be used to distinguish these two cases.
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped, or {@code null} if this map
     *         contains no mapping for the key
     */
    @Nullable
    V get(int key);


    /**
     * Returns the value to which the specified key is mapped, or
     * {@code defaultValue} if this map contains no mapping for the key.
     *
     * @param key the key whose associated value is to be returned
     * @param defaultValue the default mapping of the key
     * @return the value to which the specified key is mapped, or
     *         {@code defaultValue} if this map contains no mapping for the key

     * @deprecated Use specialization {@link #getOrDefault(int, Object)} instead
     */
    
    @Deprecated
    V getOrDefault(Object key, V defaultValue);

    /**
     * Returns the value to which the specified key is mapped, or {@code defaultValue} if this map
     * contains no mapping for the key.
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped, or
     *         {@code defaultValue} if this map contains no mapping for the key
     */
    V getOrDefault(int key, V defaultValue);



    /**
     * Performs the given {@code action} on each entry in this map until all entries
     * have been processed or the action throws an {@code Exception}.
     * Exceptions thrown by the action are relayed to the caller. The entries
     * will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which
     * differs from the entry set iterator.
     *
     * @param action The action to be performed for each entry
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    void forEach(@Nonnull IntObjConsumer<? super V> action);


    /**
     * Checks the given {@code predicate} on each entry in this map until all entries
     * have been processed or the predicate returns {@code false} for some entry,
     * or throws an {@code Exception}. Exceptions thrown by the predicate are relayed to the caller.
     *
     * <p>The entries will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which differs from
     * the entry set iterator.
     *
     * <p>If the map is empty, this method returns {@code true} immediately.
     *
     * @return {@code true} if the predicate returned {@code true} for all entries of the map,
     *         {@code false} if it returned {@code false} for the entry
     * @param predicate the predicate to be checked for each entry
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean forEachWhile(@Nonnull IntObjPredicate<? super V> predicate);

    /**
     * Returns a new cursor over the entries of this map. It's order is always correspond to the
     * entry set iterator order.
     *
     * @return a new cursor over the entries of this map
     * @see <a href="../package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    @Nonnull
    IntObjCursor<V> cursor();


    /** {@inheritDoc} */
    @Override
    @Nonnull
    IntSet keySet();

    /** {@inheritDoc} */
    @Override
    @Nonnull
    ObjCollection<V> values();

    /** {@inheritDoc} */
    @Override
    @Nonnull
    ObjSet<Entry<Integer, V>> entrySet();

    

    

    

    

    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #put(int, Object)} instead
     */
    @Override
    @Deprecated
    V put(Integer key, V value);

    /**
     * Associates the specified value with the specified key in this map (optional operation).
     * If the map previously contained a mapping for the key, the old value is replaced
     * by the specified value. (A map {@code m} is said to contain a mapping for a key {@code k}
     * if and only if {@link #containsKey(int) m.containsKey(k)} would return {@code true}.)
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with {@code key}, or {@code null} if there was
     *         no mapping for {@code key}. (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with {@code key}.)
     * @throws UnsupportedOperationException if the {@code put} operation
    *         is not supported by this map
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     */
    V put(int key, V value);

    

    

    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), associates
     * it with the given value and returns {@code null}, else returns the current value.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or
     *         {@code null} if there was no mapping for the key.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with the key,
     *         if the implementation supports null values.)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws ClassCastException if the key or value is of an inappropriate type for this map
     * @throws NullPointerException if the specified key is null
     * @throws IllegalArgumentException if some property of the specified key
     *         or value prevents it from being stored in this map
     * @deprecated Use specialization {@link #putIfAbsent(int, Object)} instead
     */
    
    @Nullable
    @Deprecated
    V putIfAbsent(Integer key, V value);

    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), associates
     * it with the given value and returns {@code null}, else returns the current value.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or {@code null}
     *         if there was no mapping for the key. (A {@code null} return
     *         can also indicate that the map previously associated {@code null}
     *         with the key, if the implementation supports such values.)
     * @throws UnsupportedOperationException if the {@code put} operation
    *         is not supported by this map
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     */
    @Nullable
    V putIfAbsent(int key, V value);


    

    

    

    


    /**
     * Attempts to compute a mapping for the specified key and its current mapped value
     * (or {@code null} if there is no current mapping).
     *
     * <p>If the function returns {@code null}, the mapping is removed (or
     * remains absent if initially absent).
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key
     * @throws ClassCastException if the class of the
     *         computed value
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V compute(int key, @Nonnull IntObjFunction<? super V, ? extends V> remappingFunction);



    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), attempts
     * to compute its value using the given mapping function and enters it into this map
     *  unless {@code null}. The most common usage is to construct
     * a new object serving as an initial mapped value or memoized result.
     *
     * <p>If the function returns {@code null} no mapping is recorded.
     *
     * <p>If the function itself throws an (unchecked) exception, the exception is rethrown,
     * and no mapping is recorded.
     *
     * @param key key with which the specified value is to be associated
     * @param mappingFunction the function to compute a value
     * @return the current (existing or computed) value associated with
     *         the specified key
     * @throws ClassCastException if the class of the
     *         computed value
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V computeIfAbsent(int key, @Nonnull IntFunction<? extends V> mappingFunction);



    /**
     * If the value for the specified key is present and non-null,
     * attempts to compute a new mapping given the key and its current mapped value.
     *
     * <p>If the function returns {@code null}, the mapping is removed.
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key,
     *         or "no entry" value
     * @throws ClassCastException if the class of the
     *         computed value
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V computeIfPresent(int key, @Nonnull IntObjFunction<? super V, ? extends V> remappingFunction);



    /**
     * If the specified key is not already associated with a value (or is mapped to {@code null}), associates
     * it with the given value, otherwise, replaces the value with the results of the given
     * remapping function.
     *
     * This method may be of use when combining multiple mapped values for a key.
     *
     * <p>If the function returns {@code null}, the mapping is removed.
     *
     * <p>If the remappingFunction itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param value the value to use if absent
     * @param remappingFunction the function to recompute a value if present
     * @return the new value associated with the specified key
     * @throws NullPointerException if the {@code remappingFunction} is null
     * @throws ClassCastException if the class of the
     *         computed value
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    V merge(int key, V value, @Nonnull BiFunction<? super V, ? super V, ? extends V> remappingFunction);



    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param key key with which the specified value is associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or {@code null} if there was
     *         no mapping for the key. (A {@code null} return can also indicate
     *         that the map previously associated {@code null} with the key,
     *         if the implementation supports null values.)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws NullPointerException if the specified key is null
     * @throws IllegalArgumentException if some property of the specified value
     *         prevents it from being stored in this map
     * @deprecated Use specialization {@link #replace(int, Object)} instead
     */
    
    @Nullable
    @Deprecated
    V replace(Integer key, V value);

    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param key key with which the specified value is associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key,
     *         or {@code null} if there was no mapping for the key.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with the key,
     *         if the implementation supports such values.)
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    @Nullable
    V replace(int key, V value);


    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param key key with which the specified value is associated
     * @param oldValue value expected to be associated with the specified key
     * @param newValue value to be associated with the specified key
     * @return {@code true} if the value was replaced
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws NullPointerException if the specified key is null
     * @throws IllegalArgumentException if some property of the specified value
     *         prevents it from being stored in this map
     * @deprecated Use specialization
     *             {@link #replace(int, Object, Object)} instead
     */
    
    @Deprecated
    boolean replace(Integer key, V oldValue, V newValue);

    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param key key with which the specified value is associated
     * @param oldValue value expected to be associated with the specified key
     * @param newValue value to be associated with the specified key
     * @return {@code true} if the value was replaced
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    boolean replace(int key, V oldValue, V newValue);



    /**
     * Replaces each entry's value with the result of invoking the given function on that entry,
     * in the order entries are returned by an entry set iterator, until all entries have been
     * processed or the function throws an exception.
     *
     * @param function the function to apply to each entry
     * @throws UnsupportedOperationException if the {@code set} operation
     *         is not supported by this map's entry set iterator
     *         or the specified replacement value is null, and this map does not permit
     *         null values (optional restriction)
     * @throws ClassCastException if the class of a replacement value
     *         prevents it from being stored in this map
     * @throws IllegalArgumentException if some property of a replacement value
     *         prevents it from being stored in this map (optional restriction)
     */
    void replaceAll(@Nonnull IntObjFunction<? super V, ? extends V> function);


    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #remove(int)} instead
     */
    @Override
    @Nullable
    @Deprecated
    V remove(Object key);

    /**
     * Removes the mapping for a key from this map if it is present (optional operation).
     *
     *
     * <p>Returns the value to which this map previously associated the key, or {@code null}
     * if the map contained no mapping for the key.
     *
     * <p>A return value of {@code null} does not <i>necessarily</i> indicate that the map
     * contained no mapping for the key; it's also possible that the map
     * explicitly mapped the key to {@code null}.
     *
     * <p>The map will not contain a mapping for the specified key once the
     * call returns.
     *
     * @param key key whose mapping is to be removed from the map
     * @return the previous value associated with {@code key}, or {@code null} if there was
     *         no mapping for {@code key}
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map (optional restriction)
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     */
    @Nullable
    V remove(int key);


    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified
     * value.
     *
     * @param key key with which the specified value is associated
     * @param value value expected to be associated with the specified key
     * @return {@code true} if the value was removed
     * @throws NullPointerException if the specified key is null
               , or if the specified value is null,
     *         and this map does not permit null values (optional restriction)
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     * @deprecated Use specialization {@link #remove(int, Object)} instead
     */
    
    @Deprecated
    boolean remove(Object key, Object value);

    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified
     * value.
     *
     * @param key key with which the specified value is associated
     * @param value value expected to be associated with the specified key
     * @return {@code true} if the value was removed
     * @throws ClassCastException if the class of the specified value
     *         prevents it from being stored in this map (optional restriction)
     * @throws NullPointerException if the specified value is null,
     *         and this map does not permit null values (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     */
    boolean remove(int key, Object value);

    /**
     * Removes all of the entries of this collection that satisfy the given predicate.
     * Errors or runtime exceptions thrown during iteration or by the predicate are relayed
     * to the caller.
     *
     * @param filter a predicate which returns {@code true} for elements to be removed
     * @return {@code true} if any elements were removed
     * @throws NullPointerException if the specified filter is null
     * @throws UnsupportedOperationException if elements cannot be removed from this collection.
     *         Implementations may throw this exception if a matching element cannot be removed
     *         or if, in general, removal is not supported.
     * @see <a href="package-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean removeIf(@Nonnull IntObjPredicate<? super V> filter);
}

