/*
 * Decompiled with CFR 0.152.
 */
package net.openhft.chronicle.queue.internal.reader;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.openhft.chronicle.bytes.MethodReader;
import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.OS;
import net.openhft.chronicle.core.io.BackgroundResourceReleaser;
import net.openhft.chronicle.core.io.IOTools;
import net.openhft.chronicle.core.time.SetTimeProvider;
import net.openhft.chronicle.core.time.TimeProvider;
import net.openhft.chronicle.queue.DirectoryUtils;
import net.openhft.chronicle.queue.ExcerptAppender;
import net.openhft.chronicle.queue.ExcerptTailer;
import net.openhft.chronicle.queue.QueueTestCommon;
import net.openhft.chronicle.queue.RollCycle;
import net.openhft.chronicle.queue.impl.single.SingleChronicleQueue;
import net.openhft.chronicle.queue.impl.single.SingleChronicleQueueBuilder;
import net.openhft.chronicle.queue.internal.reader.Say;
import net.openhft.chronicle.queue.internal.reader.SayWhen;
import net.openhft.chronicle.queue.internal.reader.TimestampComparator;
import net.openhft.chronicle.queue.reader.ChronicleReader;
import net.openhft.chronicle.queue.reader.ContentBasedLimiter;
import net.openhft.chronicle.queue.reader.Reader;
import net.openhft.chronicle.queue.rollcycles.LegacyRollCycles;
import net.openhft.chronicle.queue.rollcycles.TestRollCycles;
import net.openhft.chronicle.testframework.GcControls;
import net.openhft.chronicle.testframework.process.JavaProcessBuilder;
import net.openhft.chronicle.threads.NamedThreadFactory;
import net.openhft.chronicle.wire.DocumentContext;
import net.openhft.chronicle.wire.MicroTimestampLongConverter;
import net.openhft.chronicle.wire.ServicesTimestampLongConverter;
import net.openhft.chronicle.wire.VanillaMethodWriterBuilder;
import net.openhft.chronicle.wire.WireType;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public class ChronicleReaderTest
extends QueueTestCommon {
    private static final byte[] ONE_KILOBYTE = new byte[1024];
    private static final long TOTAL_EXCERPTS_IN_QUEUE = 24L;
    private final Queue<String> capturedOutput = new ConcurrentLinkedQueue<String>();
    private Path dataDir;
    private long lastIndex = Long.MIN_VALUE;
    private long firstIndex = Long.MAX_VALUE;

    private static long getCurrentQueueFileLength(Path dataDir) throws IOException {
        try (RandomAccessFile file = new RandomAccessFile(Files.list(dataDir).filter(p -> p.toString().endsWith("cq4")).findFirst().orElseThrow(AssertionError::new).toFile(), "r");){
            long l = file.length();
            return l;
        }
    }

    @Before
    public void before() {
        if (!(!OS.isWindows() || this.testName.getMethodName().equals("shouldThrowExceptionIfInputDirectoryDoesNotExist") || this.testName.getMethodName().equals("shouldBeAbleToReadFromReadOnlyFile") || this.testName.getMethodName().equals("shouldPrintTimestampsToLocalTime") || this.testName.getMethodName().equals("namedTailerRequiresReadWrite") || this.testName.getMethodName().equals("matchLimitThenNamedTailer"))) {
            this.expectException("Read-only mode is not supported on Windows");
        }
        this.dataDir = this.getTmpDir().toPath();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((Path)this.dataDir).sourceId(1).testBlockSize().build();){
            VanillaMethodWriterBuilder methodWriterBuilder = queue.methodWriterBuilder(Say.class);
            Say events = (Say)methodWriterBuilder.build();
            int i = 0;
            while ((long)i < 24L) {
                events.say(i % 2 == 0 ? "hello" : "goodbye");
                ++i;
            }
            this.lastIndex = queue.lastIndex();
            this.firstIndex = queue.firstIndex();
        }
        this.ignoreException("Overriding sourceId from existing metadata, was 0, overriding to 1");
    }

    @Test(timeout=10000L)
    public void shouldReadQueueInReverse() {
        this.addCountToEndOfQueue();
        new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add).inReverseOrder().suppressDisplayIndex().execute();
        List firstFourElements = this.capturedOutput.stream().limit(4L).collect(Collectors.toList());
        Assert.assertEquals(Arrays.asList("\"4\"\n", "\"3\"\n", "\"2\"\n", "\"1\"\n"), firstFourElements);
    }

    @Test
    public void reverseOrderShouldIgnoreOptionsThatDontMakeSense() {
        this.addCountToEndOfQueue();
        new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add).inReverseOrder().suppressDisplayIndex().tail().historyRecords(10L).execute();
        List firstFourElements = this.capturedOutput.stream().limit(4L).collect(Collectors.toList());
        Assert.assertEquals(Arrays.asList("\"4\"\n", "\"3\"\n", "\"2\"\n", "\"1\"\n"), firstFourElements);
    }

    @Test
    public void reverseOrderWorksWithStartPosition() {
        List<Long> indices = this.addCountToEndOfQueue();
        new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add).inReverseOrder().suppressDisplayIndex().withStartIndex(indices.get(1).longValue()).execute();
        List firstFourElements = this.capturedOutput.stream().limit(2L).collect(Collectors.toList());
        Assert.assertEquals(Arrays.asList("\"2\"\n", "\"1\"\n"), firstFourElements);
    }

    @Test(expected=IllegalArgumentException.class)
    public void reverseOrderThrowsWhenStartPositionIsAfterEndOfQueue() {
        new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add).inReverseOrder().suppressDisplayIndex().withStartIndex(this.lastIndex + 1L).execute();
    }

    @Test(expected=IllegalArgumentException.class)
    public void reverseOrderThrowsWhenStartPositionIsBeforeStartOfQueue() {
        new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add).inReverseOrder().suppressDisplayIndex().withStartIndex(this.firstIndex - 1L).execute();
    }

    private List<Long> addCountToEndOfQueue() {
        ArrayList<Long> indices = new ArrayList<Long>();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((Path)this.dataDir).sourceId(1).testBlockSize().build();
             ExcerptAppender appender = queue.createAppender();){
            for (int i = 1; i < 5; ++i) {
                appender.writeText((CharSequence)String.valueOf(i));
                indices.add(appender.lastIndexAppended());
            }
        }
        return indices;
    }

    @Test(timeout=10000L)
    public void shouldReadQueueWithNonDefaultRollCycle() {
        this.expectException("Overriding roll length from existing metadata");
        Path path = this.getTmpDir().toPath();
        path.toFile().mkdirs();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((Path)path).rollCycle((RollCycle)LegacyRollCycles.MINUTELY).testBlockSize().sourceId(1).build();){
            VanillaMethodWriterBuilder methodWriterBuilder = queue.methodWriterBuilder(Say.class);
            Say events = (Say)methodWriterBuilder.build();
            int i = 0;
            while ((long)i < 24L) {
                events.say(i % 2 == 0 ? "hello" : "goodbye");
                ++i;
            }
        }
        new ChronicleReader().withBasePath(path).withMessageSink(this.capturedOutput::add).execute();
        Assert.assertFalse((boolean)this.capturedOutput.isEmpty());
    }

    @Test(timeout=10000L)
    public void shouldReadQueueWithNonDefaultRollCycleWhenMetadataDeleted() throws IOException {
        if (!OS.isWindows()) {
            this.expectException("Failback to readonly tablestore");
        }
        Path path = this.getTmpDir().toPath();
        path.toFile().mkdirs();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((Path)path).rollCycle((RollCycle)LegacyRollCycles.MINUTELY).testBlockSize().sourceId(1).build();){
            VanillaMethodWriterBuilder methodWriterBuilder = queue.methodWriterBuilder(Say.class);
            Say events = (Say)methodWriterBuilder.build();
            int i = 0;
            while ((long)i < 24L) {
                events.say(i % 2 == 0 ? "hello" : "goodbye");
                ++i;
            }
        }
        Files.list(path).filter(f -> f.getFileName().toString().endsWith(".cq4t")).findFirst().ifPresent(p -> p.toFile().delete());
        GcControls.waitForGcCycle();
        new ChronicleReader().withBasePath(path).withMessageSink(this.capturedOutput::add).execute();
        Assert.assertFalse((boolean)this.capturedOutput.isEmpty());
    }

    @Test
    public void shouldNotFailOnEmptyQueue() {
        Path path = this.getTmpDir().toPath();
        path.toFile().mkdirs();
        if (!OS.isWindows()) {
            this.expectException("Failback to readonly tablestore");
        }
        new ChronicleReader().withBasePath(path).withMessageSink(this.capturedOutput::add).execute();
        Assert.assertTrue((boolean)this.capturedOutput.isEmpty());
    }

    @Test
    public void shouldNotFailWhenNoMetadata() throws IOException {
        if (!OS.isWindows()) {
            this.expectException("Failback to readonly tablestore");
        }
        Files.list(this.dataDir).filter(f -> f.getFileName().toString().endsWith(".cq4t")).findFirst().ifPresent(path -> path.toFile().delete());
        this.basicReader().execute();
        Assert.assertTrue((boolean)this.capturedOutput.stream().anyMatch(msg -> msg.contains("history:")));
    }

    @Test
    public void shouldIncludeMessageHistoryByDefault() {
        this.basicReader().execute();
        Assert.assertTrue((boolean)this.capturedOutput.stream().anyMatch(msg -> msg.contains("history:")));
    }

    @Test
    public void shouldApplyIncludeRegexToHistoryMessagesAndBusinessMessagesMethodReaderDummy() {
        this.basicReader().withInclusionRegex("goodbye").asMethodReader("").execute();
        Assert.assertFalse((boolean)this.capturedOutput.stream().anyMatch(msg -> msg.contains("history:")));
    }

    @Test
    public void shouldNotIncludeMessageHistoryByDefaultMethodReader() {
        this.basicReader().asMethodReader(Say.class.getName()).execute();
        Assert.assertFalse((boolean)this.capturedOutput.stream().anyMatch(msg -> msg.contains("history:")));
    }

    @Test
    public void shouldIncludeMessageHistoryMethodReaderShowHistory() {
        this.basicReader().asMethodReader(Say.class.getName()).showMessageHistory(true).execute();
        String first = this.capturedOutput.poll();
        Assert.assertTrue((boolean)first.startsWith("0x"));
        String second = this.capturedOutput.poll();
        Assert.assertTrue((String)second, (boolean)second.matches("VanillaMessageHistory.sources: .. timings: .[0-9]+. addSourceDetails=false}" + System.lineSeparator() + "say: hello\n...\n"));
    }

    @Test(timeout=5000L)
    public void readOnlyQueueTailerShouldObserveChangesAfterInitiallyObservedReadLimit() throws IOException, InterruptedException, TimeoutException, ExecutionException {
        DirectoryUtils.deleteDir(this.dataDir.toFile());
        this.dataDir.toFile().mkdirs();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((Path)this.dataDir).testBlockSize().build();){
            Say events = (Say)queue.methodWriterBuilder(Say.class).build();
            events.say("hello");
            long readerCapacity = ChronicleReaderTest.getCurrentQueueFileLength(this.dataDir);
            RecordCounter recordCounter = new RecordCounter();
            ChronicleReader chronicleReader = this.basicReader().withMessageSink((Consumer)recordCounter);
            ExecutorService executorService = Executors.newSingleThreadExecutor((ThreadFactory)new NamedThreadFactory("executor"));
            Future<?> submit = executorService.submit(() -> ((ChronicleReader)chronicleReader).execute());
            long expectedReadingDocumentCount = readerCapacity / (long)ONE_KILOBYTE.length + 1L;
            int i = 0;
            while ((long)i < expectedReadingDocumentCount) {
                events.say(new String(ONE_KILOBYTE));
                ++i;
            }
            recordCounter.latch.countDown();
            executorService.shutdown();
            executorService.awaitTermination(Jvm.isDebug() ? 50L : 5L, TimeUnit.SECONDS);
            submit.get(1L, TimeUnit.SECONDS);
            if (!OS.isWindows()) {
                Assert.assertEquals((long)expectedReadingDocumentCount, (long)(recordCounter.recordCount.get() - 1L));
            }
        }
    }

    @Test
    public void shouldBeAbleToReadFromReadOnlyFile() throws IOException {
        Assume.assumeFalse((String)"#460 read-only not supported on Windows", (boolean)OS.isWindows());
        Path queueFile = Files.list(this.dataDir).filter(f -> f.getFileName().toString().endsWith(".cq4")).findFirst().orElseThrow(() -> new AssertionError((Object)("Could not find queue file in directory " + this.dataDir)));
        Assert.assertTrue((boolean)queueFile.toFile().setWritable(false));
        this.basicReader().execute();
    }

    @Test
    public void shouldConvertEntriesToText() {
        this.basicReader().execute();
        Assert.assertEquals((long)48L, (long)this.capturedOutput.size());
        Assert.assertTrue((boolean)this.capturedOutput.stream().anyMatch(msg -> msg.contains("hello")));
    }

    @Test
    public void shouldFilterByInclusionRegex() {
        this.basicReader().withInclusionRegex(".*good.*").execute();
        Assert.assertEquals((long)24L, (long)this.capturedOutput.size());
        this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).forEach(msg -> MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.containsString((String)"goodbye")));
    }

    @Test
    public void shouldFilterByMultipleInclusionRegex() {
        this.basicReader().withInclusionRegex(".*bye$").withInclusionRegex(".*o.*").execute();
        Assert.assertEquals((long)24L, (long)this.capturedOutput.size());
        this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).forEach(msg -> MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.containsString((String)"goodbye")));
        this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).forEach(msg -> MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.not((Matcher)CoreMatchers.containsString((String)"hello"))));
    }

    @Test(expected=IllegalArgumentException.class)
    public void shouldThrowExceptionIfInputDirectoryDoesNotExist() {
        this.basicReader().withBasePath(Paths.get("/does/not/exist", new String[0])).execute();
    }

    @Test
    public void shouldFilterByExclusionRegex() {
        this.basicReader().withExclusionRegex(".*good.*").execute();
        Assert.assertEquals((long)24L, (long)this.capturedOutput.size());
        this.capturedOutput.forEach(msg -> MatcherAssert.assertThat((Object)msg, (Matcher)CoreMatchers.not((Matcher)CoreMatchers.containsString((String)"goodbye"))));
    }

    @Test
    public void shouldFilterByMultipleExclusionRegex() {
        this.basicReader().withExclusionRegex(".*bye$").withExclusionRegex(".*ell.*").execute();
        Assert.assertEquals((long)0L, (long)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count());
    }

    @Test
    public void shouldReturnNoMoreThanTheSpecifiedNumberOfMaxRecords() {
        this.basicReader().historyRecords(5L).execute();
        Assert.assertEquals((long)5L, (long)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count());
    }

    @Test
    public void shouldCombineIncludeFilterAndMaxRecords() {
        this.basicReader().historyRecords(5L).withInclusionRegex("hello").execute();
        Assert.assertEquals((long)2L, (long)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count());
    }

    @Test
    public void shouldForwardToSpecifiedIndex() {
        long knownIndex = Long.decode(this.findAnExistingIndex());
        this.basicReader().withStartIndex(knownIndex).execute();
        Assert.assertEquals((long)24L, (long)this.capturedOutput.size());
        Assert.assertTrue((boolean)this.capturedOutput.poll().contains(Long.toHexString(knownIndex)));
    }

    @Test(expected=IllegalArgumentException.class)
    public void shouldFailIfSpecifiedIndexIsBeforeFirstIndex() {
        this.basicReader().withStartIndex(1L).execute();
    }

    @Test
    public void shouldNotRewindPastStartOfQueueWhenDisplayingHistory() {
        this.basicReader().historyRecords(Long.MAX_VALUE).execute();
        MatcherAssert.assertThat((Object)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count(), (Matcher)CoreMatchers.is((Object)24L));
    }

    @Test
    public void shouldContinueToPollQueueWhenTailModeIsEnabled() {
        int expectedPollCountWhenDocumentIsEmpty = 3;
        FiniteDocumentPollMethod pollMethod = new FiniteDocumentPollMethod(3);
        try {
            this.basicReader().withDocumentPollMethod((Function)pollMethod).tail().execute();
        }
        catch (ArithmeticException arithmeticException) {
            // empty catch block
        }
        Assert.assertEquals((long)3L, (long)pollMethod.invocationCount);
    }

    @Test(timeout=10000L)
    public void shouldPrintTimestampsToLocalTime() throws IOException {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            VanillaMethodWriterBuilder methodWriterBuilder = queue.methodWriterBuilder(SayWhen.class);
            SayWhen events = (SayWhen)methodWriterBuilder.build();
            long microTimestamp = System.currentTimeMillis() * 1000L;
            ArrayList<Long> timestamps = new ArrayList<Long>();
            for (int i = 0; i < 10; ++i) {
                events.sayWhen(microTimestamp, "Hello!");
                timestamps.add(microTimestamp);
                microTimestamp += (long)(1000 * i);
            }
            this.assertTimesAreInZone(queueDir, ZoneId.of("UTC"), timestamps);
            this.assertTimesAreInZone(queueDir, ZoneId.systemDefault(), timestamps);
        }
    }

    @Test
    public void shouldOnlyOutputUpToMatchLimitAfterFiltering() {
        this.basicReader().withInclusionRegex("goodbye").withMatchLimit(3L).execute();
        List matchedMessages = this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).collect(Collectors.toList());
        Assert.assertEquals((long)3L, (long)matchedMessages.size());
        Assert.assertTrue((boolean)matchedMessages.stream().allMatch(s -> s.contains("goodbye")));
    }

    @Test
    public void matchLimitThenNamedTailer() {
        long maxRecords = 5L;
        String tailerId = "myTailer";
        this.basicReader().withMatchLimit(5L).withReadOnly(false).withTailerId("myTailer").execute();
        Assert.assertEquals((long)5L, (long)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count());
        this.capturedOutput.clear();
        this.basicReader().withReadOnly(false).withTailerId("myTailer").execute();
        Assert.assertEquals((long)19L, (long)this.capturedOutput.stream().filter(msg -> !msg.startsWith("0x")).count());
    }

    @Test(expected=IllegalArgumentException.class)
    public void namedTailerRequiresReadWrite() {
        Assume.assumeFalse((boolean)OS.isWindows());
        this.basicReader().withTailerId("tailerId").withReadOnly(true).execute();
    }

    @Test
    public void shouldStopReadingWhenContentBasedLimitHasBeenReached() {
        final AtomicInteger helloCount = new AtomicInteger();
        AtomicInteger goodbyeCount = new AtomicInteger();
        final Say say = msg -> {
            if ("hello".equals(msg)) {
                helloCount.incrementAndGet();
            }
            if ("goodbye".equals(msg)) {
                goodbyeCount.incrementAndGet();
            }
        };
        ContentBasedLimiter cbl = new ContentBasedLimiter(){
            private int limit = -1;

            public boolean shouldHaltReading(DocumentContext dc) {
                dc.wire().bytes().readSkip(-4L);
                MethodReader methodReader = dc.wire().methodReader(new Object[]{say});
                methodReader.readOne();
                return helloCount.get() > this.limit;
            }

            public void configure(Reader reader) {
                this.limit = Integer.parseInt(reader.limiterArg());
            }
        };
        this.basicReader().withContentBasedLimiter(cbl).withLimiterArg("4").execute();
        Assert.assertEquals((long)4L, (long)this.capturedOutput.stream().filter(msg -> msg.contains("hello")).count());
    }

    private void assertTimesAreInZone(File queueDir, ZoneId zoneId, List<Long> timestamps) throws IOException {
        Process readerProcess = JavaProcessBuilder.create(ChronicleReaderRunner.class).withProgramArguments(new String[]{queueDir.toString()}).withJvmArguments(new String[]{"-DtimestampLongConverters.zoneId=" + zoneId.toString()}).start();
        while (readerProcess.isAlive()) {
            Jvm.pause((long)10L);
        }
        String output = new String(IOTools.readAsBytes((InputStream)readerProcess.getInputStream()));
        MicroTimestampLongConverter mtlc = new MicroTimestampLongConverter(zoneId.toString());
        for (Long timestamp : timestamps) {
            String expectedTimestamp = mtlc.asString(timestamp.longValue());
            int timestampIndex = output.indexOf(expectedTimestamp);
            Assert.assertTrue((String)String.format("%s contains %s", output, expectedTimestamp), (timestampIndex > 0 ? 1 : 0) != 0);
            output = output.substring(timestampIndex + expectedTimestamp.length());
        }
    }

    @Test
    public void findByBinarySearch() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            for (int i = 0; i < max; ++i) {
                this.capturedOutput.clear();
                long tsToLookFor = this.getTimestampAtIndex(i);
                System.out.println("Looking for " + tsToLookFor);
                ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
                reader.execute();
                Assert.assertEquals((long)(reps * (max - i)), (long)(this.capturedOutput.size() / 2));
            }
        }
    }

    @Test
    public void findByBinarySearchReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            for (int i = 0; i < max; ++i) {
                this.capturedOutput.clear();
                long tsToLookFor = this.getTimestampAtIndex(i);
                System.out.println("Looking for " + tsToLookFor);
                ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
                reader.execute();
                Assert.assertEquals((long)(reps * (i + 1)), (long)(this.capturedOutput.size() / 2));
            }
        }
    }

    @Test
    public void findByBinarySearchSparseRepeated() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            try (ExcerptAppender appender = queue.createAppender();){
                this.writeTimestamp(appender, this.getTimestampAtIndex(1));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                appender.writeText((CharSequence)"aaaa");
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(3));
            }
            this.capturedOutput.clear();
            long tsToLookFor = this.getTimestampAtIndex(2);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)7L, (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchSparseRepeatedReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            try (ExcerptAppender appender = queue.createAppender();){
                this.writeTimestamp(appender, this.getTimestampAtIndex(1));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                appender.writeText((CharSequence)"aaaa");
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(3));
            }
            this.capturedOutput.clear();
            long tsToLookFor = this.getTimestampAtIndex(2);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)7L, (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchSparseApprox() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            try (ExcerptAppender appender = queue.createAppender();){
                this.writeTimestamp(appender, this.getTimestampAtIndex(1));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                appender.writeText((CharSequence)"aaaa");
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
            }
            this.capturedOutput.clear();
            long tsToLookFor = this.getTimestampAtIndex(3);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)3L, (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchSparseApproxReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            try (ExcerptAppender appender = queue.createAppender();){
                this.writeTimestamp(appender, this.getTimestampAtIndex(1));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                this.writeTimestamp(appender, this.getTimestampAtIndex(2));
                appender.writeText((CharSequence)"aaaa");
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
                this.writeTimestamp(appender, this.getTimestampAtIndex(4));
            }
            this.capturedOutput.clear();
            long tsToLookFor = this.getTimestampAtIndex(3);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)3L, (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchApprox() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int reps = 5;
            int max = 10;
            this.populateQueueWithTimestamps(queue, 10, 5);
            for (int i = 0; i < 10; ++i) {
                this.capturedOutput.clear();
                long tsToLookFor = this.getTimestampAtIndex(i) - 1L;
                System.out.println("Looking for " + tsToLookFor);
                ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
                reader.execute();
                Assert.assertEquals((long)(5 * (10 - i)), (long)(this.capturedOutput.size() / 2));
            }
        }
    }

    @Test
    public void findByBinarySearchApproxReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int reps = 5;
            int max = 10;
            this.populateQueueWithTimestamps(queue, 10, 5);
            for (int i = 0; i < 10; ++i) {
                this.capturedOutput.clear();
                long tsToLookFor = this.getTimestampAtIndex(i) + 1L;
                System.out.println("Looking for " + tsToLookFor);
                ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
                reader.execute();
                Assert.assertEquals((long)(5 * (i + 1)), (long)(this.capturedOutput.size() / 2));
            }
        }
    }

    @Test
    public void findByBinarySearchAfterEnd() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            long tsToLookFor = this.getTimestampAtIndex(11);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)0L, (long)this.capturedOutput.size());
        }
    }

    @Test
    public void findByBinarySearchAfterEndReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            long tsToLookFor = this.getTimestampAtIndex(11);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)(max * reps), (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchBeforeStart() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            long tsToLookFor = this.getTimestampAtIndex(-1);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)(max * reps), (long)(this.capturedOutput.size() / 2));
        }
    }

    @Test
    public void findByBinarySearchBeforeStartReverse() {
        File queueDir = this.getTmpDir();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).build();){
            int max = 10;
            int reps = 5;
            this.populateQueueWithTimestamps(queue, max, reps);
            long tsToLookFor = this.getTimestampAtIndex(-1);
            ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).inReverseOrder().withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
            reader.execute();
            Assert.assertEquals((long)0L, (long)this.capturedOutput.size());
        }
    }

    @Test
    public void findByBinarySearchWithDeletedRollCyles() {
        File queueDir = this.getTmpDir();
        SetTimeProvider timeProvider = new SetTimeProvider();
        try (SingleChronicleQueue queue = SingleChronicleQueueBuilder.binary((File)queueDir).timeProvider((TimeProvider)timeProvider).rollCycle((RollCycle)TestRollCycles.TEST_SECONDLY).build();){
            for (int i = 0; i < 5; ++i) {
                int entries = 10;
                int reps = 5;
                this.populateQueueWithTimestamps(queue, entries, reps, i);
                timeProvider.advanceMillis(3000L);
            }
        }
        BackgroundResourceReleaser.releasePendingResources();
        Assert.assertTrue((String)"Couldn't delete cycle, test is broken", (boolean)queueDir.toPath().resolve("19700101-000009T.cq4").toFile().delete());
        long tsToLookFor = this.getTimestampAtIndex(22);
        System.out.println(tsToLookFor);
        ChronicleReader reader = new ChronicleReader().withArg(ServicesTimestampLongConverter.INSTANCE.asString(tsToLookFor)).withBinarySearch(TimestampComparator.class.getCanonicalName()).withBasePath(queueDir.toPath()).withMessageSink(this.capturedOutput::add);
        reader.execute();
        Assert.assertEquals((long)180L, (long)this.capturedOutput.size());
    }

    @Test
    public void shouldRespectWireType() {
        this.basicReader().asMethodReader(Say.class.getName()).withWireType(WireType.JSON).execute();
        this.capturedOutput.poll();
        Assert.assertEquals((Object)"{\"say\":\"hello\"}", (Object)this.capturedOutput.poll().trim());
    }

    @Test
    public void shouldRespectWireType2() {
        this.basicReader().asMethodReader(Say.class.getName()).withWireType(WireType.JSON_ONLY).execute();
        this.capturedOutput.poll();
        Assert.assertEquals((Object)"{\"say\":\"hello\"}", (Object)this.capturedOutput.poll().trim());
    }

    private void populateQueueWithTimestamps(SingleChronicleQueue queue, int entries, int repeatsPerEntry) {
        this.populateQueueWithTimestamps(queue, entries, repeatsPerEntry, 0);
    }

    private void populateQueueWithTimestamps(SingleChronicleQueue queue, int entries, int repeatsPerEntry, int batch) {
        try (ExcerptAppender appender = queue.createAppender();){
            for (int i = 0; i < entries; ++i) {
                int effectiveIndex = i + entries * batch;
                for (int j = 0; j < repeatsPerEntry; ++j) {
                    long timestampAtIndex = this.getTimestampAtIndex(effectiveIndex);
                    this.writeTimestamp(appender, timestampAtIndex);
                    System.out.printf("%s:%s -- %s%n", effectiveIndex * repeatsPerEntry + j, Long.toHexString(appender.lastIndexAppended()), timestampAtIndex);
                }
            }
        }
    }

    private void writeTimestamp(ExcerptAppender appender, long timestamp) {
        try (DocumentContext dc = appender.writingDocument();){
            dc.wire().write((CharSequence)"timestamp").int64(timestamp);
        }
    }

    private long getTimestampAtIndex(int index) {
        TimeUnit timeUnit = ServicesTimestampLongConverter.timeUnit();
        long start = timeUnit.convert(1610000000000L, TimeUnit.MILLISECONDS);
        return start + (long)index * timeUnit.convert(1L, TimeUnit.SECONDS);
    }

    private String findAnExistingIndex() {
        this.basicReader().execute();
        List indicies = this.capturedOutput.stream().filter(s -> s.startsWith("0x")).collect(Collectors.toList());
        this.capturedOutput.clear();
        return ((String)indicies.get(indicies.size() / 2)).trim().replaceAll(":", "");
    }

    private ChronicleReader basicReader() {
        return new ChronicleReader().withBasePath(this.dataDir).withMessageSink(this.capturedOutput::add);
    }

    @After
    public void clearInterrupt() {
        Thread.interrupted();
    }

    static {
        Arrays.fill(ONE_KILOBYTE, (byte)7);
    }

    private static final class FiniteDocumentPollMethod
    implements Function<ExcerptTailer, DocumentContext> {
        private final int maxPollsReturningEmptyDocument;
        private int invocationCount;

        private FiniteDocumentPollMethod(int maxPollsReturningEmptyDocument) {
            this.maxPollsReturningEmptyDocument = maxPollsReturningEmptyDocument;
        }

        @Override
        public DocumentContext apply(ExcerptTailer excerptTailer) {
            DocumentContext documentContext = excerptTailer.readingDocument();
            if (!documentContext.isPresent()) {
                ++this.invocationCount;
                if (this.invocationCount >= this.maxPollsReturningEmptyDocument) {
                    throw new ArithmeticException("For testing purposes");
                }
            }
            return documentContext;
        }
    }

    private static final class RecordCounter
    implements Consumer<String> {
        private final AtomicLong recordCount = new AtomicLong();
        private final CountDownLatch latch = new CountDownLatch(1);

        private RecordCounter() {
        }

        @Override
        public void accept(String msg) {
            try {
                this.latch.await();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!msg.startsWith("0x")) {
                this.recordCount.incrementAndGet();
            }
        }
    }

    private static class ChronicleReaderRunner {
        private ChronicleReaderRunner() {
        }

        public static void main(String[] args) {
            ChronicleReader reader = new ChronicleReader().asMethodReader(SayWhen.class.getName()).withBasePath(Paths.get(args[0], new String[0])).withMessageSink(System.out::println);
            reader.execute();
        }
    }
}

