/*
 * Decompiled with CFR 0.152.
 */
package net.openhft.chronicle.queue.impl.single;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.io.Closeable;
import net.openhft.chronicle.core.io.IOTools;
import net.openhft.chronicle.core.onoes.LogLevel;
import net.openhft.chronicle.queue.QueueTestCommon;
import net.openhft.chronicle.queue.TableStoreWriteLockLockerProcess;
import net.openhft.chronicle.queue.common.ProcessRunner;
import net.openhft.chronicle.queue.impl.TableStore;
import net.openhft.chronicle.queue.impl.single.TableStoreWriteLock;
import net.openhft.chronicle.queue.impl.table.Metadata;
import net.openhft.chronicle.queue.impl.table.SingleTableBuilder;
import net.openhft.chronicle.threads.Pauser;
import net.openhft.chronicle.threads.Threads;
import org.jetbrains.annotations.NotNull;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TableStoreWriteLockTest
extends QueueTestCommon {
    private static final String TEST_LOCK_NAME = "testLock";
    private static final long TIMEOUT_MS = 100L;
    private TableStore<Metadata.NoMeta> tableStore;
    private Path storeDirectory;

    @Before
    public void setUp() {
        Path tempDir = IOTools.createTempDirectory((String)"namedTableStoreLockTest");
        tempDir.toFile().mkdirs();
        this.storeDirectory = tempDir.resolve("test_store.cq4t");
        this.tableStore = SingleTableBuilder.binary((Path)this.storeDirectory, (Metadata)Metadata.NoMeta.INSTANCE).build();
    }

    @Override
    @After
    public void tearDown() {
        Closeable.closeQuietly(this.tableStore);
    }

    @Test(timeout=5000L)
    public void lockWillThrowIllegalStateExceptionIfInterruptedWhileWaitingForLock() throws InterruptedException {
        try (TableStoreWriteLock testLock = TableStoreWriteLockTest.createTestLock(this.tableStore, 5000L);){
            testLock.lock();
            AtomicBoolean threwException = new AtomicBoolean(false);
            Thread t = new Thread(() -> {
                try {
                    testLock.lock();
                }
                catch (IllegalStateException e) {
                    threwException.set(true);
                }
            });
            t.start();
            Jvm.pause((long)10L);
            t.interrupt();
            t.join();
            Assert.assertTrue((boolean)threwException.get());
        }
    }

    @Test
    public void testIsLockedByCurrentProcess() {
        AtomicLong actualPid = new AtomicLong(-1L);
        try (TableStoreWriteLock testLock = this.createTestLock();){
            testLock.lock();
            Assert.assertTrue((boolean)testLock.isLockedByCurrentProcess(actualPid::set));
            Assert.assertEquals((long)-1L, (long)actualPid.get());
            testLock.unlock();
            Assert.assertFalse((boolean)testLock.isLockedByCurrentProcess(actualPid::set));
            Assert.assertEquals((long)Long.MIN_VALUE, (long)actualPid.get());
        }
    }

    @Test(timeout=5000L)
    public void lockWillBeAcquiredAfterTimeoutWithAWarning() throws InterruptedException {
        try (TableStoreWriteLock testLock = TableStoreWriteLockTest.createTestLock(this.tableStore, 50L);){
            Thread t = new Thread(() -> ((TableStoreWriteLock)testLock).lock());
            t.start();
            t.join();
            testLock.lock();
            Assert.assertTrue((boolean)this.exceptions.keySet().stream().anyMatch(ek -> ek.level == LogLevel.WARN && ek.clazz == TableStoreWriteLock.class && ek.message.startsWith("Forced unlock")));
            this.expectException("Unlocking forcibly");
            this.expectException("Forced unlock");
        }
    }

    @Test
    public void unlockWillWarnIfNotLocked() {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            testLock.unlock();
            Assert.assertTrue((boolean)this.exceptions.keySet().stream().anyMatch(ek -> ek.level == LogLevel.WARN && ek.clazz == TableStoreWriteLock.class && ek.message.startsWith("Write lock was already unlocked.")));
            this.expectException("Write lock was already unlocked.");
        }
    }

    @Test(timeout=5000L)
    public void unlockWillNotUnlockAndWarnIfLockedByAnotherProcess() throws IOException, InterruptedException {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            Process process = ProcessRunner.runClass(TableStoreWriteLockLockerProcess.class, this.storeDirectory.toAbsolutePath().toString(), TEST_LOCK_NAME);
            while (!testLock.locked()) {
                Jvm.pause((long)10L);
            }
            testLock.unlock();
            Assert.assertTrue((boolean)testLock.locked());
            Assert.assertTrue((boolean)this.exceptions.keySet().stream().anyMatch(ek -> ek.level == LogLevel.WARN && ek.clazz == TableStoreWriteLock.class && ek.message.startsWith("Write lock was locked by someone else!")));
            this.expectException("Write lock was locked by someone else!");
            process.destroy();
            process.waitFor();
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockWillUnlockAndWarnIfLockedByAnotherProcess() throws IOException, InterruptedException {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            Process process = ProcessRunner.runClass(TableStoreWriteLockLockerProcess.class, this.storeDirectory.toAbsolutePath().toString(), TEST_LOCK_NAME);
            while (!testLock.locked()) {
                Jvm.pause((long)10L);
            }
            testLock.forceUnlock();
            Assert.assertFalse((boolean)testLock.locked());
            Assert.assertTrue((boolean)this.exceptions.keySet().stream().anyMatch(ek -> ek.level == LogLevel.WARN && ek.clazz == TableStoreWriteLock.class && ek.message.startsWith("Forced unlock for the lock")));
            this.expectException("Forced unlock for the lock");
            process.destroy();
            process.waitFor();
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockWillNotWarnIfLockIsNotLocked() {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            testLock.forceUnlock();
            Assert.assertFalse((boolean)testLock.locked());
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockWillWarnIfLockIsLockedByCurrentProcess() {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            testLock.lock();
            testLock.forceUnlock();
            Assert.assertFalse((boolean)testLock.locked());
            Assert.assertTrue((boolean)this.exceptions.keySet().stream().anyMatch(ek -> ek.level == LogLevel.WARN && ek.clazz == TableStoreWriteLock.class && ek.message.startsWith("Forced unlock for the lock")));
            this.expectException("Forced unlock for the lock");
        }
    }

    @Test(timeout=5000L)
    public void forceUnlockQuietlyWillUnlockWithNoWarningIfLockedByAnotherProcess() throws IOException, InterruptedException {
        try (TableStoreWriteLock testLock = this.createTestLock();){
            Process process = ProcessRunner.runClass(TableStoreWriteLockLockerProcess.class, this.storeDirectory.toAbsolutePath().toString(), TEST_LOCK_NAME);
            while (!testLock.locked()) {
                Jvm.pause((long)10L);
            }
            testLock.forceUnlockQuietly();
            Assert.assertFalse((boolean)testLock.locked());
            process.destroy();
            process.waitFor();
        }
    }

    @Test
    public void lockPreventsConcurrentAcquisition() {
        AtomicBoolean lockIsAcquired = new AtomicBoolean(false);
        try (TableStoreWriteLock testLock = TableStoreWriteLockTest.createTestLock(this.tableStore, 10000L);){
            int numThreads = Math.min(6, Runtime.getRuntime().availableProcessors());
            ExecutorService executorService = Executors.newFixedThreadPool(numThreads);
            CyclicBarrier barrier = new CyclicBarrier(numThreads);
            Collection futures = IntStream.range(0, numThreads).mapToObj(v -> executorService.submit(new LockAcquirer(testLock, lockIsAcquired, 30, barrier))).collect(Collectors.toList());
            futures.forEach(fut -> {
                try {
                    fut.get();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            Threads.shutdown((ExecutorService)executorService);
        }
    }

    private TableStoreWriteLock createTestLock() {
        return TableStoreWriteLockTest.createTestLock(this.tableStore, 100L);
    }

    @NotNull
    private static TableStoreWriteLock createTestLock(TableStore<Metadata.NoMeta> tableStore, long timeoutMilliseconds) {
        return new TableStoreWriteLock(tableStore, Pauser::balanced, Long.valueOf(timeoutMilliseconds), TEST_LOCK_NAME);
    }

    static class LockAcquirer
    implements Runnable {
        private final TableStoreWriteLock tableStoreWriteLock;
        private final AtomicBoolean lockIsAcquired;
        private final int numberOfIterations;
        private final CyclicBarrier barrier;

        LockAcquirer(TableStoreWriteLock tableStoreWriteLock, AtomicBoolean lockIsAcquired, int numberOfIterations, CyclicBarrier barrier) {
            this.tableStoreWriteLock = tableStoreWriteLock;
            this.lockIsAcquired = lockIsAcquired;
            this.numberOfIterations = numberOfIterations;
            this.barrier = barrier;
        }

        @Override
        public void run() {
            try {
                this.barrier.await();
                for (int i = 0; i < this.numberOfIterations; ++i) {
                    this.tableStoreWriteLock.lock();
                    try {
                        this.lockIsAcquired.compareAndSet(false, true);
                        Jvm.pause((long)10L);
                        this.lockIsAcquired.compareAndSet(true, false);
                        continue;
                    }
                    finally {
                        this.tableStoreWriteLock.unlock();
                        Jvm.pause((long)1L);
                    }
                }
            }
            catch (Exception e) {
                throw new AssertionError((Object)e);
            }
        }
    }
}

