/*
 * Decompiled with CFR 0.152.
 */
package net.openhft.chronicle.bytes;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.lang.reflect.Proxy;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.spi.AbstractInterruptibleChannel;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import net.openhft.chronicle.bytes.Bytes;
import net.openhft.chronicle.bytes.BytesInternal;
import net.openhft.chronicle.bytes.MappedBytesStore;
import net.openhft.chronicle.bytes.MappedBytesStoreFactory;
import net.openhft.chronicle.bytes.NewChunkListener;
import net.openhft.chronicle.bytes.ReadOnlyMappedBytesStore;
import net.openhft.chronicle.bytes.VanillaBytes;
import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.OS;
import net.openhft.chronicle.core.ReferenceCounted;
import net.openhft.chronicle.core.ReferenceCounter;
import net.openhft.chronicle.core.io.Closeable;
import net.openhft.chronicle.core.io.IORuntimeException;
import org.jetbrains.annotations.NotNull;

public class MappedFile
implements ReferenceCounted {
    private static final long DEFAULT_CAPACITY = 0x800000000000L;
    private static final Object GLOBAL_FILE_LOCK = FileChannel.class;
    @NotNull
    private final RandomAccessFile raf;
    private final FileChannel fileChannel;
    private final long chunkSize;
    private final long overlapSize;
    private final List<WeakReference<MappedBytesStore>> stores = new ArrayList<WeakReference<MappedBytesStore>>();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final ReferenceCounter refCount = ReferenceCounter.onReleased(this::performRelease);
    private final long capacity;
    @NotNull
    private final File file;
    private final boolean readOnly;
    private NewChunkListener newChunkListener = MappedFile::logNewChunk;

    protected MappedFile(@NotNull File file, @NotNull RandomAccessFile raf, long chunkSize, long overlapSize, long capacity, boolean readOnly) {
        this.file = file;
        this.raf = raf;
        this.fileChannel = raf.getChannel();
        this.chunkSize = OS.mapAlign(chunkSize);
        this.overlapSize = overlapSize > 0L && overlapSize < 65536L ? chunkSize : OS.mapAlign(overlapSize);
        this.capacity = capacity;
        this.readOnly = readOnly;
        if (Jvm.isJava9Plus()) {
            this.doNotCloseOnInterrupt9(this.fileChannel);
        } else {
            this.doNotCloseOnInterrupt(this.fileChannel);
        }
        assert (MappedFile.registerMappedFile(this));
    }

    private static void logNewChunk(String filename, int chunk, long delayMicros) {
        if (!Jvm.isDebugEnabled(MappedFile.class)) {
            return;
        }
        String message = BytesInternal.acquireStringBuilder().append("Allocation of ").append(chunk).append(" chunk in ").append(filename).append(" took ").append((double)delayMicros / 1000.0).append(" ms.").toString();
        Jvm.debug().on(MappedFile.class, message);
    }

    private static boolean registerMappedFile(MappedFile mappedFile) {
        return true;
    }

    public static void checkMappedFiles() {
    }

    @NotNull
    public static MappedFile of(@NotNull File file, long chunkSize, long overlapSize, boolean readOnly) throws FileNotFoundException {
        RandomAccessFile raf = new RandomAccessFile(file, readOnly ? "r" : "rw");
        long capacity = 0x800000000000L;
        return new MappedFile(file, raf, chunkSize, overlapSize, 0x800000000000L, readOnly);
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull File file, long chunkSize) throws FileNotFoundException {
        return MappedFile.mappedFile(file, chunkSize, (long)OS.pageSize());
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull String filename, long chunkSize) throws FileNotFoundException {
        return MappedFile.mappedFile(filename, chunkSize, (long)OS.pageSize());
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull String filename, long chunkSize, long overlapSize) throws FileNotFoundException {
        return MappedFile.mappedFile(new File(filename), chunkSize, overlapSize);
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull File file, long chunkSize, long overlapSize) throws FileNotFoundException {
        return MappedFile.mappedFile(file, chunkSize, overlapSize, false);
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull File file, long chunkSize, long overlapSize, boolean readOnly) throws FileNotFoundException {
        return MappedFile.of(file, chunkSize, overlapSize, readOnly);
    }

    @NotNull
    public static MappedFile readOnly(@NotNull File file) throws FileNotFoundException {
        long chunkSize = file.length();
        long overlapSize = 0L;
        if (OS.isWindows() && chunkSize > 0x80000000L) {
            chunkSize = 0x80000000L;
            overlapSize = OS.pageSize();
        }
        return MappedFile.of(file, chunkSize, overlapSize, true);
    }

    @NotNull
    public static MappedFile mappedFile(@NotNull File file, long capacity, long chunkSize, long overlapSize, boolean readOnly) throws IOException {
        RandomAccessFile raf = new RandomAccessFile(file, readOnly ? "r" : "rw");
        if (raf.length() < capacity) {
            raf.setLength(capacity);
        }
        return new MappedFile(file, raf, chunkSize, overlapSize, capacity, readOnly);
    }

    public static void warmup() {
        try {
            Jvm.disableDebugHandler();
            File file = File.createTempFile("delete", "me");
            file.deleteOnExit();
            long mapAlignment = OS.mapAlignment();
            int chunks = 64;
            int compileThreshold = Jvm.compileThreshold();
            for (int j = 0; j <= compileThreshold; j += 64) {
                try {
                    try (RandomAccessFile raf = new RandomAccessFile(file, "rw");){
                        MappedFile mappedFile = new MappedFile(file, raf, mapAlignment, 0L, mapAlignment * 64L, false);
                        MappedFile.warmup0(mapAlignment, 64, mappedFile);
                        mappedFile.release();
                    }
                    Thread.yield();
                    Files.delete(file.toPath());
                    continue;
                }
                catch (IOException e) {
                    Jvm.debug().on(MappedFile.class, "Error during warmup", e);
                }
            }
        }
        catch (IOException e) {
            Jvm.warn().on(MappedFile.class, "Error during warmup", e);
        }
        Jvm.resetExceptionHandlers();
    }

    private static void warmup0(long mapAlignment, int chunks, @NotNull MappedFile mappedFile) throws IOException {
        for (int i = 0; i < chunks; ++i) {
            mappedFile.acquireBytesForRead((long)i * mapAlignment).release();
            mappedFile.acquireBytesForWrite((long)i * mapAlignment).release();
        }
    }

    private void doNotCloseOnInterrupt(FileChannel fc) {
        try {
            Field field = AbstractInterruptibleChannel.class.getDeclaredField("interruptor");
            Jvm.setAccessible(field);
            field.set(fc, thread -> System.err.println(this.getClass().getName() + " - " + fc + " not closed on interrupt"));
        }
        catch (Throwable e) {
            Jvm.warn().on(this.getClass(), "Couldn't disable close on interrupt", e);
        }
    }

    private void doNotCloseOnInterrupt9(FileChannel fc) {
        try {
            Field field = AbstractInterruptibleChannel.class.getDeclaredField("interruptor");
            Class<?> interruptibleClass = field.getType();
            Jvm.setAccessible(field);
            field.set(fc, Proxy.newProxyInstance(interruptibleClass.getClassLoader(), new Class[]{interruptibleClass}, (p, m, a) -> {
                System.err.println(this.getClass().getName() + " - " + fc + " not closed on interrupt");
                return null;
            }));
        }
        catch (Throwable e) {
            Jvm.warn().on(this.getClass(), "Couldn't disable close on interrupt", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public MappedFile withSizes(long chunkSize, long overlapSize) {
        chunkSize = OS.mapAlign(chunkSize);
        overlapSize = OS.mapAlign(overlapSize);
        if (chunkSize == this.chunkSize && overlapSize == this.overlapSize) {
            return this;
        }
        try {
            MappedFile mappedFile = new MappedFile(this.file, this.raf, chunkSize, overlapSize, this.capacity, this.readOnly);
            return mappedFile;
        }
        finally {
            this.release();
        }
    }

    @NotNull
    public File file() {
        return this.file;
    }

    @NotNull
    public MappedBytesStore acquireByteStore(long position) throws IOException, IllegalArgumentException, IllegalStateException {
        return this.acquireByteStore(position, this.readOnly ? ReadOnlyMappedBytesStore::new : MappedBytesStore::new);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @NotNull
    public <T extends MappedBytesStore> T acquireByteStore(long position, @NotNull MappedBytesStoreFactory<T> mappedBytesStoreFactory) throws IOException, IllegalArgumentException, IllegalStateException {
        if (this.closed.get()) {
            throw new IOException("Closed");
        }
        if (position < 0L) {
            throw new IOException("Attempt to access a negative position: " + position);
        }
        int chunk = (int)(position / this.chunkSize);
        List<WeakReference<MappedBytesStore>> list = this.stores;
        synchronized (list) {
            MappedBytesStore mbs;
            while (this.stores.size() <= chunk) {
                this.stores.add(null);
            }
            WeakReference<MappedBytesStore> mbsRef = this.stores.get(chunk);
            if (mbsRef != null && (mbs = (MappedBytesStore)mbsRef.get()) != null && mbs.tryReserve()) {
                return (T)mbs;
            }
            long start = System.nanoTime();
            long minSize = ((long)chunk + 1L) * this.chunkSize + this.overlapSize;
            long size = this.fileChannel.size();
            if (size < minSize && !this.readOnly) {
                try {
                    Object object = GLOBAL_FILE_LOCK;
                    synchronized (object) {
                        size = this.fileChannel.size();
                        if (size < minSize) {
                            long time0 = System.nanoTime();
                            try (FileLock ignore = this.fileChannel.lock();){
                                size = this.fileChannel.size();
                                if (size < minSize) {
                                    this.raf.setLength(minSize);
                                }
                            }
                            long time1 = System.nanoTime() - time0;
                            if (time1 >= 1000000L) {
                                Jvm.warn().on(this.getClass(), "Took " + time1 / 1000L + " us to grow file " + this.file());
                            }
                        }
                    }
                }
                catch (IOException ioe) {
                    throw new IOException("Failed to resize to " + minSize, ioe);
                }
            }
            long mappedSize = this.chunkSize + this.overlapSize;
            FileChannel.MapMode mode = this.readOnly ? FileChannel.MapMode.READ_ONLY : FileChannel.MapMode.READ_WRITE;
            long startOfMap = (long)chunk * this.chunkSize;
            long address = OS.map(this.fileChannel, mode, startOfMap, mappedSize);
            T mbs2 = mappedBytesStoreFactory.create(this, (long)chunk * this.chunkSize, address, mappedSize, this.chunkSize);
            this.stores.set(chunk, new WeakReference<T>(mbs2));
            long time2 = System.nanoTime() - start;
            if (this.newChunkListener != null) {
                this.newChunkListener.onNewChunk(this.file.getPath(), chunk, time2 / 1000L);
            }
            if (time2 > 5000000L) {
                Jvm.warn().on(this.getClass(), "Took " + time2 / 1000L + " us to add mapping for " + this.file());
            }
            return mbs2;
        }
    }

    @NotNull
    public Bytes acquireBytesForRead(long position) throws IOException, IllegalStateException, IllegalArgumentException {
        MappedBytesStore mbs = this.acquireByteStore(position);
        Bytes bytes = mbs.bytesForRead();
        bytes.readPositionUnlimited(position);
        mbs.release();
        return bytes;
    }

    public void acquireBytesForRead(long position, @NotNull VanillaBytes bytes) throws IOException, IllegalStateException, IllegalArgumentException {
        MappedBytesStore mbs = this.acquireByteStore(position);
        bytes.bytesStore(mbs, position, mbs.capacity() - position);
    }

    @NotNull
    public Bytes acquireBytesForWrite(long position) throws IOException, IllegalStateException, IllegalArgumentException {
        MappedBytesStore mbs = this.acquireByteStore(position);
        Bytes bytes = mbs.bytesForWrite();
        bytes.writePosition(position);
        mbs.release();
        return bytes;
    }

    public void acquireBytesForWrite(long position, @NotNull VanillaBytes bytes) throws IOException, IllegalStateException, IllegalArgumentException {
        MappedBytesStore mbs = this.acquireByteStore(position);
        bytes.bytesStore(mbs, position, mbs.capacity() - position);
        bytes.writePosition(position);
    }

    @Override
    public void reserve() throws IllegalStateException {
        this.refCount.reserve();
    }

    @Override
    public void release() throws IllegalStateException {
        this.refCount.release();
    }

    @Override
    public long refCount() {
        return this.refCount.get();
    }

    @Override
    public boolean tryReserve() {
        return this.refCount.tryReserve();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void performRelease() {
        try {
            for (int i = 0; i < this.stores.size(); ++i) {
                WeakReference<MappedBytesStore> storeRef = this.stores.get(i);
                if (storeRef == null) continue;
                MappedBytesStore mbs = (MappedBytesStore)storeRef.get();
                if (mbs != null) {
                    while (mbs.refCount() != 0L) {
                        try {
                            mbs.release();
                        }
                        catch (IllegalStateException e) {
                            Jvm.debug().on(this.getClass(), e);
                        }
                    }
                }
                this.stores.set(i, null);
            }
        }
        finally {
            Closeable.closeQuietly((Object)this.raf.getChannel());
            Closeable.closeQuietly((Object)this.raf);
            Closeable.closeQuietly((Object)this.fileChannel);
            this.closed.set(true);
        }
    }

    @NotNull
    public String referenceCounts() {
        StringBuilder sb = new StringBuilder();
        sb.append("refCount: ").append(this.refCount());
        for (WeakReference<MappedBytesStore> store : this.stores) {
            MappedBytesStore mbs;
            long count = 0L;
            if (store != null && (mbs = (MappedBytesStore)store.get()) != null) {
                count = mbs.refCount();
            }
            sb.append(", ").append(count);
        }
        return sb.toString();
    }

    public long capacity() {
        return this.capacity;
    }

    public long chunkSize() {
        return this.chunkSize;
    }

    public long overlapSize() {
        return this.overlapSize;
    }

    public NewChunkListener getNewChunkListener() {
        return this.newChunkListener;
    }

    public void setNewChunkListener(NewChunkListener listener) {
        this.newChunkListener = listener;
    }

    public long actualSize() throws IORuntimeException {
        boolean interrupted = Thread.interrupted();
        try {
            long l = this.fileChannel.size();
            return l;
        }
        catch (ArrayIndexOutOfBoundsException aiooe) {
            long l = this.actualSize();
            return l;
        }
        catch (ClosedByInterruptException cbie) {
            this.closed.set(true);
            interrupted = true;
            throw new IllegalStateException(cbie);
        }
        catch (IOException e) {
            boolean open = this.fileChannel.isOpen();
            if (open) {
                throw new IORuntimeException(e);
            }
            this.closed.set(true);
            throw new IllegalStateException(e);
        }
        finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public boolean isClosed() {
        return this.closed.get();
    }

    @NotNull
    public RandomAccessFile raf() {
        return this.raf;
    }
}

