/*
 * OfficeFloor - http://www.officefloor.net
 * Copyright (C) 2005-2012 Daniel Sagenschneider
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.officefloor.plugin.servlet;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.RequestDispatcher;
import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.officefloor.compile.spi.section.SectionDesigner;
import net.officefloor.compile.spi.section.SectionInput;
import net.officefloor.compile.spi.section.SectionObject;
import net.officefloor.compile.spi.section.SectionTask;
import net.officefloor.compile.spi.section.SectionWork;
import net.officefloor.compile.spi.section.TaskObject;
import net.officefloor.compile.spi.section.source.SectionSource;
import net.officefloor.compile.spi.section.source.SectionSourceContext;
import net.officefloor.compile.spi.section.source.impl.AbstractSectionSource;
import net.officefloor.compile.spi.work.source.TaskTypeBuilder;
import net.officefloor.compile.spi.work.source.WorkSource;
import net.officefloor.compile.spi.work.source.WorkSourceContext;
import net.officefloor.compile.spi.work.source.WorkTypeBuilder;
import net.officefloor.compile.spi.work.source.impl.AbstractWorkSource;
import net.officefloor.frame.api.build.None;
import net.officefloor.frame.api.execute.Task;
import net.officefloor.frame.api.execute.TaskContext;
import net.officefloor.frame.api.manage.OfficeFloor;
import net.officefloor.frame.util.AbstractSingleTask;
import net.officefloor.plugin.servlet.bridge.ServletBridge;

/**
 * {@link SectionSource} for servicing by a {@link Servlet} container resource.
 * 
 * @author Daniel Sagenschneider
 */
public class ServletContainerResourceSectionSource extends
		AbstractSectionSource {

	/**
	 * {@link HttpServletRequest} attribute name to indicate a {@link Servlet}
	 * container resource.
	 */
	private static final String SERVLET_CONTAINER_RESOURCE_ATTRIBUTE_NAME = "servlet.container.resource.attribute.name";

	/**
	 * Dependency keys for the {@link ServletContainerResourceWorkSource}.
	 */
	public static enum DependencyKeys {
		SERVLET_BRIDGE
	}

	/**
	 * Completes the servicing after {@link OfficeFloor} functionality for the
	 * {@link OfficeFloorServletFilter}.
	 * 
	 * @param request
	 *            {@link HttpServletRequest}.
	 * @param response
	 *            {@link HttpServletResponse}.
	 * @return <code>true</code> if serviced.
	 * @throws ServletException
	 *             As per {@link Servlet} API.
	 * @throws IOException
	 *             As per {@link Servlet} API.
	 */
	public static boolean completeServletService(HttpServletRequest request,
			HttpServletResponse response) throws ServletException, IOException {

		// Obtain the request attribute
		ServletContainerResourceTask task;
		synchronized (request) {
			task = (ServletContainerResourceTask) request
					.getAttribute(SERVLET_CONTAINER_RESOURCE_ATTRIBUTE_NAME);
		}

		// Determine if requires handling
		if (task == null) {
			return true; // serviced
		}

		// Determine if require dispatch
		if (task.requestDispatcherPath != null) {

			// Dispatch request to Servlet container resource
			RequestDispatcher dispatcher = request
					.getRequestDispatcher(task.requestDispatcherPath);
			dispatcher.forward(request, response);

			// Serviced
			return true;
		}

		// Not serviced
		return false;
	}

	/*
	 * ==================== SectionSource ========================
	 */

	@Override
	protected void loadSpecification(SpecificationContext context) {
		// No required specification
	}

	@Override
	public void sourceSection(SectionDesigner designer,
			SectionSourceContext context) throws Exception {

		// Create the Servlet Bridge dependency
		SectionObject servletBridge = designer.addSectionObject(
				"SERVLET_BRIDGE", ServletBridge.class.getName());

		// Create the non-handled task
		String nonHandledInputName = context.getSectionLocation();
		this.addServletResource(nonHandledInputName, null, servletBridge,
				designer);

		// Create the Servlet container resource tasks
		Set<String> registeredResources = new HashSet<String>();
		for (String inputName : context.getPropertyNames()) {

			// Obtain the request dispatcher path
			String requestDispatcherPath = context.getProperty(inputName);

			// Ensure only register the resource once
			if (registeredResources.contains(requestDispatcherPath)) {
				continue;
			}

			// Add Servlet Resource
			this.addServletResource(inputName, requestDispatcherPath,
					servletBridge, designer);

			// Resource registered
			registeredResources.add(requestDispatcherPath);
		}
	}

	/**
	 * Adds handling for a {@link Servlet} resource.
	 * 
	 * @param inputName
	 *            {@link SectionInput} name.
	 * @param requestDispatcherPath
	 *            {@link RequestDispatcher} path.
	 * @param servletBridge
	 *            {@link ServletBridge} {@link SectionObject}.
	 * @param designer
	 *            {@link SectionDesigner}.
	 */
	private void addServletResource(String inputName,
			String requestDispatcherPath, SectionObject servletBridge,
			SectionDesigner designer) {

		// Create the task to indicate Servlet container resource
		SectionWork work = designer.addSectionWork(inputName,
				ServletContainerResourceWorkSource.class.getName());
		work.addProperty(
				ServletContainerResourceWorkSource.PROPERTY_SERVLET_CONTAINER_RESOURCE,
				requestDispatcherPath);
		SectionTask task = work.addSectionTask(inputName, "RESOURCE");
		TaskObject dependency = task
				.getTaskObject(DependencyKeys.SERVLET_BRIDGE.name());
		designer.link(dependency, servletBridge);

		// Link input for task
		SectionInput input = designer.addSectionInput(inputName, null);
		designer.link(input, task);
	}

	/**
	 * {@link Task} to link to {@link Servlet} container resource.
	 */
	public static class ServletContainerResourceTask
			extends
			AbstractSingleTask<ServletContainerResourceTask, DependencyKeys, None> {

		/**
		 * {@link RequestDispatcher} path.
		 */
		public final String requestDispatcherPath;

		/**
		 * Initiate.
		 * 
		 * @param requestDispatcherPath
		 *            {@link RequestDispatcher} path.
		 */
		public ServletContainerResourceTask(String requestDispatcherPath) {
			this.requestDispatcherPath = requestDispatcherPath;
		}

		/*
		 * ===================== Task ============================
		 */

		@Override
		public Object doTask(
				TaskContext<ServletContainerResourceTask, DependencyKeys, None> context) {

			// Obtain the Servlet bridge
			ServletBridge bridge = (ServletBridge) context
					.getObject(DependencyKeys.SERVLET_BRIDGE);

			// Indicate to use Servlet container resource
			HttpServletRequest request = bridge.getRequest();
			synchronized (request) {
				request.setAttribute(SERVLET_CONTAINER_RESOURCE_ATTRIBUTE_NAME,
						this);
			}

			// Nothing further
			return null;
		}
	}

	/**
	 * {@link WorkSource} to link to {@link Servlet} container resource.
	 */
	public static class ServletContainerResourceWorkSource extends
			AbstractWorkSource<ServletContainerResourceTask> {

		/**
		 * Name of the property for the {@link Servlet} container resource.
		 */
		public static final String PROPERTY_SERVLET_CONTAINER_RESOURCE = "servlet.container.resource";

		/*
		 * ====================== WorkSource ===========================
		 */

		@Override
		protected void loadSpecification(SpecificationContext context) {
			context.addProperty(PROPERTY_SERVLET_CONTAINER_RESOURCE);
		}

		@Override
		public void sourceWork(
				WorkTypeBuilder<ServletContainerResourceTask> workTypeBuilder,
				WorkSourceContext context) throws Exception {

			// Obtain the request dispatcher path
			String requestDispatcherPath = context.getProperty(
					PROPERTY_SERVLET_CONTAINER_RESOURCE, null);

			// Add the task
			ServletContainerResourceTask factory = new ServletContainerResourceTask(
					requestDispatcherPath);
			workTypeBuilder.setWorkFactory(factory);
			TaskTypeBuilder<DependencyKeys, None> task = workTypeBuilder
					.addTaskType("RESOURCE", factory, DependencyKeys.class,
							None.class);
			task.addObject(ServletBridge.class).setKey(
					DependencyKeys.SERVLET_BRIDGE);
		}
	}

}