package net.odoframework.sql.util.schema;

import lombok.Getter;
import net.odoframework.sql.util.Key;
import net.odoframework.util.ListBackedSet;
import net.odoframework.util.Strings;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

public class TableBuilder<T> {

    @Getter
    private String name;
    @Getter
    private Set<Relation> relationships;
    @Getter
    private Map<String, Column> columns;

    private Set<Column<T, ?, ?>> primaryKey;
    @Getter
    private Function<Key, T> constructor;
    @Getter
    private Class<T> type;


    public TableBuilder(String name, Class<T> type) {
        this.name = Strings.requireNotBlank(name, "name is required");
        this.type = Objects.requireNonNull(type, "type is required");
    }

    public static <K> TableBuilder<K> table(String name, Class<K> type) {
        var mapping = new TableBuilder<K>(name, type);
        return mapping;
    }

    public TableBuilder<T> constructor(Function<Key, T> constructor) {
        this.constructor = constructor;
        return this;
    }

    public <K> TableBuilder<T> primaryKey(String name, Function<T, K> getter, BiConsumer<T, K> setter) {
        this.getPrimaryKey().add(new Column<>(name, setter, getter));
        return this;
    }

    public <K> TableBuilder<T> primaryKey(String name, Function<T, K> getter) {
        return primaryKey(name, getter, null);
    }


    @SafeVarargs
    public final TableBuilder<T> primaryKey(Column<T, ?, ?>... columns) {
        final var columnSet = Arrays.stream(columns).collect(Collectors.toSet());
        getPrimaryKey().addAll(columnSet);
        columnSet.forEach(this::column);
        return this;
    }


    public Set<Column<T, ?, ?>> getPrimaryKey() {
        if (this.primaryKey == null) {
            this.primaryKey = new ListBackedSet<>(1);
        }
        return this.primaryKey;
    }

    public <K, Z> TableBuilder<T> column(Column<T, K, Z> column) {
        initColumns();
        columns.put(column.getName(), column);
        return this;
    }

    private void initColumns() {
        if (columns == null) {
            columns = new LinkedHashMap<>();
        }
    }

    public <K> TableBuilder<T> column(String columnName, Function<T, K> getter, BiConsumer<T, K> setter) {
        initColumns();
        columns.put(columnName, new Column<>(columnName, setter, getter));
        return this;
    }

    public <K,Z> ColumnBuilder<T,K,Z> column() {
        initColumns();
        return new ColumnBuilder<>(this);
    }

    public <K, Z> TableBuilder<T> column(String columnName, Function<T, K> getter, BiConsumer<T, K> setter, Function<Z, K> dbToObjectConverter, Function<K, Z> objectToDbConverter) {
        initColumns();
        columns.put(columnName, new Column<>(columnName, setter, getter, dbToObjectConverter, objectToDbConverter));
        return this;
    }

    public <K> TableBuilder<T> column(String columnName, Function<T, K> getter) {
        initColumns();
        columns.put(columnName, new Column<>(columnName, null, getter));
        return this;
    }


    public final <K> OneToManyBuilder<T, K> oneToMany(Class<K> target) {
        return new OneToManyBuilder<T, K>(this).owner(getName()).target(target);
    }

    public final TableBuilder<T> addRelation(Relation<T, ?> relation) {
        if (relationships == null) {
            relationships = new LinkedHashSet<>();
        }
        relationships.add(Objects.requireNonNull(relation, "relation cannot be null"));
        return this;
    }

    public final <K> ManyToOneBuilder<T, K> manyToOne(Class<K> target) {
        return new ManyToOneBuilder<T, K>(this).owner(getName()).target(target);
    }

    public final <K> OneToOneBuilder<T, K> oneToOne(Class<K> target) {
        return (OneToOneBuilder<T, K>) new OneToOneBuilder<T, K>(this).owner(getName()).target(target);
    }


    public Table<T> build() {
        final var table = new Table<T>(name, this.type, this.constructor, this.primaryKey);
        columns.forEach((name, column) -> table.column(column));
        if (relationships != null) {
            relationships.forEach(table::addRelationship);
        }
        return table;
    }


}
