package net.odoframework.sql.util.schema;

import lombok.Getter;
import net.odoframework.sql.SQLUtils;
import net.odoframework.sql.util.Key;
import net.odoframework.sql.util.MappingContext;
import net.odoframework.util.ListBackedMap;
import net.odoframework.util.Pair;
import net.odoframework.util.Strings;

import java.sql.ResultSet;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static net.odoframework.util.Pair.cons;
import static net.odoframework.util.Strings.requireNotBlank;

public class Table<T> {

    @Getter
    private final String name;
    @Getter
    private final PrimaryKey<T> primarykey;
    @Getter
    private final Function<Key, T> constructor;
    @Getter
    private final Class<T> type;
    @Getter
    private Schema schema;
    private Set<Relation<T, ?>> relationships;
    private Map<String, Column> columns;

    public Table(String name, Class<T> type, Function<Key, T> constructor, Set<Column<T, ?, ?>> primaryKey) {
        this(name, type, constructor, new PrimaryKey<>(primaryKey));
    }

    public Table(String name, Class<T> type, Function<Key, T> constructor, PrimaryKey<T> primaryKey) {
        this.name = requireNotBlank(name, "name is required");
        this.type = Objects.requireNonNull(type);
        this.primarykey = Objects.requireNonNull(primaryKey, "primary key cannot be null");
        if (primaryKey.size() == 0) {
            throw new IllegalArgumentException("primary key must contain at least one column");
        }
        this.constructor = Objects.requireNonNull(constructor, "constructor is required for type " + type.getName());
    }


    protected void setSchema(Schema schema) {
        this.schema = schema;
    }

    protected Map getColumns() {
        if (columns == null) {
            columns = new LinkedHashMap<>();
        }
        return columns;
    }

    @SuppressWarnings("unchecked")
    public <K, Z> Table<T> column(Column<T, K, Z> column) {
        getColumns().put(column.getName(), column);
        return this;
    }

    @SuppressWarnings("unchecked")
    public Set<Relation<T, ?>> getRelationships() {
        if (relationships == null) {
            relationships = new LinkedHashSet<>();
        }
        return relationships;
    }

    @SuppressWarnings("unchecked")
    public Table<T> addRelationship(Relation<T, ?> relation) {
        getRelationships().add(Objects.requireNonNull(relation));
        return this;
    }

    @SuppressWarnings("unchecked")
    public Pair<Key, T> mapInstance(MappingContext mappingContext, ResultSet rs) {
        final var columnIndex = mappingContext.getTableIndexMap(getName());
        if (columnIndex == null) {
            return null;
        }
        var pkColumns = primarykey.getPrimaryKeyColumns();
        var primaryKey = primarykey.createKey(columnIndex.extract(pkColumns), rs);
        final var result = (T) mappingContext
                .get(getName(), primaryKey)
                .orElseGet(() -> {
                    var instance = constructor.apply(primaryKey);
                    this.primarykey.mapInstance(instance, columnIndex, rs);
                    mappingContext.set(getName(), primaryKey, instance);
                    for (String column : columns.keySet()) {
                        var value = SQLUtils.getColumn(rs, columnIndex.get(column));
                        var colDef = this.columns.get(column);
                        if (colDef.isWritable()) {
                            colDef.setFromDB(instance, value);
                        }
                    }

                    for (Relation relationship : relationships) {
                        final var targetTable = schema.getByType(relationship.getTarget());
                        if (mappingContext.getTableIndexMap(targetTable.getName()) == null) {
                            continue;
                        }
                        relationship.map(rs, schema, mappingContext, primaryKey, instance);
                    }
                    return instance;
                });
        return cons(primaryKey, result);
    }

    @SuppressWarnings("unchecked")
    public Map<String, Object> toMap(T instance) {
        Map<String, Object> map = new ListBackedMap<>(primarykey.size() + columns.size());
        this.primarykey.toMap(instance, map);
        for (var columnEntry : columns.entrySet()) {
            final var column = columnEntry.getValue();
            var value = column.getForDB(instance);
            map.put(columnEntry.getKey(), value);
        }
        getRelationships().stream()
                .filter(it -> it instanceof ManyToOne)
                .forEach(it -> {
                    var relationship = (ManyToOne<T, ?>) it;
                    var relatedInstance = relationship.getGetter().apply(instance);
                    if (relatedInstance != null) {
                        var relatedTable = schema.getByType(relatedInstance.getClass());
                        relationship.getColumnBindings().forEach((key, value) -> {
                            Column column = relatedTable
                                    .getPrimarykey()
                                    .getColumn(value)
                                    .orElseThrow(() -> new IllegalStateException(value + " is not a column on " + relatedTable.name));
                            map.put(key, column.getForDB(relatedInstance));
                        });
                    }
                });
        return map;
    }

    public Column<T, ?, ?> getColumn(String columnName) {
        return (Column<T, ?, ?>) getColumns().get(columnName);
    }


    public String buildJoin(int fetchDepth) {
        var sql = new StringBuilder();
        var tableSet = new HashSet<String>();
        tableSet.add(getFullName());
        appendRelationships(sql, 1, fetchDepth, tableSet);

        var columns = tableSet
                .stream()
                .map(it -> String.join(".", it, "*"))
                .collect(Collectors.joining(", "));

        var selectClause = new StringBuilder("select ")
                .append(columns)
                .append(" from ")
                .append(getFullName());
        return selectClause.append('\n').append(sql).toString();

    }

    private void appendRelationships(StringBuilder sql, int currentDepth, int fetchDepth, Set<String> tables) {
        if (currentDepth >= fetchDepth) {
            return;
        }
        var oneToManys = getRelationships()
                .stream()
                .filter(it -> it instanceof OneToMany)
                .collect(Collectors.toList());
        for (Relation relationship : oneToManys) {
            sql.append('\n').append(relationship.leftJoin(schema));
        }
        for (Relation relationship : oneToManys) {
            final var targetTable = getSchema().getByType(relationship.getTarget());
            targetTable.appendRelationships(sql, currentDepth + 1, fetchDepth, tables);
            tables.add(targetTable.getFullName());
        }
    }


    public String getFullName() {
        if (Strings.isNotBlank(this.schema.getName())) {
            return String.join(".", this.schema.getName(), getName());
        }
        return getName();
    }

}
