package net.odoframework.sql.util.schema;

import lombok.Getter;
import lombok.NoArgsConstructor;
import net.odoframework.util.Strings;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

@NoArgsConstructor
public class Schema {

    @Getter
    private String name = "";
    private Map<String, Table> tables;

    public Schema(String name) {
        this.name = Strings.requireNotBlank(name, "name is a required parameter");
    }

    @SuppressWarnings("unchecked")
    protected Map<String, Table> getTables() {
        if (tables == null) {
            tables = new LinkedHashMap<>();
        }
        return tables;
    }

    public Schema add(Table<?> table) {
        table.setSchema(this);
        getTables().put(table.getName(), table);
        return this;
    }

    public Schema add(TableBuilder<?> table) {
        var result = table.build();
        result.setSchema(this);
        getTables().put(result.getName(), result);
        return this;
    }


    @SuppressWarnings("unchecked")
    public <T> Table<T> getTable(String name) {
        final var theTable = getTables().get(name);
        if (theTable == null) {
            throw new IllegalArgumentException(name + " does not exist in the schema");
        }
        return theTable;
    }

    public <T> Table<T> getByType(Class<T> type) {
        return getDefn(type).orElseThrow(() -> new IllegalArgumentException(type.getName() + " is not registered"));
    }

    @SuppressWarnings("unchecked")
    public <T> Optional<Table<T>> getDefn(Class<T> type) {
        return getTables()
                .entrySet()
                .stream()
                .filter(it -> Objects.equals(type, it.getValue().getType()))
                .findFirst()
                .map(Map.Entry::getValue);
    }
}
