package net.odoframework.sql.util.schema;

import lombok.Data;
import net.odoframework.sql.BaseDBStatement;
import net.odoframework.sql.ColumnIndex;
import net.odoframework.sql.SQLUtils;
import net.odoframework.sql.util.Key;

import java.sql.ResultSet;
import java.util.*;
import java.util.stream.Collectors;

@Data

public class PrimaryKey<T> {

    private Map<String, Column<T, ?, ?>> primaryKey;
    private String[] primaryKeyNamesCache = null;
    private String whereClauseCache = null;

    public PrimaryKey() {
        this.primaryKey = new LinkedHashMap<>();
    }

    public PrimaryKey(Set<Column<T, ?, ?>> primaryKey) {
        this();
        Objects.requireNonNull(primaryKey, "primary key cannot be null").forEach(this::add);
        if (this.primaryKey.isEmpty()) {
            throw new IllegalArgumentException("primary key must at least contain one column");
        }
    }

    public Optional<Column<T, ?, ?>> getColumn(String name) {
        return Optional.ofNullable(this.primaryKey.get(name));
    }

    public PrimaryKey<T> add(Column<T, ?, ?> column) {
        this.primaryKey.put(column.getName(), Objects.requireNonNull(column, "column argument cannot be null"));
        return this;
    }

    public String[] getPrimaryKeyColumns() {
        if (primaryKeyNamesCache == null) {
            primaryKeyNamesCache = this.primaryKey.keySet().toArray(new String[0]);
        }
        return primaryKeyNamesCache;
    }

    Key createKey(ColumnIndex columnIndex, ResultSet rs) {
        return Key.createKey(columnIndex.extract(getPrimaryKeyColumns()), rs);
    }

    void mapInstance(T instance, ColumnIndex columnIndex, ResultSet rs) {
        for (var pkColumn : this.primaryKey.values()) {
            if (pkColumn.isWritable()) {
                pkColumn.setFromDB(instance, SQLUtils.getColumn(rs, columnIndex.get(pkColumn.getName())));
            }
        }
    }

    public int size() {
        return this.primaryKey.size();
    }

    Map<String, Object> toMap(T instance, Map<String, Object> targetMap) {
        for (Column<T, ?, ?> pkColumn : this.primaryKey.values()) {
            var value = pkColumn.getForDB(instance);
            targetMap.put(pkColumn.getName(), value);
        }
        return targetMap;
    }

    public String whereClause() {
        if (whereClauseCache == null) {
            whereClauseCache = Arrays.stream(getPrimaryKeyColumns())
                    .map(it -> String.join(" = ", it, "?"))
                    .collect(Collectors.joining(", "));
        }
        return whereClauseCache;
    }

    public void populateStatement(T instance, BaseDBStatement statement) {
        this.primaryKey
                .values()
                .stream().map(it -> it.getForDB(instance))
                .forEach(statement::addBinding);
    }

}
