package net.odoframework.sql.util.schema;

import lombok.Builder;
import net.odoframework.sql.util.Key;
import net.odoframework.sql.util.MappingContext;

import java.sql.ResultSet;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Function;

public class OneToMany<T, K> extends BaseRelationship<T, K> {

    private BiConsumer<T, K> setter;
    private Function<T, ? extends Collection<K>> getter;


    @Builder
    public OneToMany(String name, String owner, Class<K> target, Function<T, ? extends Collection<K>> getter, BiConsumer<T, K> setter, Map<String, String> columnBindings) {
        super(name, owner, target, columnBindings);
        this.getter = Objects.requireNonNull(getter, "getter is a required parameter");
        this.setter = Objects.requireNonNull(setter, "setter is a required parameter");
    }

    @Override
    @SuppressWarnings("unchecked")
    public void map(ResultSet rs, Schema schema, MappingContext context, Key ownerKey, T result) {
        var targetTable = schema.getByType(getTarget());
        var tableIndex = context.getTableIndexMap(targetTable.getName()).extract(columnBindings.values().toArray(new String[]{}));
        var foreignKey = Key.createKey(tableIndex, rs);
        if (foreignKey.isNull()) {
            return;
        }
        var targetInstance = targetTable.mapInstance(context, rs);
        if (targetInstance == null) {
            return;
        }
        boolean isSame = true;
        for (var binding : columnBindings.entrySet()) {
            isSame &= Objects.equals(ownerKey.get(binding.getKey()), foreignKey.get(binding.getValue()));
        }
        if (isSame) {
            setter.accept(result, (K) targetInstance.getRight());
        } else {
            throw new IllegalStateException(ownerKey + " didn't match " + foreignKey);
        }
    }

    public BiConsumer<T, K> getSetter() {
        return setter;
    }

    public Function<T, ? extends Collection<K>> getGetter() {
        return getter;
    }
}
