package net.odoframework.sql.util;

import lombok.Getter;
import net.odoframework.sql.ColumnIndex;
import net.odoframework.sql.SQLSupplier;
import net.odoframework.sql.SQLWrappedException;
import net.odoframework.util.Pair;

import java.util.*;
import java.util.stream.Collectors;

import static net.odoframework.util.Pair.cons;

public class MappingContext {

    @Getter
    private final Map<String, Map<Key, MappedInstance<Object>>> context;
    @Getter
    private final Map<String, ColumnIndex> indexMap;


    public MappingContext(Map<String, ColumnIndex> indexMap) {
        this.indexMap = Objects.requireNonNull(indexMap, "index map");
        context = new LinkedHashMap<>();
    }


    public <T> T set(String table, Key key, T instance) {
        return set(table, key, () -> instance);
    }

    public <T> T set(String table, Key key, SQLSupplier<T> instance) {
        synchronized (context) {
            var tableInstances = context.computeIfAbsent(table, k -> new LinkedHashMap<>());
            return (T) tableInstances.computeIfAbsent(key, k -> new MappedInstance<>(SQLWrappedException.get(instance), key)).getInstance();
        }
    }

    public <T> Collection<T> get(final String table) {
        return context
                .computeIfAbsent(table, k -> new LinkedHashMap<>()).values()
                .stream().map(it -> (T) ((MappedInstance<?>) it).getInstance())
                .collect(Collectors.toSet());
    }

    public <T> Collection<Pair<Key, MappedInstance<T>>> getInstances(final String table) {
        return context
                .computeIfAbsent(table, k -> new LinkedHashMap<>()).entrySet()
                .stream().map(it -> cons(it.getKey(), (MappedInstance<T>) it.getValue()))
                .collect(Collectors.toSet());
    }

    public <T> Optional<T> get(final String table, final Key key) {
        var tableInstances = context.get(table);
        if (tableInstances != null) {
            return Optional.ofNullable((T) tableInstances.get(key)).map(it -> (T) ((MappedInstance<?>) it).getInstance());
        }
        return Optional.empty();
    }

    public <T> Optional<MappedInstance<T>> getInstance(final String table, final Key key) {
        var tableInstances = context.get(table);
        if (tableInstances != null) {
            return Optional.ofNullable((MappedInstance<T>) tableInstances.get(key));
        }
        return Optional.empty();
    }

    public MappingContext setReference(String table, String id, Key pk, Key foreignKey) {
        getInstance(table, pk).ifPresent(it -> {
            //it.reference(id, foreignKey.);
        });
        return this;
    }


    public ColumnIndex getTableIndexMap(String table) {
        return getIndexMap().get(table);
    }
}
