package net.odoframework.sql.util;

import lombok.Getter;
import lombok.Setter;
import net.odoframework.sql.DBStatement;
import net.odoframework.sql.SQLStatement;
import net.odoframework.sql.SQLTemplate;
import net.odoframework.sql.SQLUtils;
import net.odoframework.sql.util.schema.ManyToOne;
import net.odoframework.sql.util.schema.OneToMany;
import net.odoframework.sql.util.schema.Schema;
import net.odoframework.sql.util.schema.Table;
import net.odoframework.util.Assertions;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;

public abstract class DefaultRepository<T> implements Repository<T> {

    private final Table<T> defn;
    @Getter
    private Class<T> type;
    @Getter
    @Setter
    private Schema schema;
    @Getter
    @Setter
    private SQLTemplate sqlTemplate;

    public DefaultRepository(Class<T> type, Schema schema, SQLTemplate sqlTemplate) {
        this.type = Assertions.notNull(type, "type");
        this.schema = Assertions.notNull(schema, "schema");
        this.sqlTemplate = Assertions.notNull(sqlTemplate, "sqlTemplate");
        this.defn = schema
                .getDefn(type)
                .orElseThrow(() -> new IllegalStateException("no definition for " + this.type + " is registered in the schema"));
    }

    @Override
    public int create(T instance) {
        final var insert = new Insert<>(this.schema, requireNonNull(instance, "instance cannot be null"));
        return this.sqlTemplate.execute(insert);
    }


    @Override
    public int save(T instance) {
        final var update = new Update<>(this.schema, requireNonNull(instance, "instance cannot be null"));
        return this.sqlTemplate.execute(update);
    }

    @Override
    public Optional<T> get(Key key, int fetchDepth) {
        var instanceSQL = defn.buildJoin(fetchDepth);
        var pkClause = key.keySet()
                .stream()
                .map(o -> String.join(" = ", String.join(".", defn.getFullName(), o), "?"))
                .collect(Collectors.joining(" and "));
        var sqlStmt = String.join("\nwhere ", instanceSQL, pkClause);
        var stmt = SQLStatement.sql(sqlStmt);
        for (Object value : key.values()) {
            stmt.bind(value);
        }
        return find(stmt).findFirst();
    }


    @Override
    public Stream<T> findAll(int fetchDepth) {
        var instanceSQL = defn.buildJoin(fetchDepth);
        var stmt = SQLStatement.sql(instanceSQL);
        return find(stmt);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Stream<T> find(DBStatement statement) {
        final AtomicReference<MappingContext> ctx = new AtomicReference<>();
        return sqlTemplate.stream(statement, rs -> {
            if (ctx.get() == null) {
                ctx.set(new MappingContext(SQLUtils.getIndex(statement.getSql(), rs)));
            }
            return (T) defn.mapInstance(ctx.get(), rs).getRight();
        });
    }
}
