package net.odoframework.service.web;

import net.odoframework.container.util.Json;
import net.odoframework.service.Invocation;
import net.odoframework.util.Strings;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.BiFunction;

import static java.util.Objects.requireNonNull;

public abstract class HttpRouter implements WebFunction {


    private Json json;
    private Map<String, Map<String, BiFunction<WebRequest, Invocation, WebResponse>>> handlers = new LinkedHashMap<>();

    public HttpRouter(Json json) {
        this.json = requireNonNull(json, "json is a required parameter");
        build();
    }

    @Override
    public WebResponse apply(WebRequest s, Invocation invocation) {
        var operationHandler = handlers.get(s.getMethod());
        if (operationHandler == null || operationHandler.isEmpty()) {
            return unsupported("Unsupported Operation");
        }
        for (var function : operationHandler.entrySet()) {
            if (s.matches(function.getKey())) {
                final var apply = function.getValue().apply(s, invocation);
                if (apply == null) {
                    return serverError().body("call failed");
                }
                return apply;
            }
        }
        return notFound().body("NOT FOUND");
    }

    public HttpRouter addMapping(String operation, String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        if (!handlers.containsKey(Strings.requireNotBlank(operation, "operation is required"))) {
            handlers.put(operation, new LinkedHashMap<>());
        }
        handlers.get(operation).put(Strings.requireNotBlank(path, "path is required"), requireNonNull(handler, "handler is required"));
        return this;
    }

    public HttpRouter get(String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        return addMapping("GET", path, handler);
    }

    public HttpRouter post(String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        return addMapping("POST", path, handler);
    }

    public HttpRouter put(String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        return addMapping("PUT", path, handler);
    }

    public HttpRouter delete(String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        return addMapping("DELETE", path, handler);
    }

    public HttpRouter head(String path, BiFunction<WebRequest, Invocation, WebResponse> handler) {
        return addMapping("HEAD", path, handler);
    }


    protected <T> WebResponse ok(Object body) {
        if (body instanceof String) {
            return response().ok().body(body.toString());
        }
        return response().ok().body(getJson().marshal(body));
    }

    protected <T> WebResponse userError(Object body) {
        if (body instanceof String) {
            return response().userError().body(body.toString());
        }
        return response().userError().body(getJson().marshal(body));
    }

    protected <T> WebResponse serverError(Object body) {
        if (body instanceof String) {
            return response().serverError().body(body.toString());
        }
        return response().serverError().body(getJson().marshal(body));
    }

    protected <T> WebResponse notFound(Object body) {
        if (body instanceof String) {
            return response().notFound().body(body.toString());
        }
        return response().notFound().body(getJson().marshal(body));
    }

    public abstract void build();

    public Json getJson() {
        return json;
    }

    public Map<String, Map<String, BiFunction<WebRequest, Invocation, WebResponse>>> getHandlers() {
        return handlers;
    }
}
