package net.odoframework.service;


import net.odoframework.container.events.ContainerStartedEvent;
import net.odoframework.container.events.EventPublisher;
import net.odoframework.container.injection.Container;
import net.odoframework.container.metrics.Metrics;
import net.odoframework.container.tx.TxManager;
import net.odoframework.util.Timer;

import java.time.LocalDateTime;
import java.util.Optional;
import java.util.ServiceLoader.Provider;
import java.util.function.BiFunction;

import static java.time.LocalTime.now;


public class Bootstrap {


    private static final Provider<Bootstrap> BOOTSTRAP = new Provider<>() {
        @Override
        public Class<? extends Bootstrap> type() {
            return Bootstrap.class;
        }

        @Override
        public Bootstrap get() {
            return new Bootstrap(Container.getModuleContainer());
        }
    };

    private static Bootstrap BOOTSTRAP_CACHE = null;

    public static Bootstrap getBoostrap() {
        if (BOOTSTRAP_CACHE == null) {
            synchronized (BOOTSTRAP) {
                //double null check locking pattern
                if (BOOTSTRAP_CACHE == null) {
                    BOOTSTRAP_CACHE = BOOTSTRAP.get();
                }
            }
        }
        return BOOTSTRAP_CACHE;
    }


    private final Container container;
    private final BiFunction<?, ?, ?> handler;
    private final Optional<TxManager> txManager;
    protected final Metrics metrics;
    private final RequestConverter<?, ?> requestConverter;
    private final ResponseConverter<?, ?> responseConverter;

    protected final ProviderRuntime<?, ?, ?> runtime;


    public Bootstrap(Container container) {
        var timer = LocalDateTime.now();
        try {
            this.container = container;

            handler = (BiFunction<?, ?, ?>) container.resolve(ServiceFunction.class)
                    .orElseThrow(() -> new IllegalStateException("No " + ServiceFunction.class.getName() + " provided"));

            txManager = container.resolve(TxManager.class);

            metrics = container.resolve(Metrics.class)
                    .orElseThrow(() -> new IllegalStateException("No " + TxManager.class.getName() + " provided"));

            runtime = container.resolve(ProviderRuntime.class)
                    .orElseThrow(() -> new IllegalStateException("No " + ProviderRuntime.class.getName() + " provided"));

            requestConverter = container.resolve(RequestConverter.class)
                    .orElse(runtime.getProviderDefaultRequestConverter());

            responseConverter = container.resolve(ResponseConverter.class)
                    .orElse(runtime.getProviderDefaultResponseConverter());
        } finally {
            EventPublisher.publish(new ContainerStartedEvent(timer));
        }
    }

    public Container getContainer() {
        return container;
    }

    public BiFunction<?, ?, ?> getHandler() {
        return handler;
    }

    public Optional<TxManager> getTxManager() {
        return txManager;
    }

    public Metrics getMetrics() {
        return metrics;
    }

    public ProviderRuntime<?, ?, ?> getRuntime() {
        return runtime;
    }

    public RequestConverter<?, ?> getRequestConverter() {
        return requestConverter;
    }

    public ResponseConverter<?, ?> getResponseConverter() {
        return responseConverter;
    }

    @SuppressWarnings("unchecked")
    public Object handleRequest(Object payload, Object context) {
        ProviderRuntime runtime = getRuntime();
        return runtime.handleRequest(this, payload, context);
    }

}



