package org.odoframework.service;


import org.odoframework.container.injection.Container;
import org.odoframework.container.metrics.Metrics;
import org.odoframework.container.tx.TxManager;
import org.odoframework.util.Timer;

import java.util.Optional;
import java.util.ServiceLoader.Provider;
import java.util.logging.Logger;


public class Bootstrap {


    private static final Provider<Bootstrap> BOOTSTRAP = new Provider<>() {
        @Override
        public Class<? extends Bootstrap> type() {
            return Bootstrap.class;
        }

        @Override
        public Bootstrap get() {
            return new Bootstrap(Container.getModuleContainer());
        }
    };

    private static Bootstrap BOOTSTRAP_CACHE = null;

    public static Bootstrap getBoostrap() {
        if (BOOTSTRAP_CACHE == null) {
            synchronized (BOOTSTRAP) {
                //double null check locking pattern
                if (BOOTSTRAP_CACHE == null) {
                    BOOTSTRAP_CACHE = BOOTSTRAP.get();
                }
            }
        }
        return BOOTSTRAP_CACHE;
    }

    private static final Logger LOG = Logger.getLogger(Bootstrap.class.getName());

    private final Container container;
    private final ServiceFunction<?, ?> handler;
    private Optional<TxManager> txManager;
    protected Metrics metrics;
    private RequestConverter<?, ?> requestConverter;
    private ResponseConverter<?, ?> responseConverter;

    protected ProviderRuntime<?, ?, ?> runtime;


    public Bootstrap(Container container) {
        var timer = Timer.start();
        try {
            this.container = container;

            handler = container.resolve(ServiceFunction.class)
                    .orElseThrow(() -> new IllegalStateException("No " + ServiceFunction.class.getName() + " provided"));

            txManager = container.resolve(TxManager.class);

            metrics = container.resolve(Metrics.class)
                    .orElseThrow(() -> new IllegalStateException("No " + TxManager.class.getName() + " provided"));

            runtime = container.resolve(ProviderRuntime.class)
                    .orElseThrow(() -> new IllegalStateException("No " + ProviderRuntime.class.getName() + " provided"));

            requestConverter = container.resolve(RequestConverter.class)
                    .orElse(runtime.getProviderDefaultRequestConverter());

            responseConverter = container.resolve(ResponseConverter.class)
                    .orElse(runtime.getProviderDefaultResponseConverter());
        } finally {
            LOG.fine("TIME TAKEN -> " + timer.timeTaken());
        }
    }

    public Container getContainer() {
        return container;
    }

    public ServiceFunction<?, ?> getHandler() {
        return handler;
    }

    public Optional<TxManager> getTxManager() {
        return txManager;
    }

    public Metrics getMetrics() {
        return metrics;
    }

    public ProviderRuntime<?, ?, ?> getRuntime() {
        return runtime;
    }

    public RequestConverter<?, ?> getRequestConverter() {
        return requestConverter;
    }

    public ResponseConverter<?, ?> getResponseConverter() {
        return responseConverter;
    }

    public Object handleRequest(Object payload, Object context) {
        ProviderRuntime runtime = getRuntime();
        return runtime.handleRequest(this, payload, context);
    }

}



