package net.odoframework.beans;

import net.odoframework.beans.types.ConverterRegistry;
import net.odoframework.util.Pair;

import java.util.function.BiConsumer;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;

public class ConvertingPropertyBinding<Source, TargetType, SourceType> extends PropertyBinding<Source, TargetType> {

    @SuppressWarnings("unchecked")
    private Function<SourceType, TargetType> toTarget = (src) -> (TargetType) src;
    @SuppressWarnings("unchecked")
    private Function<TargetType, SourceType> fromTarget = (src) -> (SourceType) src;

    public ConvertingPropertyBinding(
            Function<Source, TargetType> getter,
            BiConsumer<Source, TargetType> setter,
            Function<SourceType, TargetType> toTarget,
            Function<TargetType, SourceType> fromTarget) {
        super(getter, setter);
        this.toTarget = toTarget;
        this.fromTarget = fromTarget;
    }

    public ConvertingPropertyBinding(Function<Source, TargetType> getter) {
        super(getter);
    }

    public ConvertingPropertyBinding(BiConsumer<Source, TargetType> setter) {
        super(setter);
    }


    public SourceType getConverted(Source instance) {
        var targetType = super.get(instance);
        return requireNonNull(fromTarget, "fromTarget is required").apply(targetType);
    }

    public void setConverted(Source instance, SourceType value) {
        var convertedValue = requireNonNull(toTarget, "toTarget is required").apply(value);
        super.set(instance, convertedValue);
    }

    public static <T, K> ConvertingPropertyBinding<T, K, K> passThrough(
            Function<T, K> getter,
            BiConsumer<T, K> setter) {

        return new ConvertingPropertyBinding<>(
                getter,
                setter,
                (it) -> it,
                (it) -> it);
    }

    public static <T, K, Z> ConvertingPropertyBinding<T, K, Z> withConversion(
            Function<T, K> getter,
            BiConsumer<T, K> setter,
            Class<K> target,
            Class<Z> raw
    ) {

        var targetConverter = ConverterRegistry.get(target, raw).orElseThrow();
        var rawConverter = ConverterRegistry.get(raw, target).orElseThrow();

        return new ConvertingPropertyBinding<>(
                getter,
                setter,
                rawConverter,
                targetConverter);
    }

}
