package net.odoframework.sql;

import net.odoframework.util.ListBackedMap;
import net.odoframework.util.Strings;
import net.odoframework.util.TrackedMap;

import java.util.*;

public class Record {


    private TrackedMap<String, Object> row;
    private String table;
    private String[] idFields;
    private Map<String, List<Record>> children;


    public Record(String table, String... primaryKeyFields) {
        this.table = Strings.requireNotBlank(table, "table is a required parameter").trim();
        this.idFields = primaryKeyFields;
        this.row = new TrackedMap<>(new ListBackedMap<>());
        this.children = new TrackedMap<>(new ListBackedMap<>());
    }

    public Record primaryKeyFields(String... fields) {
        this.idFields = fields;
        return this;
    }


    public Map<String, ?> getKey() {
        if (this.idFields.length == 0) {
            throw new IllegalStateException("idFields has not been initialized");
        }
        var keyMap = new ListBackedMap<String, Object>();
        for (String s : idFields) {
            var value = this.row.get(s);
            if (value == null) {
                throw new IllegalArgumentException("missing primary key field " + s);
            }
            keyMap.put(s, value);
        }
        return keyMap;
    }

    public Map<String, ?> getRow() {
        return Collections.unmodifiableMap(row);
    }


    @SuppressWarnings("unchecked")
    public <K> K get(String field) {
        return (K) row.get(field);
    }

    public int getInt(String name) {
        return get(name);
    }

    public long getLong(String name) {
        return get(name);
    }

    public boolean getBoolean(String name) {
        return get(name);
    }

    public String getString(String name) {
        return get(name);
    }

    public char getChar(String name) {
        return get(name);
    }

    public <K extends Enum<K>> K getEnum(String name, Class<K> enumType) {
        Object val = get(name);
        if (val == null) {
            return null;
        }
        if (val instanceof String) {
            return (K) Enum.valueOf(enumType, val.toString());
        }
        if (val instanceof Integer) {
            return enumType.getEnumConstants()[(int) val];
        }
        throw new IllegalStateException(val.getClass().getName() + " cannot be converted to an enumeration");
    }

    public Record set(String field, Object value) {
        row.put(field, value);
        return this;
    }

    public Record remove(String field) {
        row.remove(field);
        return this;
    }

    public Record removeChild(String table) {
        children.remove(table);
        return this;
    }

    public Record addChild(String table, Record child) {
        if (!this.children.containsKey(table)) {
            this.children.put(table, new ArrayList<>());
        }
        this.children.get(table).add(child);
        return this;
    }

    public List<Record> getChildren(String table) {
        if (!children.containsKey(table)) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(children.get(table));
    }

    public String getTable() {
        return table;
    }

    public String[] getIdFields() {
        return idFields;
    }

    public Map<String, List<Record>> getChildren() {
        return Collections.unmodifiableMap(children);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Record)) return false;
        Record record = (Record) o;
        return row.equals(record.row) && table.equals(record.table) && idFields.equals(record.idFields) && children.equals(record.children);
    }

    @Override
    public int hashCode() {
        return Objects.hash(row, table, idFields, children);
    }

    @Override
    public String toString() {
        return table + "[" + getKey().toString() + ']';
    }
}
