package net.odoframework.beans;


import lombok.EqualsAndHashCode;
import lombok.Getter;

import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Function;

@EqualsAndHashCode
public class PropertyBinding<T, K> implements Binding<T, K> {


    @Getter
    protected final Function<T, K> getter;
    @Getter
    protected final BiConsumer<T, K> setter;

    public PropertyBinding(Function<T, K> getter, BiConsumer<T, K> setter) {
        this.getter = getter;
        this.setter = setter;
    }

    public PropertyBinding(Function<T, K> getter) {
        this(getter, null);
    }

    public PropertyBinding(BiConsumer<T, K> setter) {
        this(null, setter);
    }

    public static <T, K> Binding<T,K> binding(Function<T, K> getter, BiConsumer<T, K> setter) {
        if (getter == null && setter == null) {
            throw new IllegalArgumentException("You must have at least one null getter and setter");
        }
        return new PropertyBinding<>(getter, setter);
    }

    public static <T, K> Binding<T,K> binding(BiConsumer<T, K> setter) {
        return new PropertyBinding<>(null, setter);
    }

    public static <T, K> Binding<T,K> binding(Function<T, K> getter) {
        return new PropertyBinding<>(getter, null);
    }

    @Override
    public K get(T instance) {
        return Objects.requireNonNull(getter, "no getter for this property binding")
                .apply(Objects.requireNonNull(instance));
    }

    @Override
    public void set(T instance, K value) {
        Objects.requireNonNull(setter, "no setter for this property binding")
                .accept(Objects.requireNonNull(instance), value);
    }

}
