package net.odoframework.beans;

import net.odoframework.util.ListBackedMap;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;
import static net.odoframework.util.Strings.requireNotBlank;

public class ObjectToMap<Source> {

    private final Map<String, Object> targetMap;
    private Map<String, ConvertingPropertyBinding> bindings;
    private final boolean strict;
    private MapToObjectRegistry registry;
    private Supplier<Source> newInstanceConstructor;

    public ObjectToMap(boolean strict, MapToObjectRegistry registry, Supplier<Source> newInstanceConstructor) {
        targetMap = new ListBackedMap<>();
        this.strict = strict;
        this.registry = registry;
        newInstanceConstructor(newInstanceConstructor);

    }

    public ObjectToMap(boolean strict, Supplier<Source> newInstanceConstructor) {
        this(strict, null, newInstanceConstructor);
    }

    public ObjectToMap(Supplier<Source> newInstanceConstructor) {
        this(true, null, newInstanceConstructor);
    }

    public ObjectToMap<Source> binding(String name, ConvertingPropertyBinding<Source, ?, ?> binding) {
        if (bindings == null) {
            bindings = new LinkedHashMap<>();
        }
        bindings.put(
                requireNotBlank(name, "name is a required parameter"),
                requireNonNull(binding, "binding is a required parameter")
        );
        return this;
    }

    public ObjectToMap newInstanceConstructor(Supplier<Source> source) {
        this.newInstanceConstructor = Objects.requireNonNull(source, "source is a required constructor");
        return this;
    }

    public Supplier<Source> getNewInstanceConstructor() {
        return this.newInstanceConstructor;
    }

    @SuppressWarnings("unchecked")
    public Source fromTarget(Map<String, ?> instance) {
        requireNonNull(instance);
        if (bindings == null || bindings.isEmpty()) {
            throw new IllegalStateException("no bindings have been defined");
        }
        var newInstance = getNewInstanceConstructor().get();
        this.bindings.forEach((name, binding) -> {
            if (instance.containsKey(name)) {
                var sourceValue = targetMap.get(name);
                if (sourceValue instanceof Map && registry != null && registry.contains(sourceValue.getClass())) {
                    registry.get(sourceValue.getClass()).map(it -> it.fromTarget((Map<String, ?>) sourceValue));
                } else {
                    binding.setConverted(instance, sourceValue);
                }
            } else if (strict) {
                throw new IllegalStateException("targetMap has no key" + name);
            }
        });
        return newInstance;
    }

    public Map<String, ?> toTarget(Source instance) {
        requireNonNull(instance);
        if (bindings == null || bindings.isEmpty()) {
            throw new IllegalStateException("no bindings have been defined");
        }
        this.bindings.forEach((name, binding) -> {
            var value = binding.getConverted(instance);
            targetMap.put(name, value);
        });
        return targetMap;
    }

    public boolean isStrict() {
        return strict;
    }
}
