package net.odoframework.container.injection;

import net.odoframework.container.events.Log;
import net.odoframework.util.ConfigLoader;
import net.odoframework.util.EnvironmentUtils;

import java.util.Properties;

import static java.util.Objects.requireNonNull;
import static net.odoframework.util.Strings.isNotBlank;

public class ConfigurationProperties {

    public static final String ODO_PROFILE = "ODO_PROFILE";
    public static final String SYS_ODO_PROFILE = "odo.profile";
    public static final String APPLICATION_PROPERTIES = "/application.properties";
    public static final String ODO_CONFIGURATION = "odo.configuration";


    public synchronized static Properties loadConfig(Class<?> source) {
        var config = getConfig();
        if (config != null) {
            return config;
        }
        requireNonNull(source.getModule());
        var properties = new Properties();
        var actualEnv = EnvironmentUtils.resolve(ODO_PROFILE, SYS_ODO_PROFILE).orElse(null);
        if (isNotBlank(actualEnv)) {
            Log.debug(ConfigurationProperties.class, "SET ODO ENVIRONMENT TO '" + actualEnv + "'");
            var envPropertiesFile = "/application-" + actualEnv.trim() + ".properties";
            Log.debug(ConfigurationProperties.class, "LOADING -> " + envPropertiesFile);
            properties.putAll(ConfigLoader.loadProperties(APPLICATION_PROPERTIES, envPropertiesFile));
        } else {
            properties.putAll(ConfigLoader.loadProperties(APPLICATION_PROPERTIES));
            Log.warn(ConfigurationProperties.class, "NO ENVIRONMENT CONFIGURED FOR ODO");
        }
        System.getenv().forEach((key, value) -> properties.put("env." + key, value));
        properties.putAll(System.getProperties());
        setConfig(properties);
        return properties;
    }

    public static Properties getConfig() {
        return (Properties) System.getProperties().get(ODO_CONFIGURATION);
    }

    public static void reset() {
        System.getProperties().remove(ODO_CONFIGURATION);
    }

    public synchronized static void setConfig(Properties p) {
        System.getProperties().put(ODO_CONFIGURATION, p);
    }


    public static Properties findByPrefix(String prefix, boolean stripPrefix) {
        if (getConfig() == null) {
            loadConfig(ConfigurationProperties.class);
        }
        return ConfigLoader.findByPrefix(getConfig(), prefix, stripPrefix);
    }


}
