package org.odoframework.container.injection;

import jakarta.inject.Provider;
import jakarta.inject.Singleton;
import org.odoframework.container.ApplicationBuilder;
import org.odoframework.container.Module;
import org.odoframework.container.metrics.MetricsService;
import org.odoframework.container.tx.TxResource;
import org.odoframework.container.tx.TxScope;

import java.util.*;
import java.util.function.Supplier;
import java.util.logging.Logger;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.*;
import static org.odoframework.util.Strings.requireNotBlank;

public class Container {


    private final Map<String, Stack<BeanDefinition<?>>> instances;
    private final Map<String, Object> singletons;
    private final Properties configuration;
    private Set<String> startedBeans;

    private static Container SINGLETON_CONTAINER = null;
    private static Supplier<Container> CONTAINER_SUPPLIER = null;


    private static final Logger LOG = Logger.getLogger(Container.class.getName());


    public Container(Properties configuration) {
        instances = new LinkedHashMap<>();
        singletons = new LinkedHashMap<>();
        this.configuration = ofNullable(configuration).orElse(new Properties());
    }

    public <T> Optional<T> resolve(Class<T> type) {
        return resolve(requireNonNull(type.getName(), "type is a required parameter"));
    }

    public boolean contains(Class<?> name) {
        return instances.containsKey(name.getName());
    }

    public boolean contains(String name) {
        return instances.containsKey(name);
    }

    Collection<BeanDefinition<?>> getDefinitions(String name) {
        return instances.get(name);
    }



    @SuppressWarnings("unchecked")
    public <T> Optional<T> resolve(String name) {
        return MetricsService.getMetrics().doSection(name, () -> {
            final var componentList = instances.get(requireNotBlank(name, "name is a required parameter"));
            if (componentList == null || componentList.empty()) {
                return empty();
            }
            BeanDefinition<?> component = null;
            for (BeanDefinition<?> beanDefinition : componentList) {
                if (beanDefinition.getCondition().test(this)) {
                    component = beanDefinition;
                }
            }
            if (component == null) {
                return empty();
            }
            if (component.isTransactional()) {
                final var finalComponent = component;
                try {
                    return of(TxScope.define(component.getName(), () -> createInstance(name, finalComponent)));
                } finally {
                    TxScope.endScope();
                }
            }
            return of(createInstance(component.getName(), component));
        });

    }

    @SuppressWarnings("unchecked")
    private <T> T createInstance(String name, BeanDefinition<?> component) {
        if (singletons.containsKey(name)) {
            return (T) singletons.get(name);
        }
        var instance = (T) component.apply(this);
        final var singleton = isSingleton(component, instance);
        if (singleton) {
            singletons.put(name, instance);
        }
        runStart(name, instance, singleton);
        if (instance instanceof TxResource) {
            TxScope.addResource((TxResource) instance);
        }
        return instance;
    }

    private <T> boolean isSingleton(BeanDefinition<?> component, T instance) {
        return component.isSingleton() || instance.getClass().isAnnotationPresent(Singleton.class);
    }

    private <T> void runStart(String name, T instance, boolean isSingleton) {
        if (this.startedBeans == null) {
            this.startedBeans = new HashSet<>();
        }
        if (instance instanceof Runnable && !this.startedBeans.contains(name)) {
            LOG.fine("Running " + name + " as startup");
            ((Runnable) instance).run();
            this.startedBeans.add(name);
        }
        if (instance instanceof Runnable && !isSingleton) {
            LOG.fine("Running " + name + " as startup in prototype mode");
            ((Runnable) instance).run();
        }
    }

    public Container register(BeanDefinition<?> beanDefinition) {
        requireNonNull(beanDefinition, "componentDefinition is a required parameter");
        LOG.fine("Registering bean: " + requireNotBlank(beanDefinition.getName(), "name is required on the component"));
        if (this.instances.containsKey(beanDefinition.getName())) {
            LOG.warning("Container already contains bean: " + beanDefinition.getName());
        } else {
            this.instances.put(beanDefinition.getName(), new Stack<>());
        }
        this.instances.get(beanDefinition.getName()).push(beanDefinition);
        return this;
    }

    public Optional<String> getValue(String name) {
        return Optional.ofNullable(this.configuration.getProperty(requireNotBlank(name, "name parameter cannot be null")));
    }

    public Optional<Integer> getValueInteger(String name) {
        return Optional
                .ofNullable(this.configuration.getProperty(requireNotBlank(name, "name parameter cannot be null")))
                .map(Integer::valueOf);
    }

    public Optional<Boolean> getValueBoolean(String name) {
        return Optional
                .ofNullable(this.configuration.getProperty(requireNotBlank(name, "name parameter cannot be null")))
                .map(Boolean::parseBoolean);
    }

    public Optional<String> getValue(String name, String defaultValue) {
        return Optional.ofNullable(this.configuration.getProperty(requireNotBlank(name, "name parameter cannot be null")));
    }

    //hack for Lambda
    private Container getContainer() {
        return this;
    }

    public <T> Provider<Optional<T>> getLazyBean(String name) {
        return () -> getContainer().resolve(name);
    }

    public <T> Provider<Optional<T>> getLazyBean(Class<T> type) {
        return getLazyBean(requireNonNull(type, "type is a required parameter").getName());
    }


    public Properties getConfiguration() {
        return configuration;
    }

    public static void setContainerInstance(Container singletonContainer) {
        SINGLETON_CONTAINER = singletonContainer;
    }

    public static void setContainerBuilder(Supplier<Container> supplier) {
        CONTAINER_SUPPLIER = requireNonNull(supplier, "container supplier is a required parameter");
        getContainerInstance();
    }

    public static synchronized Container getContainerInstance() {
        if (SINGLETON_CONTAINER == null) {
            var lookupModule = getModuleContainer();
            if (lookupModule != null) {
                SINGLETON_CONTAINER = lookupModule;
            }
            return SINGLETON_CONTAINER = CONTAINER_SUPPLIER.get();
        }
        return SINGLETON_CONTAINER;
    }

    public static synchronized Container getModuleContainer() {
        return ServiceLoader.load(Module.class)
                .stream()
                .filter(it -> ApplicationBuilder.class.isAssignableFrom(it.type()))
                .peek(it -> LOG.warning("LOADING -> " + it.getClass().getName()))
                .map(ServiceLoader.Provider::get)
                .map(it -> ((ApplicationBuilder) it).getContainer())
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("Could not find an instance of " + ApplicationBuilder.class.getName() + " via loaded modules"));
    }

    public Optional<Long> getValueLong(String value) {
        return getValue(value).map(Long::parseLong);
    }

    public Optional<Double> getValueDouble(String value) {
        return getValue(value).map(Double::parseDouble);
    }
}
