package org.odoframework.container.injection;

import org.odoframework.util.ConfigLoader;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Properties;
import java.util.logging.Logger;

import static java.util.Objects.requireNonNull;
import static org.odoframework.util.Strings.isNotBlank;

public class ConfigurationProperties {

    public static final String ODO_PROFILE = "ODO_PROFILE";
    public static final String SYS_ODO_PROFILE = "odo.profile";
    public static final String APPLICATION_PROPERTIES = "/application.properties";
    private static Properties CONFIG;

    private static final Logger LOG = Logger.getLogger(ConfigurationProperties.class.getName());

    public static Properties loadConfig(Class<?> source, String defaultProfile) {
        requireNonNull(source.getModule());
        var env = System.getenv().get(ODO_PROFILE);
        var sysEnv = System.getProperty(SYS_ODO_PROFILE);
        var files = new ArrayList<String>();
        InputStream properties = null;
        try {
            LOG.fine("Loading: " + APPLICATION_PROPERTIES);
            properties = source.getModule().getResourceAsStream(APPLICATION_PROPERTIES);
            if (properties != null) {
                LOG.fine("Loaded: " + APPLICATION_PROPERTIES);
                files.add(APPLICATION_PROPERTIES);
            }
        } catch (IOException e) {
            //do nothing
            LOG.severe(APPLICATION_PROPERTIES + " not found: " + e.getMessage());
        }
        var actualEnv = isNotBlank(sysEnv)
                ? sysEnv
                : env;
        if (isNotBlank(actualEnv)) {
            LOG.fine("SET ODO ENVIRONMENT TO '" + actualEnv + "'");
            var envPropertiesFile = "/application-" + actualEnv.trim() + ".properties";
            InputStream envProperties = null;
            try {
                envProperties = source.getModule().getResourceAsStream(envPropertiesFile);
                if (envProperties != null) {
                    LOG.fine("Loading: " + envPropertiesFile);
                    files.add(envPropertiesFile);
                    LOG.fine("Loaded: " + envPropertiesFile);
                } else {
                    throw new IllegalStateException(actualEnv + " configured but could not load " + envPropertiesFile);
                }
            } catch (IOException e) {
                throw new IllegalStateException(actualEnv + " configured but could not load " + envPropertiesFile, e);
            }
        } else {
            LOG.warning("NO ENVIRONMENT CONFIGURED FOR ODO");
        }
        CONFIG = ConfigLoader.loadProperties(source.getModule(), files.toArray(new String[0]));
        return getConfig();
    }

    public static Properties getConfig() {
        return CONFIG;
    }


    public static Properties findByPrefix(String prefix, boolean stripPrefix) {
        return ConfigLoader.findByPrefix(getConfig(), prefix, stripPrefix);
    }


}
