package org.odoframework.container;

import org.odoframework.container.injection.BeanDefinition;
import org.odoframework.container.injection.Container;
import org.odoframework.container.metrics.Metrics;
import org.odoframework.container.metrics.MetricsService;

import java.util.*;

import static org.odoframework.container.injection.BeanDefinition.bean;
import static org.odoframework.util.Strings.requireNotBlank;

public abstract class ModuleBuilder implements Module {

    private Container container;
    private static final Set<String> startupBeans = new LinkedHashSet<>();

    public ModuleBuilder() {
    }

    protected ModuleBuilder register(BeanDefinition<?> beanDefinition) {
        this.container.register(beanDefinition);
        return this;
    }


    public <T> BeanDefinition<T> provide(String name) {
        final var beanDefinition = new BeanDefinition<T>(name);
        this.register(beanDefinition);
        return beanDefinition;
    }

    public <T> BeanDefinition<T> provide(Class<T> type) {
        final var beanDefinition = new BeanDefinition<T>(type);
        this.register(beanDefinition);
        return beanDefinition;
    }

    public Container getContainer() {
        return container;
    }

    @Override
    public final Container apply(Container container, Properties properties) {
        this.container = container;
        containerCreated(container);
        provide(Metrics.class).with(MetricsService::getMetrics);
        beforeContainerCreated(container);
        build();
        afterContainerCreated(container);
        return this.container;
    }

    protected void beforeContainerCreated(Container container) {

    }


    protected void afterContainerCreated(Container container) {

    }


    public ModuleBuilder addStartupBean(BeanDefinition<?> bean) {
        register(bean);
        startupBeans.add(requireNotBlank(bean.getName()));
        return this;
    }

    public ModuleBuilder addStartupBean(Class<?> type) {
        final BeanDefinition<?> bean = BeanDefinition.bean(type);
        register(bean);
        startupBeans.add(bean.getName());
        return this;
    }


    protected void containerCreated(Container container) {

    }

    public abstract void build();

    public static Set<String> getStartupBeans() {
        return startupBeans;
    }



}
