package org.odoframework.container;

import org.odoframework.container.injection.ConfigurationProperties;
import org.odoframework.container.injection.Container;
import org.odoframework.container.injection.ContainerFactory;

import java.util.Comparator;
import java.util.ServiceLoader;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public abstract class ApplicationBuilder extends ModuleBuilder {


    private static final Logger LOG = Logger.getLogger(ApplicationBuilder.class.getName());

    private static String DEFAULT_PROFILE = "dev";


    public ApplicationBuilder() {
        var config = ConfigurationProperties.loadConfig(getClass(), DEFAULT_PROFILE);
        var container = ContainerFactory.create(config);
        apply(container, ConfigurationProperties.getConfig());
    }


    @Override
    protected final void containerCreated(Container container) {
    }


    @Override
    protected void beforeContainerCreated(Container container) {
        final var modules = ServiceLoader
                .load(Module.class);
        LOG.fine("DETECTED " + modules.stream().count() + " from the System module loader");
        //check system modules
        var sanitizedModules = modules
                .stream()
                .filter(it -> !it.type().getName().equals(this.getClass().getName()))
                .collect(Collectors.toList());
        for (var moduleProvider : sanitizedModules) {
            var module = moduleProvider.get();
            LOG.fine("loading module " + module.getClass().getName());
            if (module.getClass().getName().equals(this.getClass().getName())) {
                continue;
            }
            if (!module.getClass().getPackageName().startsWith("org.odoframework") && module.getPrecedence() < 0) {
                throw new IllegalStateException("Module " + module.getClass().getName() + " odo reserves module preferences below 0");
            }
        }

        sanitizedModules
                .stream()
                .sorted(Comparator.comparing(a -> a.get().getPrecedence()))
                .map(ServiceLoader.Provider::get)
                .peek(it -> LOG.info("LOADED MODULE " + it.getClass().getName() + " from Module: " + it.getClass().getModule().getName()))
                .forEach(it -> it.apply(container, container.getConfiguration()));
    }

    public static void setDefaultProfile(String defaultProfile) {
        DEFAULT_PROFILE = defaultProfile;
    }

    @Override
    protected final void afterContainerCreated(Container container) {
        for (String startupBean : getStartupBeans()) {
            //this works by getting an instance of the bean and then if the component is Runnable it will call run
            container
                    .resolve(startupBean)
                    .map(bean -> {
                        if (!(bean instanceof Runnable)) {
                            throw new IllegalStateException(startupBean + " must be an instance of Runnable to run in startup");
                        }
                        return bean;
                    })
                    .orElseThrow(() -> new IllegalStateException(startupBean + " does not exist but is configured to run in startup"));
        }
        postContainerCreated(container);
    }

    private void postContainerCreated(Container container) {

    }
}
