/*
 * Decompiled with CFR 0.152.
 */
package net.named_data.jndn.encrypt;

import java.io.IOException;
import java.security.Key;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import net.named_data.jndn.Data;
import net.named_data.jndn.Face;
import net.named_data.jndn.Interest;
import net.named_data.jndn.KeyLocatorType;
import net.named_data.jndn.Name;
import net.named_data.jndn.NetworkNack;
import net.named_data.jndn.OnData;
import net.named_data.jndn.OnNetworkNack;
import net.named_data.jndn.OnTimeout;
import net.named_data.jndn.encrypt.EncryptError;
import net.named_data.jndn.encrypt.EncryptedContent;
import net.named_data.jndn.encrypt.EncryptorV2;
import net.named_data.jndn.security.KeyChain;
import net.named_data.jndn.security.SafeBag;
import net.named_data.jndn.security.pib.Pib;
import net.named_data.jndn.security.pib.PibIdentity;
import net.named_data.jndn.security.pib.PibKey;
import net.named_data.jndn.security.v2.Validator;
import net.named_data.jndn.util.Blob;

public class DecryptorV2 {
    private final PibKey credentialsKey_;
    private final Face face_;
    private final KeyChain keyChain_;
    private final KeyChain internalKeyChain_;
    private final HashMap<Name, ContentKey> contentKeys_ = new HashMap();
    private static final Logger logger_ = Logger.getLogger(DecryptorV2.class.getName());

    public DecryptorV2(PibKey credentialsKey, Validator validator, KeyChain keyChain, Face face) {
        this.credentialsKey_ = credentialsKey;
        this.face_ = face;
        this.keyChain_ = keyChain;
        try {
            this.internalKeyChain_ = new KeyChain("pib-memory:", "tpm-memory:");
        }
        catch (Exception ex) {
            throw new Error("Error creating in-memory KeyChain: " + ex);
        }
    }

    public void shutdown() {
        for (ContentKey contentKey : this.contentKeys_.values()) {
            if (contentKey.pendingInterest <= 0L) continue;
            this.face_.removePendingInterest(contentKey.pendingInterest);
            contentKey.pendingInterest = 0L;
            for (ContentKey.PendingDecrypt pendingDecrypt : contentKey.pendingDecrypts) {
                pendingDecrypt.onError.onError(EncryptError.ErrorCode.CkRetrievalFailure, "Canceling pending decrypt as ContentKey is being destroyed");
            }
            contentKey.pendingDecrypts.clear();
        }
    }

    public final void decrypt(EncryptedContent encryptedContent, DecryptSuccessCallback onSuccess, EncryptError.OnError onError) throws IOException {
        boolean isNew;
        if (encryptedContent.getKeyLocator().getType() != KeyLocatorType.KEYNAME) {
            logger_.log(Level.INFO, "Missing required KeyLocator in the supplied EncryptedContent block");
            onError.onError(EncryptError.ErrorCode.MissingRequiredKeyLocator, "Missing required KeyLocator in the supplied EncryptedContent block");
            return;
        }
        if (!encryptedContent.hasInitialVector()) {
            logger_.log(Level.INFO, "Missing required initial vector in the supplied EncryptedContent block");
            onError.onError(EncryptError.ErrorCode.MissingRequiredInitialVector, "Missing required initial vector in the supplied EncryptedContent block");
            return;
        }
        Name ckName = encryptedContent.getKeyLocatorName();
        ContentKey contentKey = this.contentKeys_.get(ckName);
        boolean bl = isNew = contentKey == null;
        if (isNew) {
            contentKey = new ContentKey();
            this.contentKeys_.put(ckName, contentKey);
        }
        if (contentKey.isRetrieved) {
            DecryptorV2.doDecrypt(encryptedContent, contentKey.bits, onSuccess, onError);
        } else {
            logger_.log(Level.INFO, "CK {0} not yet available, so adding to the pending decrypt queue", ckName);
            contentKey.pendingDecrypts.add(new ContentKey.PendingDecrypt(encryptedContent, onSuccess, onError));
        }
        if (isNew) {
            this.fetchCk(ckName, contentKey, onError, 3);
        }
    }

    private void fetchCk(final Name ckName, final ContentKey contentKey, final EncryptError.OnError onError, final int nTriesLeft) {
        logger_.log(Level.INFO, "Fetching CK {0}", ckName);
        try {
            contentKey.pendingInterest = this.face_.expressInterest(new Interest(ckName).setMustBeFresh(false).setCanBePrefix(true), new OnData(){

                @Override
                public void onData(Interest ckInterest, Data ckData) {
                    try {
                        contentKey.pendingInterest = 0L;
                        Name[] kdkPrefix = new Name[1];
                        Name[] kdkIdentityName = new Name[1];
                        Object[] kdkKeyName = new Name[1];
                        if (!DecryptorV2.extractKdkInfoFromCkName(ckData.getName(), ckInterest.getName(), onError, kdkPrefix, kdkIdentityName, (Name[])kdkKeyName)) {
                            return;
                        }
                        PibIdentity kdkIdentity = null;
                        try {
                            kdkIdentity = DecryptorV2.this.internalKeyChain_.getPib().getIdentity(kdkIdentityName[0]);
                        }
                        catch (Pib.Error error) {
                            // empty catch block
                        }
                        if (kdkIdentity != null) {
                            PibKey kdkKey = null;
                            try {
                                kdkKey = kdkIdentity.getKey(kdkKeyName[0]);
                            }
                            catch (Pib.Error error) {
                                // empty catch block
                            }
                            if (kdkKey != null) {
                                logger_.log(Level.INFO, "KDK {0} already exists, so directly using it to decrypt the CK", kdkKeyName);
                                DecryptorV2.this.decryptCkAndProcessPendingDecrypts(contentKey, ckData, (Name)kdkKeyName[0], onError);
                                return;
                            }
                        }
                        DecryptorV2.this.fetchKdk(contentKey, kdkPrefix[0], ckData, onError, 3);
                    }
                    catch (Exception ex) {
                        onError.onError(EncryptError.ErrorCode.General, "Error in fetchCk onData: " + ex);
                    }
                }
            }, new OnTimeout(){

                @Override
                public void onTimeout(Interest interest) {
                    contentKey.pendingInterest = 0L;
                    if (nTriesLeft > 1) {
                        DecryptorV2.this.fetchCk(ckName, contentKey, onError, nTriesLeft - 1);
                    } else {
                        onError.onError(EncryptError.ErrorCode.CkRetrievalTimeout, "Retrieval of CK [" + interest.getName().toUri() + "] timed out");
                    }
                }
            }, new OnNetworkNack(){

                @Override
                public void onNetworkNack(Interest interest, NetworkNack networkNack) {
                    contentKey.pendingInterest = 0L;
                    onError.onError(EncryptError.ErrorCode.CkRetrievalFailure, "Retrieval of CK [" + interest.getName().toUri() + "] failed. Got NACK (" + (Object)((Object)networkNack.getReason()) + ")");
                }
            });
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.General, "expressInterest error: " + ex);
        }
    }

    private void fetchKdk(final ContentKey contentKey, final Name kdkPrefix, final Data ckData, final EncryptError.OnError onError, final int nTriesLeft) {
        Name kdkName = new Name(kdkPrefix);
        kdkName.append(EncryptorV2.NAME_COMPONENT_ENCRYPTED_BY).append(this.credentialsKey_.getName());
        logger_.log(Level.INFO, "Fetching KDK {0}", kdkName);
        try {
            contentKey.pendingInterest = this.face_.expressInterest(new Interest(kdkName).setMustBeFresh(true).setCanBePrefix(false), new OnData(){

                @Override
                public void onData(Interest kdkInterest, Data kdkData) {
                    contentKey.pendingInterest = 0L;
                    boolean isOk = DecryptorV2.this.decryptAndImportKdk(kdkData, onError);
                    if (!isOk) {
                        return;
                    }
                    Name kdkKeyName = kdkPrefix.getPrefix(-2).append("KEY").append(kdkPrefix.get(-1));
                    DecryptorV2.this.decryptCkAndProcessPendingDecrypts(contentKey, ckData, kdkKeyName, onError);
                }
            }, new OnTimeout(){

                @Override
                public void onTimeout(Interest interest) {
                    contentKey.pendingInterest = 0L;
                    if (nTriesLeft > 1) {
                        DecryptorV2.this.fetchKdk(contentKey, kdkPrefix, ckData, onError, nTriesLeft - 1);
                    } else {
                        onError.onError(EncryptError.ErrorCode.KdkRetrievalTimeout, "Retrieval of KDK [" + interest.getName().toUri() + "] timed out");
                    }
                }
            }, new OnNetworkNack(){

                @Override
                public void onNetworkNack(Interest interest, NetworkNack networkNack) {
                    contentKey.pendingInterest = 0L;
                    onError.onError(EncryptError.ErrorCode.KdkRetrievalFailure, "Retrieval of KDK [" + interest.getName().toUri() + "] failed. Got NACK (" + (Object)((Object)networkNack.getReason()) + ")");
                }
            });
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.General, "expressInterest error: " + ex);
        }
    }

    private boolean decryptAndImportKdk(Data kdkData, EncryptError.OnError onError) {
        try {
            logger_.log(Level.INFO, "Decrypting and importing KDK {0}", kdkData.getName());
            EncryptedContent encryptedContent = new EncryptedContent();
            encryptedContent.wireDecodeV2(kdkData.getContent());
            SafeBag safeBag = new SafeBag(encryptedContent.getPayload());
            Blob secret = this.keyChain_.getTpm().decrypt(encryptedContent.getPayloadKey().buf(), this.credentialsKey_.getName());
            if (secret.isNull()) {
                onError.onError(EncryptError.ErrorCode.TpmKeyNotFound, "Could not decrypt secret, " + this.credentialsKey_.getName().toUri() + " not found in TPM");
                return false;
            }
            this.internalKeyChain_.importSafeBag(safeBag, secret.buf());
            return true;
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.DecryptionFailure, "Failed to decrypt KDK [" + kdkData.getName().toUri() + "]: " + ex);
            return false;
        }
    }

    private void decryptCkAndProcessPendingDecrypts(ContentKey contentKey, Data ckData, Name kdkKeyName, EncryptError.OnError onError) {
        Blob ckBits;
        logger_.log(Level.INFO, "Decrypting CK data {0}", ckData.getName());
        EncryptedContent content = new EncryptedContent();
        try {
            content.wireDecodeV2(ckData.getContent());
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.InvalidEncryptedFormat, "Error decrypting EncryptedContent: " + ex);
            return;
        }
        try {
            ckBits = this.internalKeyChain_.getTpm().decrypt(content.getPayload().buf(), kdkKeyName);
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.DecryptionFailure, "Error decrypting the CK EncryptedContent " + ex);
            return;
        }
        if (ckBits.isNull()) {
            onError.onError(EncryptError.ErrorCode.TpmKeyNotFound, "Could not decrypt secret, " + kdkKeyName.toUri() + " not found in TPM");
            return;
        }
        contentKey.bits = ckBits;
        contentKey.isRetrieved = true;
        for (ContentKey.PendingDecrypt pendingDecrypt : contentKey.pendingDecrypts) {
            DecryptorV2.doDecrypt(pendingDecrypt.encryptedContent, contentKey.bits, pendingDecrypt.onSuccess, pendingDecrypt.onError);
        }
        contentKey.pendingDecrypts.clear();
    }

    private static void doDecrypt(EncryptedContent content, Blob ckBits, DecryptSuccessCallback onSuccess, EncryptError.OnError onError) {
        Blob plainData;
        if (!content.hasInitialVector()) {
            onError.onError(EncryptError.ErrorCode.MissingRequiredInitialVector, "Expecting Initial Vector in the encrypted content, but it is not present");
            return;
        }
        try {
            Cipher cipher = Cipher.getInstance("AES/CBC/PKCS5PADDING");
            cipher.init(2, (Key)new SecretKeySpec(ckBits.getImmutableArray(), "AES"), new IvParameterSpec(content.getInitialVector().getImmutableArray()));
            plainData = new Blob(cipher.doFinal(content.getPayload().getImmutableArray()), false);
        }
        catch (Exception ex) {
            onError.onError(EncryptError.ErrorCode.DecryptionFailure, "Decryption error in doDecrypt: " + ex);
            return;
        }
        try {
            onSuccess.onSuccess(plainData);
        }
        catch (Throwable exception) {
            logger_.log(Level.SEVERE, "Error in onSuccess", exception);
        }
    }

    private static Name convertKekNameToKdkPrefix(Name kekName, EncryptError.OnError onError) {
        if (kekName.size() < 2 || !kekName.get(-2).equals(EncryptorV2.NAME_COMPONENT_KEK)) {
            onError.onError(EncryptError.ErrorCode.KekInvalidName, "Invalid KEK name [" + kekName.toUri() + "]");
            return null;
        }
        return kekName.getPrefix(-2).append(EncryptorV2.NAME_COMPONENT_KDK).append(kekName.get(-1));
    }

    private static boolean extractKdkInfoFromCkName(Name ckDataName, Name ckName, EncryptError.OnError onError, Name[] kdkPrefix, Name[] kdkIdentityName, Name[] kdkKeyId) {
        if (ckDataName.size() < ckName.size() + 1 || !ckDataName.getPrefix(ckName.size()).equals(ckName) || !ckDataName.get(ckName.size()).equals(EncryptorV2.NAME_COMPONENT_ENCRYPTED_BY)) {
            onError.onError(EncryptError.ErrorCode.CkInvalidName, "Invalid CK name [" + ckDataName.toUri() + "]");
            return false;
        }
        Name kekName = ckDataName.getSubName(ckName.size() + 1);
        kdkPrefix[0] = DecryptorV2.convertKekNameToKdkPrefix(kekName, onError);
        if (kdkPrefix[0] == null) {
            return false;
        }
        kdkIdentityName[0] = kekName.getPrefix(-2);
        kdkKeyId[0] = kekName.getPrefix(-2).append("KEY").append(kekName.get(-1));
        return true;
    }

    public static class ContentKey {
        public boolean isRetrieved = false;
        public Blob bits;
        public long pendingInterest = 0L;
        public ArrayList<PendingDecrypt> pendingDecrypts = new ArrayList();

        public static class PendingDecrypt {
            public EncryptedContent encryptedContent;
            public DecryptSuccessCallback onSuccess;
            public EncryptError.OnError onError;

            public PendingDecrypt(EncryptedContent encryptedContent, DecryptSuccessCallback onSuccess, EncryptError.OnError onError) {
                this.encryptedContent = encryptedContent;
                this.onSuccess = onSuccess;
                this.onError = onError;
            }
        }
    }

    public static interface DecryptSuccessCallback {
        public void onSuccess(Blob var1);
    }
}

