/**
 * Copyright (c) 2012-present 铭软科技(mingsoft.net)
 * 本软件及相关文档文件（以下简称“软件”）的版权归 铭软科技 所有
 * 遵循 铭软科技《服务协议》中的《保密条款》
 */






package net.mingsoft.basic.aop;

import cn.hutool.core.io.FileTypeUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.file.FileNameUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.core.util.ZipUtil;
import net.mingsoft.base.entity.ResultData;
import net.mingsoft.basic.bean.UploadConfigBean;
import net.mingsoft.basic.util.BasicUtil;
import net.mingsoft.config.MSProperties;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 验证zip是否合法的aop
 * 重写basic增加自定义配置
 */
@Component
@Aspect
public class FileVerifyAop extends BaseAop{

    private static final Logger LOGGER = LoggerFactory.getLogger(FileVerifyAop.class);


    /**
     * 切入点
     */
    @Pointcut("execution(* net.mingsoft.basic.action.ManageFileAction.upload(..)) || " +
            "execution(* net.mingsoft.basic.action.ManageFileAction.uploadTemplate(..))")
    public void uploadPointCut(){}


    /**
     * 后台上传文件的时候，将验证zip里的文件
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("uploadPointCut()")
    public Object uploadAop(ProceedingJoinPoint joinPoint) throws Throwable {
        UploadConfigBean bean = super.getType(joinPoint, UploadConfigBean.class);
        if (bean.getFile() == null) {
            return ResultData.build().error("文件不能为空!");
        }
        //检查文件名是否为空
        String uploadFileName = FileNameUtil.cleanInvalid(bean.getFile().getOriginalFilename());
        if (StringUtils.isBlank(uploadFileName) || StringUtils.isBlank(FileNameUtil.mainName(uploadFileName))) {
            return ResultData.build().error("文件名不能为空!");
        }
        if (uploadFileName.lastIndexOf(".") < 0) {
            LOG.info("文件格式错误:{}", uploadFileName);
            return ResultData.build().error("文件名错误");
        }
        InputStream inputStream = null;

        //校验压缩包里的文件
        try {
            inputStream = bean.getFile().getInputStream();
            //文件的真实类型
            String mimeType = BasicUtil.getMimeType(inputStream,uploadFileName);
            if ("zip".equalsIgnoreCase(mimeType) || "zip".equalsIgnoreCase(FileNameUtil.extName(uploadFileName))) {
                checkZip(bean.getFile(), false);
            }
        } catch (Exception e) {
            return ResultData.build().error(e.getMessage());
        }finally {
            if (inputStream != null){
                try {
                    inputStream.close();
                }catch (IOException e){
                    e.printStackTrace();
                }
            }
        }
        return joinPoint.proceed();
    }

    /**
     * web上传文件的时候，将验证zip里的文件
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("execution(* net.mingsoft.basic.action.web.FileAction.upload(..))")
    public Object WebUploadAop(ProceedingJoinPoint joinPoint) throws Throwable {
        UploadConfigBean bean = super.getType(joinPoint, UploadConfigBean.class);
        if (bean.getFile() == null) {
            return ResultData.build().error("文件不能为空!");
        }
        //检查文件名是否为空
        String uploadFileName = FileNameUtil.cleanInvalid(bean.getFile().getOriginalFilename());
        if (StringUtils.isBlank(uploadFileName) || StringUtils.isBlank(FileNameUtil.mainName(uploadFileName))) {
            return ResultData.build().error("文件名不能为空!");
        }
        if (uploadFileName.lastIndexOf(".") < 0) {
            LOG.info("文件格式错误:{}", uploadFileName);
            return ResultData.build().error("文件名错误");
        }
        InputStream inputStream = null;

        //校验压缩包里的文件
        try {
            inputStream = bean.getFile().getInputStream();
            //文件的真实类型
            String mimeType = BasicUtil.getMimeType(inputStream,uploadFileName);
            if ("zip".equalsIgnoreCase(mimeType) || "zip".equalsIgnoreCase(FileNameUtil.extName(uploadFileName))) {
                checkZip(bean.getFile(), true);
            }
        } catch (Exception e) {
            return ResultData.build().error(e.getMessage());
        }finally {
            if (inputStream != null){
                try {
                    inputStream.close();
                }catch (IOException e){
                    e.printStackTrace();
                }
            }
        }
        return joinPoint.proceed();
    }


    /**
     * 检查压缩包
     */
    private void checkZip(MultipartFile multipartFile, boolean isWeb) throws Exception{
        //创建临时解压文件夹
        File tempFilePath = FileUtil.mkdir(FileUtil.getTmpDirPath()+"/Zip"+IdUtil.simpleUUID());
        File zipFile = FileUtil.file(tempFilePath.getAbsolutePath() + "/" + IdUtil.simpleUUID() +".zip");
        InputStream inputStream = multipartFile.getInputStream();
        FileUtils.copyInputStreamToFile(inputStream, zipFile);
        FileInputStream fileInputStream = null;
        try {
            ZipUtil.unzip(zipFile,tempFilePath);
            //获取文件夹下所有文件
            List<File> files = FileUtil.loopFiles(tempFilePath);
            //移除压缩包自身
            files.remove(zipFile);
            //禁止上传的格式
            List<String> deniedList = Arrays.stream(MSProperties.upload.denied.split(",")).map(String::toLowerCase).collect(Collectors.toList());
            for (File file : files) {
                fileInputStream = new FileInputStream(file);
                //文件的类型 受文件幻数影响
                String fileType = FileTypeUtil.getType(file).toLowerCase();
                // 文件后缀名
                String fileSuffixName = FileUtil.getSuffix(file);
                //通过yml的配置检查文件格式是否合法
                if (deniedList.contains(fileSuffixName)) {
                    IOUtils.closeQuietly(fileInputStream);
                    throw new RuntimeException(StrUtil.format("压缩包内文件{}后缀{}禁止上传", file.getName(), fileSuffixName));
                } else if (deniedList.contains(fileType)) {
                    IOUtils.closeQuietly(fileInputStream);
                    throw new RuntimeException(StrUtil.format("压缩包内文件{}的类型{}禁止上传", file.getName(), fileType));

                }

               IOUtils.closeQuietly(fileInputStream);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (fileInputStream != null) {
                IOUtils.closeQuietly(fileInputStream);
            }
            IOUtils.closeQuietly(inputStream);
            FileUtil.del(tempFilePath);
        }

    }

}
