/**
The MIT License (MIT) * Copyright (c) 2016 铭飞科技(mingsoft.net)

 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.mingsoft.basic.action.web;

import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import com.alibaba.druid.sql.visitor.functions.If;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import net.mingsoft.base.filter.DateValueFilter;
import net.mingsoft.base.filter.DoubleValueFilter;
import net.mingsoft.basic.action.BaseAction;
import net.mingsoft.basic.biz.ICategoryBiz;
import net.mingsoft.basic.constant.e.GlobalModelCodelEnum;
import net.mingsoft.basic.constant.e.SessionConstEnum;
import net.mingsoft.basic.entity.AppEntity;
import net.mingsoft.basic.entity.CategoryEntity;
import net.mingsoft.basic.entity.ManagerSessionEntity;

import cn.hutool.core.util.ObjectUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.mingsoft.basic.bean.EUListBean;
import net.mingsoft.basic.util.BasicUtil;
import springfox.documentation.annotations.ApiIgnore;

/**
 * 供前端查询分类信息使用
 * @author 铭飞开发团队
 * @version 
 * 版本号：100-000-000<br/>
 * 创建日期：2012-03-15<br/>
 * 历史修订：<br/>
 */
@Api("供前端查询分类信息使用接口")
@Controller("webCategory")
@RequestMapping("/category")
public class CategoryAction extends BaseAction{
	
	/**
	 * 分类业务处理层注入
	 */
	@Autowired
	private ICategoryBiz categoryBiz;
	
	/**
	 * 根据分类id查找其父分类实体,如果父分类不存在则返回该分类实体
	 * @param categoryId 分类ID
	 * @param request HttpServletRequest对象
	 * @param response HttpServletResponse对象
	 */
	@ApiOperation(value = "根据分类id查找其父分类,如果父分类不存在则返回该分类")
	@ApiImplicitParam(name = "categoryId", value = "分类编号", required = true,paramType="path")
	@GetMapping("/{categoryId}/getParentCategory")
	@ResponseBody
	public void getParentCategory(@PathVariable @ApiIgnore int categoryId,HttpServletRequest request, HttpServletResponse response){
		CategoryEntity category = (CategoryEntity)categoryBiz.getEntity(categoryId);
		if(ObjectUtil.isNotNull(category)){
			CategoryEntity paCategory = (CategoryEntity)categoryBiz.getEntity(category.getCategoryCategoryId());
			if(ObjectUtil.isNull(paCategory)){
				this.outJson(response, JSONObject.toJSONString(category));
			}else{
				this.outJson(response, JSONObject.toJSONString(paCategory));
			}
			
		}
	}
	
	/**
	 * 根据指定分类id查询其子分类
	 * @param categoryId 分类id
	 * @param request HttpServletRequest对象
	 * @param response HttpServletResponse对象
	 */
	@ApiOperation(value = "根据指定分类id查询其子分类")
	@ApiImplicitParam(name = "categoryId", value = "分类编号", required = true,paramType="path")
	@GetMapping("/{categoryId}/queryChildren")
	public void queryChildren(@PathVariable @ApiIgnore int categoryId,HttpServletRequest request, HttpServletResponse response){
		CategoryEntity category = (CategoryEntity) this.categoryBiz.getEntity(categoryId);
		if(category!=null){
			List<CategoryEntity> list = this.categoryBiz.queryChilds(category);
			this.outJson(response, JSONObject.toJSONStringWithDateFormat(list,"yyyy-MM-dd HH:mm:ss"));
		}
	}
	
	/**
	 * 
	 * 分类列表
	 * @param category 分类实体，categoryModelId 模块编号必传
	 * <i>category参数包含字段信息参考：</i><br/>
	 * categoryId 类别ID   <br/>
	 * categoryModelId 模块编号  <br/>
	 * categoryTitle 类别标题<br/>
	 * categorySmallimg 略缩图<br/>
	 * categoryDescription 栏目描述<br/>
	 * <dt><span class="strong">返回</span></dt><br/>
	 * <dd>{ <br/>
	 * categoryId: 类别ID<br/>
	 * categoryTitle: 类别标题<br/>
	 * categorySort: 类别排序<br/>
	 * categoryDatetime: 类别发布时间<br/>
	 * categoryManagerid: 发布用户ID<br/>
	 * categoryModelid: 所属模块ID<br/>
	 * categoryCategoryid: 父类别编号<br/>
	 * categorySmallimg: 略缩图<br/>
	 * categoryAppid: 应用编号<br/>
	 * categoryDescription: 栏目描述<br/>
	 * categoryParentId: 父类型编号<br/>
	 * categoryDictId: 字典对应编号<br/>
	 * createBy: 创建人<br/>
	 * createDate: 创建时间<br/>
	 * updateBy: 更新人<br/>
	 * updateDate: 更新时间<br/>
	 * del: 删除状态<br/>
	 * }</dd><br/>
	 * @param response
	 * @param request
	 * @param model
	 */
	@ApiOperation(value = "根据指定分类id查询其子分类")
	@ApiImplicitParams({
		@ApiImplicitParam(name = "categoryModelId", value = "所属模块编号", required = true,paramType="query"),
		@ApiImplicitParam(name = "categoryDateTime", value = "类别发布时间", required = false,paramType="query"),
		@ApiImplicitParam(name = "categoryManagerId", value = "发布用户编号(发布者编号)", required = false,paramType="query"),
		@ApiImplicitParam(name = "categoryCategoryId", value = "父类别的编号", required = false,paramType="query"),
		@ApiImplicitParam(name = "categoryDictId", value = "字典对应编号", required = false,paramType="query"),
		@ApiImplicitParam(name = "categoryTitle", value = "类别的标题", required = false,paramType="query")
	})
	@GetMapping("/list")
	@ResponseBody
	public void list(@ModelAttribute @ApiIgnore CategoryEntity category,HttpServletResponse response, HttpServletRequest request,ModelMap model) {
		if(category.getCategoryModelId() <= 0){
			this.outJson(response, null,false,getResString("err.empty", this.getResString("category.modelid")));
			return;	
		}
		// 查询指定的appId下的分类
		category.setCategoryAppId(BasicUtil.getAppId());
		BasicUtil.startPage();
		List categoryList = categoryBiz.query(category);
		this.outJson(response, net.mingsoft.base.util.JSONArray.toJSONString(new EUListBean(categoryList,(int)BasicUtil.endPage(categoryList).getTotal()),new DoubleValueFilter(),new DateValueFilter()));
	}
}