package net.megavex.scoreboardlibrary.implementation.packetAdapter.util.reflect;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

public class ReflectUtil {
  // Inspired by
  // https://github.com/dmulloy2/ProtocolLib/blob/02e917cd08cf5b37a52052e22b223272a040e0df/src/main/java/com/comphenix/protocol/reflect/accessors/MethodHandleHelper.java

  private static final Object UNSAFE;
  private static final MethodHandles.Lookup LOOKUP;
  private static final MethodHandle ALLOCATE_INSTANCE_HANDLE;
  private static final MethodType VOID_METHOD_TYPE = MethodType.methodType(void.class);
  private static final MethodType VIRTUAL_FIELD_SETTER = MethodType.methodType(void.class, Object.class, Object.class);

  static {
    MethodHandles.Lookup normalLookup = MethodHandles.lookup();

    try {
      Class<?> unsafeClass = Class.forName("sun.misc.Unsafe");
      Field theUnsafe = unsafeClass.getDeclaredField("theUnsafe");
      theUnsafe.setAccessible(true);
      UNSAFE = theUnsafe.get(null);
      ALLOCATE_INSTANCE_HANDLE = normalLookup.findVirtual(UNSAFE.getClass(), "allocateInstance", MethodType.methodType(Object.class, Class.class));
    } catch (ClassNotFoundException | NoSuchFieldException | IllegalAccessException | NoSuchMethodException e) {
      throw new ExceptionInInitializerError(e);
    }

    MethodHandles.Lookup lookup;
    try {
      MethodHandle staticFieldOffset = normalLookup.findVirtual(UNSAFE.getClass(), "staticFieldOffset", MethodType.methodType(long.class, Field.class));
      MethodHandle staticFieldBase = normalLookup.findVirtual(UNSAFE.getClass(), "staticFieldBase", MethodType.methodType(Object.class, Field.class));
      MethodHandle getObject = normalLookup.findVirtual(UNSAFE.getClass(), "getObject", MethodType.methodType(Object.class, Object.class, long.class));

      Field trustedLookup = MethodHandles.Lookup.class.getDeclaredField("IMPL_LOOKUP");
      long offset = (long) staticFieldOffset.invoke(UNSAFE, trustedLookup);
      Object baseValue = staticFieldBase.invoke(UNSAFE, trustedLookup);

      lookup = (MethodHandles.Lookup) getObject.invoke(UNSAFE, baseValue, offset);
    } catch (Throwable e) {
      e.printStackTrace();
      lookup = normalLookup;
    }

    LOOKUP = lookup;
  }

  private ReflectUtil() {
  }

  public static <T, V> @NotNull FieldAccessor<T, V> findField(@NotNull Class<T> clazz, int index, @NotNull Class<V> valueClass) {
    int i = 0;
    for (Field field : clazz.getDeclaredFields()) {
      if (Modifier.isStatic(field.getModifiers()) || field.getType() != valueClass) {
        continue;
      }

      if (i == index) {
        try {
          MethodHandle setter = LOOKUP.unreflectSetter(field);
          return new FieldAccessor<>(setter.asType(VIRTUAL_FIELD_SETTER));
        } catch (IllegalAccessException e) {
          throw new RuntimeException("failed to unreflect field setter", e);
        }
      }

      i++;
    }

    throw new IllegalStateException(
      "couldn't find field with class " + valueClass.getSimpleName() + " on " + clazz.getSimpleName() + " at index " + index
    );
  }

  public static <T> @NotNull ConstructorAccessor<T> findConstructorOrThrow(@NotNull Class<T> clazz, @NotNull Class<?>... args) {
    ConstructorAccessor<T> accessor = findConstructorOrNull(clazz, args);
    if (accessor == null) {
      throw new IllegalStateException("couldn't get constructor accessor for class " + clazz.getSimpleName());
    }

    return accessor;
  }

  public static <T> @Nullable ConstructorAccessor<T> findConstructorOrNull(@NotNull Class<T> clazz, @NotNull Class<?>... args) {
    try {
      MethodHandle handle = LOOKUP.findConstructor(clazz, MethodType.methodType(void.class, args));
      return new ConstructorAccessor<>(convertToGeneric(handle));
    } catch (NoSuchMethodException | IllegalAccessException e) {
      return null;
    }
  }

  public static <T> @NotNull PacketConstructor<T> getEmptyConstructor(@NotNull Class<T> packetClass) {
    try {
      MethodHandle constructor = LOOKUP.findConstructor(packetClass, VOID_METHOD_TYPE);
      return () -> {
        try {
          // noinspection unchecked
          return (T) constructor.invoke();
        } catch (Throwable e) {
          throw new IllegalStateException("couldn't create packet instance", e);
        }
      };
    } catch (NoSuchMethodException | IllegalAccessException ignored) {
    }

    return () -> {
      try {
        // noinspection unchecked
        return (T) ALLOCATE_INSTANCE_HANDLE.invoke(UNSAFE, packetClass);
      } catch (Throwable e) {
        throw new IllegalStateException("couldn't allocate packet instance using Unsafe", e);
      }
    };
  }

  private static @NotNull MethodHandle convertToGeneric(@NotNull MethodHandle handle) {
    MethodHandle target = handle.asFixedArity();
    MethodType methodType = MethodType.genericMethodType(0, true);
    target = target.asSpreader(Object[].class, handle.type().parameterCount());
    return target.asType(methodType);
  }
}
