/*
 * Copyright (c) 2016 Torsten Krause, Markenwerk GmbH
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package net.markenwerk.utils.json.parser;

/**
 * A {@link JsonHandler} handles the events that are generated by a
 * {@link JsonPushParser} and calculates a result.
 * 
 * @param <Result>
 *            The result type.
 * @author Torsten Krause (tk at markenwerk dot net)
 * @since 1.0.0
 */
public interface JsonHandler<Result> {

	/**
	 * Called when the parsing of a JSON document started.
	 */
	public void onDocumentBegin();

	/**
	 * Called when the parsing of a JSON document ended.
	 */
	public void onDocumentEnd();

	/**
	 * Called when the parsing of a JSON array started. A corresponding call for
	 * the {@link JsonHandler#onArrayEnd() end} will eventually be called.
	 */
	public void onArrayBegin();

	/**
	 * Called when the parsing of a JSON array ended.
	 */
	public void onArrayEnd();

	/**
	 * Called when the parsing of a JSON object started. A corresponding call
	 * for the {@link JsonHandler#onEndObject() end} will eventually be called.
	 */
	public void onObjectBegin();

	/**
	 * Called when the parsing of a JSON object encountered the name for the
	 * next JSON value.
	 * 
	 * @param name
	 *            The name for the next JSON value.
	 */
	public void onName(String name);

	/**
	 * Called when the parsing of a JSON object ended.
	 */
	public void onEndObject();

	/**
	 * Called when the parsing of a JSON object encountered a JSON null.
	 */
	public void onNull();

	/**
	 * Called when the parsing of a JSON object encountered a JSON boolean.
	 * 
	 * @param value
	 *            The value of the JSON boolean.
	 */
	public void onBoolean(boolean value);

	/**
	 * Called when the parsing of a JSON object encountered a JSON number, that
	 * represents an integer value.
	 * 
	 * @param value
	 *            The value of the JSON number.
	 */
	public void onLong(long value);

	/**
	 * Called when the parsing of a JSON object encountered a JSON number, that
	 * represents a decimal value.
	 * 
	 * @param value
	 *            The value of the JSON number.
	 */
	public void onDouble(double value);

	/**
	 * Called when the parsing of a JSON object encountered a JSON string.
	 * 
	 * @param value
	 *            The value of the JSON string.
	 */
	public void onString(String value);

	/**
	 * Returns the calculated result.
	 * 
	 * @return The calculated result.
	 */
	public Result getResult();

}
