package LinkFuture.Core.MemoryManager.StaticMemoryCache;

import LinkFuture.Core.MemoryManager.Meta.MemoryInfo;
import LinkFuture.Core.MemoryManager.Meta.MemoryMetaInfo;
import LinkFuture.Core.MemoryManager.Meta.MemoryType;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Init.Debugger;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static LinkFuture.Core.MemoryManager.Meta.MemoryType.Absolute;
import static LinkFuture.Core.MemoryManager.Meta.MemoryType.Never;

/**
 * User: Cyokin Zhang
 * Date: 10/12/13
 * Time: 2:19 PM
 */
public class StaticMemoryHelper {
    private static final Map<String,StaticMemoryInfo> cachedObject = new ConcurrentHashMap<>();
    private static long lastClearTime = 0;
    private static final long clearUpInterval= 5*60*1000;

    public static <T> MemoryInfo<T> addNeverExpiredMemoryCache(String key, Operation operation) throws Exception {
        MemoryMetaInfo meta = new MemoryMetaInfo();
        meta.CacheType = Never;
        meta.Key = key;
        meta.Enable = true;
        meta.Action =operation;
        return addMemoryCache(meta);
    }
    public static <T> MemoryInfo<T> addMemoryCache(MemoryMetaInfo meta) throws Exception {
        //if not enabled, just return the value
        if(!meta.Enable)
        {
            Debugger.LogFactory.trace("this cache setting({}) has been disabled, call it directly ", meta.Key);
            MemoryInfo<T> output = new MemoryInfo<>();
            output.meta = meta;
            output.cachedObject = (T)meta.Action.call();
            return output;
        }
        if(!cachedObject.containsKey(meta.getUniqueKey()))
        {
            Debugger.LogFactory.trace("add memory into local({}) for key {}",meta.CacheType,meta.Key);
            runAction(meta,System.currentTimeMillis());
        }
        //noinspection unchecked
        return getMemory(meta);
    }
    public static void clear(MemoryMetaInfo meta){
        if(cachedObject.containsKey(meta.getUniqueKey()))
        {
            Debugger.LogFactory.trace("removing {}", meta.getUniqueKey());
            cachedObject.remove(meta.getUniqueKey());
        }
    }
    public static void refresh(){
        Debugger.LogFactory.warn("refresh memory");
        for (StaticMemoryInfo memory :cachedObject.values())
        {
            if(memory.meta.CacheType== MemoryType.Absolute && memory.hasExpired())
            {
                Debugger.LogFactory.trace("this memory({}) has been expired", memory.meta.Key);
                clear(memory.meta);
            }
        }
    }
    private static synchronized void refreshByTime(){
        new Thread(() -> {
            long currentTime = System.currentTimeMillis();
            if(currentTime > lastClearTime + clearUpInterval){
                lastClearTime = currentTime;
                refresh();
            }
        }).start();
    }
    /*
    * Get cached object from memory, it will return null if not exist.
    * */
    public static <T> MemoryInfo<T> getMemory(MemoryMetaInfo meta)
    {
        long currentTime = System.currentTimeMillis();
        String key = meta.getUniqueKey();
        if(cachedObject.containsKey(key))
        {
            StaticMemoryInfo cacheInfo =  cachedObject.get(key);
            if(!cacheInfo.meta.Enable)
            {
                cachedObject.remove(key);
                return null;
            }
            switch (cacheInfo.meta.CacheType)
            {
                case Never:
                    Debugger.LogFactory.trace("load memory from local({}) for key {}",meta.CacheType,meta.Key);
                    return cacheInfo.getMemoryInfo((T)cacheInfo.cachedObject);
                case Absolute:
                    if(!cacheInfo.hasExpired())
                    {
                        Debugger.LogFactory.trace("load memory from local({}) for key {}",meta.CacheType,meta.Key);
                        return cacheInfo.getMemoryInfo((T)cacheInfo.cachedObject);
                    }
                    if(cacheInfo.meta.AutoRefresh)
                    {
                        try {
                            Debugger.LogFactory.trace("renewing memory from local({}) for key {}",meta.CacheType,meta.Key);
                            runAction(cacheInfo.meta,currentTime);
                            return cacheInfo.getMemoryInfo((T)cacheInfo.cachedObject);
                        } catch (Exception e) {
                            throw new RuntimeException(e.getMessage());
                        }
                    }
                    //expired? remove it
                    cachedObject.remove(key);
                    return null;
            }
        }
        refreshByTime();
        return null;
    }
    public static <T> MemoryInfo<T> getMemory(String key)
    {
        MemoryMetaInfo meta = new MemoryMetaInfo();
        meta.Key = key;
        return getMemory(meta);
    }

    private static void runAction(MemoryMetaInfo meta, long currentTime) throws Exception {
        StaticMemoryInfo memory = new StaticMemoryInfo();
        memory.cachedObject = meta.Action.call();
        memory.cachedTime = currentTime;
        memory.meta = meta;
        if(meta.CacheType == Absolute)
        {
            memory.expiredTime = currentTime +  meta.getTimeSpan().getTimeStamp();
        }
        cachedObject.put(meta.getUniqueKey(),memory);
        refreshByTime();
    }

    public static String getMemoryStatus(){
        StringBuilder sb = new StringBuilder();
        sb.append("<MemoryController>");
        for (StaticMemoryInfo memory :cachedObject.values())
        {
            sb.append(memory.toString());
        }
        sb.append("</MemoryController>");
        return sb.toString();
    }

    public static void destroy(){
        Debugger.LogFactory.trace("destroy all");
        cachedObject.clear();
    }
}
