package LinkFuture.Core.ContentManager.ContentResource;

import LinkFuture.Core.ContentManager.ContentController;
import LinkFuture.Core.ContentManager.ContentEvent.IContentEvent;
import LinkFuture.Core.ContentManager.ContentSelector.ContentSelectorController;
import LinkFuture.Core.ContentManager.Model.*;
import LinkFuture.Core.MemoryManager.MemoryFactory;
import LinkFuture.Core.MemoryManager.Meta.MemoryInfo;
import LinkFuture.Core.MemoryManager.Meta.MemoryMetaInfo;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Init.Config;
import LinkFuture.Init.Debugger;
import LinkFuture.Init.Extensions.StringExtension;


/**
 * User: Cyokin Zhang
 * Date: 10/12/13
 * Time: 8:14 PM
 */
public abstract class ContentBaseResource implements IContentResource
 {
     public ResourceInfo CurrentResource;
     public AddInfo CurrentAdd;
     private String resourceIdentity;
     public ContentBaseResource(ResourceRunningInfo runningInfo)
     {
         this.CurrentResource = runningInfo.resource;
         this.CurrentAdd = runningInfo.add;
     }

     public abstract ContentResultInfo RetrieveResource(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception;

     protected ContentResultType getResponseType( ContentParameterCollectionInfo parameters)
     {
         if(parameters.containsKey(ContentController.LF_RESPONSE_TYPE))
         {
             ResponseType responseType = (ResponseType)parameters.get(ContentController.LF_RESPONSE_TYPE);
             switch (responseType)
             {

                 case Xml:
                     return ContentResultType.Xml;
                 case Json:
                 case JsonP:
                     return ContentResultType.Json;
                 case Text:
                     return ContentResultType.Text;
                 case Binary:
                     return ContentResultType.Binary;
                 case Html:
                     return ContentResultType.Html;
             }
         }
         //default is json format
         return ContentResultType.Json;
     }

     public String GerResourceIdentity(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception {
         if(StringExtension.IsNullOrEmpty(resourceIdentity))
         {
             this.resourceIdentity = BuildResourceIdentity(content,parameters).replaceAll("\n", Config.Empty);
             Debugger.LogFactory.trace("Generated ResourceIdentity ContentBaseResource {} => {}",this.CurrentResource.Name, this.resourceIdentity );
         }
         return this.resourceIdentity;
     }

     public abstract String BuildResourceIdentity(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception;
     public abstract void Verify(ResourceInfo currentResource) throws Exception;

     public ContentResultInfo GerResource(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception {
         ContentResultInfo result=null;
         long startTime = System.currentTimeMillis();
         try
         {
             //must build selector before cache logic
             ContentSelectorController.BuilderSelector(this.CurrentAdd, parameters,AddCycleType.Before);
             MemoryMetaInfo meta = MemoryFactory.findCacheMetaInfo(this.CurrentResource.CacheSettingKey);
             Object[] passArgs= new Object[] {content,parameters,this.CurrentAdd};
             if(meta==null || !meta.Enable)
             {
                 result = Callback(passArgs);
             }
             else
             {
                 meta.Key  =this.GerResourceIdentity(content, parameters);
                 meta.Action = new Operation<Object>(passArgs) {
                     @Override
                     public Object call() throws Exception {
                         return Callback(this.params);
                     }
                 };
                 MemoryInfo<ContentResultInfo> memory = MemoryFactory.set(meta);
                 if(memory!=null)
                 {
                     result.CachedTime = memory.cachedTime;
                     result.ExpiredTime = memory.expiredTime;
                     result = memory.cachedObject;
                 }
             }

         }
         catch (Exception e)
         {
             result = new ContentResultInfo();
             result.Name  = this.CurrentResource.Name;
             result.Success = false;
             result.Error = e;
             OnCompletedResource(this.CurrentAdd,parameters,result);
         }
         finally {
             result.Resource = this.CurrentResource;
             result.Add = this.CurrentAdd;
             if(result!=null)
             {
                 result.LoadTimeInMs = System.currentTimeMillis() - startTime;
                 if(result.Success)
                 {
                     return result;
                 }
                 //fail
                 if(this.CurrentAdd!=null && this.CurrentAdd.IgnoreException)
                 {
                     return result;
                 }
                 else
                 {
                     throw result.Error;
                 }
             }
             throw new RuntimeException(String.format("Unexpected error on retrieve resource: %s",this.CurrentResource.Name));
         }
     }

     public ContentResultInfo Callback(Object... args) throws Exception {
         ContentItemInfo content = (ContentItemInfo)args[0];
         ContentParameterCollectionInfo parameters = (ContentParameterCollectionInfo)args[1];
         AddInfo add = (AddInfo)args[2];
         ContentResultInfo result=null;
         Debugger.LogFactory.trace("Retrieve Resource: {}", this.CurrentResource.Name);
         result = RetrieveResource(content, parameters);
         result.Name = this.CurrentResource.Name;
         OnCompletedResource(add,parameters,result);
         if(result.Success)
         {
             return result;
         }
         else
         {
             //so no cache
             throw result.Error;
         }
     }

     private void OnCompletedResource(AddInfo add,ContentParameterCollectionInfo parameters,ContentResultInfo result) throws Exception {
         if(add!=null)
         {
             IContentEvent event = add.getOnCompletedEvent();
             if(event!=null)
             {
                 Object[] context = new Object[]{result,this};
                 event.OnTrigger(add,context,parameters);
             }
         }
     }

     @Override
     public boolean MatchCondition(ContentItemInfo content, ContentParameterCollectionInfo parameters) {
         if(content!=null)
         {
             if(this.CurrentAdd!=null && !StringExtension.IsNullOrEmpty(this.CurrentAdd.Condition))
             {
                 Boolean matched = LinkFuture.Core.Utility.stringExpression(parameters, this.CurrentAdd.Condition);
                 Debugger.LogFactory.trace("Content {} match condition:{} = {},",content.Name, this.CurrentAdd.Condition,matched);
                 return matched;
             }
         }
         return true;
     }
 }