package LinkFuture.Core.DBHelper;
/**
 * User: Cyokin Zhang
 * Date: 10/1/13
 * Time: 10:48 AM
 */

import LinkFuture.Core.DBHelper.Model.*;
import LinkFuture.Core.MemoryManager.StaticMemoryCache.StaticMemoryCacheHelper;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Core.Utility;
import LinkFuture.Init.Config;
import LinkFuture.Init.Debugger;
import LinkFuture.Init.Extensions.DateExtension;
import LinkFuture.Init.Extensions.StringExtension;
import org.json.JSONArray;
import org.json.JSONObject;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class DBHelper implements AutoCloseable {
    //public static String DefaultDriverClassName =  "com.mysql.jdbc.Driver";
//    public static String DefaultDriverClassName =  "com.microsoft.sqlserver.jdbc.SQLServerDriver";
//    static {
//        try {
//            Class.forName(DefaultDriverClassName);
//        } catch (ClassNotFoundException e) {
//            throw new RuntimeException(e);
//        }
//    }
    public static final List<Integer> comparableTypes =
            Arrays.asList(Types.BIGINT,Types.INTEGER,Types.SMALLINT,Types.TINYINT,Types.DECIMAL,Types.FLOAT,Types.DOUBLE,Types.NUMERIC,Types.REAL,Types.TIME,Types.TIMESTAMP,Types.DATE);
    public static final List<Integer> likeableTypes =
            Arrays.asList(Types.VARCHAR,Types.NVARCHAR);
    public static final List<Integer> numberTypes = Arrays.asList(Types.BIGINT,Types.INTEGER,Types.SMALLINT,Types.TINYINT,Types.DECIMAL,Types.FLOAT,Types.DOUBLE,Types.NUMERIC,Types.REAL);
    public static final List<Integer> stringTypes = Arrays.asList(Types.VARCHAR,Types.LONGVARCHAR,Types.NVARCHAR, Types.LONGNVARCHAR,Types.BLOB);
    public static Connection getConnection(String connectionString) throws SQLException, NamingException, ClassNotFoundException {
        if(StringExtension.IsNullOrEmpty(connectionString))
        {
            throw new IllegalArgumentException("Please specific connection string");
        }
        Debugger.LogFactory.trace("Connect to {} ",connectionString);
        if(connectionString.startsWith("java:/"))
        {
            InitialContext cxt = new InitialContext();
            DataSource ds = (DataSource) cxt.lookup(connectionString);
            return ds.getConnection();
        }
        else
        {
            return DriverManager.getConnection(connectionString);
        }
    }

    private Connection conn;
    @SuppressWarnings("unused")
    public Connection getCurrentConnection(){
        return conn;
    }

    private static final Map<String, SPInfo> CachedSPMeta = new ConcurrentHashMap<>();
    private static final Map<String, TableInfo> CachedTableMeta = new ConcurrentHashMap<>();
    private static final Map<String, ConcurrentHashMap<String,TableInfo>> CachedAllTableMeta = new ConcurrentHashMap<>();

    public Map<String, SPParameterInfo> inputParameterList = new LinkedHashMap<>();
    public String dbUrl;
    public SPInfo SPMetaInfo = null;

    public Statement statement;
    private boolean autoClose = true;
    public DBTypeInfo DBType = null;

    //region DBHelper Structure
    public DBHelper(Connection conn) throws SQLException, ClassNotFoundException {
        //don't not close connection if connect come from others
        this.autoClose = false;
        this.init(conn);
    }
    public DBHelper(String connectionString) throws IOException, SQLException, ClassNotFoundException, NamingException {
        this.init(getConnection(connectionString));
    }
    //endregion

    private void init(Connection conn) throws SQLException {
        this.conn = conn;
        this.dbUrl = DBConnectionInfo.Parser(conn.getMetaData().getURL()).Url;
        this.DBType = Utility.enumParser(DBTypeInfo.class, this.conn.getMetaData().getDatabaseProductName());
    }

    //region Parameter
    public void addParameter(String name, Object value)
    {
        addParameter(name,value,null);
    }
    public void addOutParameter(String name)
    {
        addParameter(name,null, ParameterTypeInfo.procedureColumnOut);
    }
    private void addParameter(String name, Object value, ParameterTypeInfo type)
    {
        if(type==null) type = ParameterTypeInfo.procedureColumnIn;
        SPParameterInfo parameter = new SPParameterInfo();
        parameter.parameterName = name;
        parameter.parameterValue = value;
        parameter.parameterType = type;
        inputParameterList.put(buildParamKey(name), parameter);
    }
    //endregion

    //region Execute

    //region Execute SP
    public ArrayList<ArrayList<?>> executeSP(String commandText,Class<?>... outputType) throws Exception {
        this.setCommand(commandText,CommandTypeInfo.StoredProcedure);
        boolean hadResults =this.Execute(CommandTypeInfo.StoredProcedure);
        if(hadResults)
        {
            return new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
        }
        return null;
    }
    public<T> ArrayList<T> executeSP(String commandText,Class<T> outputType) throws Exception {
        this.setCommand(commandText,CommandTypeInfo.StoredProcedure);
        boolean hadResults =this.Execute(CommandTypeInfo.StoredProcedure);
        if(hadResults)
        {
            ArrayList<ArrayList<?>> results = new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
            if(results.size() >0)
            {
                //noinspection unchecked
                return (ArrayList<T>)results.get(0);
            }
        }
        return null;
    }
    public String executeToXml(String commandText,CommandTypeInfo commandType) throws Exception {
        this.setCommand(commandText,commandType);
        boolean hadResults =this.Execute(commandType);
        if(!hadResults)
        {
            return null;
        }
        return new DBXmlReader(this.statement,this.SPMetaInfo).Read();
    }
    public JSONObject executeToJson(String commandText,CommandTypeInfo commandType) throws Exception {
        this.setCommand(commandText,commandType);
        boolean hadResults =this.Execute(commandType);
        if(!hadResults)
        {
            return null;
        }
        return new DBJsonReader(this.statement,this.SPMetaInfo).Read();
    }
    //endregion

    //region Execute TSQL
    @SuppressWarnings("unused")
    public ArrayList<ArrayList<?>> executeSQL(String sql,Class<?>... outputType) throws Exception {
        this.setCommand(sql,CommandTypeInfo.TSQL);
        boolean hadResults =this.Execute(CommandTypeInfo.TSQL);
        if(hadResults)
        {
            return new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
        }
        return null;
    }
    public<T> ArrayList<T> executeSQL(String sql,Class<T> outputType) throws Exception {
        this.setCommand(sql,CommandTypeInfo.TSQL);
        boolean hadResults =this.Execute(CommandTypeInfo.TSQL);
        if(hadResults)
        {
            ArrayList<ArrayList<?>> results = new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
            if(results.size() >0)
            {
                //noinspection unchecked
                return (ArrayList<T>)results.get(0);
            }
        }
        return null;

    }
    public int executeSQL(String sql) throws Exception {
        this.setCommand(sql, CommandTypeInfo.TSQL);
        return this.TSQLExecuteUpdate();
    }
    //endregion

    public List<Object> insert(String commandText) throws Exception {
        this.setCommand(commandText,CommandTypeInfo.TSQL);
        return this.TSQLExecuteInsert();
    }

    //endregion

    public void close(){
        try {
            if(this.statement!=null && !this.statement.isClosed())
            {
                this.statement.close();
                this.statement = null;
            }
            if(this.autoClose && this.conn!=null && !this.conn.isClosed())
            {
                if(DBType==DBTypeInfo.PostgreSQL)
                {
                    this.conn.setAutoCommit(true);
                }
                this.conn.close();
            }
        } catch (SQLException e) {
            Debugger.LogFactory.error("Close DB error", e);
        }
    }

    //region Private Method
    private void setCommand(String commandText,CommandTypeInfo commandType) throws SQLException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        Debugger.LogFactory.trace("setCommand:" + commandText);
        if(commandType == CommandTypeInfo.StoredProcedure)
        {
            this.SPMetaInfo = this.findSPInfo(commandText);
        }
        else
        {
            this.SPMetaInfo =  this.findTSQLInfo(commandText);
        }
    }
    private boolean Execute(CommandTypeInfo commandType) throws Exception {
        switch (commandType)
        {
            case TSQL:
                return TSQLExecute();
            case StoredProcedure:
                default:
                return SPExecute();
        }
    }
    private boolean SPExecute() throws Exception {
        CallableStatement callableStatement = this.conn.prepareCall(this.SPMetaInfo.sqlCall);
        this.statement =callableStatement;
        //fill
        if(DBType==DBTypeInfo.PostgreSQL)
        {
            this.conn.setAutoCommit(false);
        }
        this.fillParameter(this.SPMetaInfo, callableStatement);
        //exec
        Debugger.LogFactory.trace("SPExecute:{}",this.statement.toString());
        return callableStatement.execute();
    }
    private boolean TSQLExecute() throws Exception {
        PreparedStatement preparedStatement;
        if(this.SPMetaInfo.sqlCall.toUpperCase().indexOf("INSERT ") >0)
        {
            preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall,Statement.RETURN_GENERATED_KEYS);
        }
        else
        {
            preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall);
        }

        this.statement =preparedStatement;
        //fill
        this.fillParameter(this.SPMetaInfo,preparedStatement);
        //exec
        Debugger.LogFactory.trace("TSQLExecute:{}",this.statement.toString());
        return preparedStatement.execute();
    }
    private int TSQLExecuteUpdate() throws Exception {
        PreparedStatement preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall);

        this.statement =preparedStatement;
        //fill
        this.fillParameter(this.SPMetaInfo,preparedStatement);
        //exec
        Debugger.LogFactory.trace("TSQLExecuteUpdate:{}",this.statement.toString());
        return preparedStatement.executeUpdate();
    }
    private List<Object> TSQLExecuteInsert() throws Exception {
        PreparedStatement preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall,Statement.RETURN_GENERATED_KEYS);

        this.statement =preparedStatement;
        //fill
        this.fillParameter(this.SPMetaInfo,preparedStatement);
        //exec
        Debugger.LogFactory.trace("TSQLExecuteInsert:{}",this.statement.toString());
        preparedStatement.executeUpdate();
        ResultSet rs = this.statement.getGeneratedKeys();
        List<Object> identityList = new ArrayList<>();
        while (rs.next()){
            identityList.add(rs.getObject(1));
        }
        if(identityList.size()==0)
        {
            return null;
        }
        return identityList;
    }
    private void fillParameter(SPInfo spMeta, PreparedStatement stmt) throws Exception {
        if(spMeta.parameterList.size()>0)
        {
            boolean indexModel = spMeta.parameterList.get(0).parameterName.equalsIgnoreCase("?");
            int length =spMeta.parameterList.size();
            @SuppressWarnings("unchecked")
            List keys =  new ArrayList(inputParameterList.keySet());
            for (int i=0;i<length;i++)
            {
                if(indexModel)
                {
                    String paramKey = ((String)keys.get(i));
                    SPParameterInfo passedParam = inputParameterList.get(buildParamKey(paramKey));
                    stmt.setObject(i + 1, passedParam.parameterValue);
                }
                else
                {
                    SPParameterInfo param = spMeta.parameterList.get(i);
                    SPParameterInfo passedParam = inputParameterList.get(buildParamKey(param.parameterName));
                    if(passedParam==null)
                    {
                        throw new IllegalArgumentException("Missing value for param:" + param.parameterName);
                    }
                    setObject(stmt,i + 1, passedParam.parameterValue, param);
                }
            }
        }
    }
    public Object toSQLArray(SPParameterInfo param,Object value) throws Exception {
        String item = value.toString();
        String elementType = StringExtension.TrimEnd(param.sqlTypeName, "[]");
        TableInfo typeInfo = findTypeInfo(elementType);
        if(!item.startsWith("{"))
        {
            if(item.startsWith("["))
            {
                JSONArray array = new JSONArray(item);
                Object[] list = new Object[array.length()];
                //composite type?
                if(typeInfo.columnList.size()>0)
                {
                    for (int i = 0 ; i < array.length(); i++) {
                        list[i] =toSQLStruct(elementType, array.getJSONObject(i));
                    }
                }
                else
                {
                    for (int i = 0 ; i < array.length(); i++) {
                        list[i] = array.get(i);
                    }
                }
                return this.conn.createArrayOf(elementType,list);
            }
        }
        return value;
    }
    public Object toSQLStruct(String sqlTypeName,Object value) throws Exception {
        //return JsonController.JSONtoSQLStruct(value,findTypeInfo(sqlTypeName));
        TableInfo structInfo = findTypeInfo(sqlTypeName);
        String item = value.toString();
        //if it is json format
        if(item.startsWith("{"))
        {
            List<String> valueList = new ArrayList<>();
            JSONObject json = new JSONObject(item);
            Set keyList = json.keySet();
            for (ColumnInfo column:structInfo.columnList)
            {
                if(keyList.contains(column.columnName))
                {
                    SPParameterInfo childParam = new SPParameterInfo();
                    childParam.sqlTypes = column.sqlType;
                    childParam.sqlTypeName = column.sqlType == Types.ARRAY? column.getArrayElementTypeName():column.sqlTypeName;
                    String output = buildPassedValue(json.get(column.columnName),childParam).toString();
                    //encoding " to double "
                    if(stringTypes.contains(childParam.sqlTypes))
                    {
                        output = output.replace("\"","\"\"");
                    }
                    valueList.add("\"" + output + "\"");
                }
                else
                {
                    valueList.add(Config.Empty);
                }
            }
            return "(" + StringExtension.Join(valueList,",") + ")";
        }
        return value;
    }
    private Object buildPassedValue(Object passValue,SPParameterInfo param) throws Exception {
        if(param.sqlTypes == Types.BIT)
        {
            String stringPassValue = passValue.toString();
            if(stringPassValue.equalsIgnoreCase("true") || stringPassValue.equalsIgnoreCase("t")) return 1;
            if(stringPassValue.equalsIgnoreCase("false") || stringPassValue.equalsIgnoreCase("f")) return 0;
            return passValue;
        }
        if(param.sqlTypes == Types.ARRAY)
        {
            return toSQLArray(param,passValue);
        }
        if(param.sqlTypes == Types.STRUCT)
        {
            return toSQLStruct(param.sqlTypeName,passValue);
        }
        if((param.sqlTypes == Types.TIMESTAMP || param.sqlTypes == Types.DATE) && (passValue instanceof String || passValue instanceof Number))
        {
            return DateExtension.Parse(passValue.toString());
        }
        return passValue;
    }
    private void setObject(PreparedStatement stmt,int index, Object passValue,SPParameterInfo param) throws Exception {
        if(passValue==null)
        {
            stmt.setNull(index,param.sqlTypes);
            return;
        }
        if(  param.sqlTypes == Types.ARRAY
           || param.sqlTypes == Types.STRUCT
           || param.sqlTypes == Types.BIT
           )
        {
            stmt.setObject(index, buildPassedValue(passValue, param));
            return;
        }
        if((param.sqlTypes == Types.TIMESTAMP || param.sqlTypes == Types.DATE) && (passValue instanceof String || passValue instanceof Number))
        {
            if(DBType==DBTypeInfo.MySql)
            {
                stmt.setObject(index, buildPassedValue(passValue, param));
                return;
            }
            else
            {
                stmt.setObject(index, buildPassedValue(passValue, param), param.sqlTypes);
                return;
            }
        }
        stmt.setObject(index, passValue, param.sqlTypes);
    }

    protected String getUniqueSqlParameterName(String paramName){
        String paramKey = buildParamKey(paramName);
        while (inputParameterList.containsKey(paramKey))
        {
            return getUniqueSqlParameterName(paramName + "_" + System.currentTimeMillis());
        }
        return paramName;
    }
    private void fillParameter(SPInfo spMeta, CallableStatement stmt) throws Exception {
        if(spMeta.parameterList.size()>0)
        {
            for(int i=0;i<spMeta.parameterList.size();i++){
                SPParameterInfo param =  spMeta.parameterList.get(i);
                String paramKey = buildParamKey(param.parameterName) ;
                String passParamName=param.parameterName;

                if(DBType==DBTypeInfo.PostgreSQL && param.isRefcursor() && param.parameterType ==ParameterTypeInfo.procedureColumnIn)
                {
                    setObject(stmt,i + 1, paramKey, param);
                    continue;
                }

                if(inputParameterList.containsKey(paramKey) && param.parameterType ==ParameterTypeInfo.procedureColumnIn)
                {
                    //Types.TIMESTAMP
                    Object passValue = inputParameterList.get(paramKey).parameterValue;
                    if(DBType==DBTypeInfo.PostgreSQL)
                    {
                        setObject(stmt,i + 1, passValue,param);
                    } else
                    {
                        stmt.setObject(passParamName, buildPassedValue(passValue,param));
                    }
                }
                else
                {
                    //append null value as default
                    if(param.parameterType ==ParameterTypeInfo.procedureColumnIn)
                    {
                        if(DBType==DBTypeInfo.PostgreSQL)
                        {
                            stmt.setNull(i+1,param.sqlTypes);
                        }
                        else
                        {
                            stmt.setNull(passParamName,param.sqlTypes);
                        }
                    }
                }
                if(param.parameterType == ParameterTypeInfo.procedureColumnOut
                        || param.parameterType ==ParameterTypeInfo.procedureColumnInOut
                        || param.parameterType == ParameterTypeInfo.procedureColumnReturn
                        )
                {
                    if(DBType==DBTypeInfo.PostgreSQL)
                    {
                        stmt.registerOutParameter(i+1, param.sqlTypes);
                    }
                    else
                    {
                        stmt.registerOutParameter(passParamName, param.sqlTypes);
                    }
                }
            }
        }
    }
    //endregion

    //region SPMetaInfo
    public SPInfo findSPInfo(String spName) throws SQLException {
    /*
     http://technet.microsoft.com/en-us/library/ms378416.aspx
     AS mySQL not support default in SP, so just ignore it
     PROCEDURE_NAME: The name of the stored procedure.
     COLUMN_NAME:  The name of the column.
     COLUMN_TYPE:  The type of the column. It can be one of the following values:
         procedureColumnUnknown (0)
         procedureColumnIn (1)
         procedureColumnInOut (2)
         procedureColumnOut (4)
         procedureColumnReturn (5)
         procedureColumnResult (3)
    DATA_TYPE: java.sql.Types (smallint)
    TYPE_NAME: The name of the sql data type. (String)
    NULLABLE:  Indicates if the column can contain a null value. It can be one of the following values:
         procedureNoNulls (0)
         procedureNullable (1)
         procedureNullableUnknown (2)
    COLUMN_DEF: The default value of the column.
     */
        String key = (dbUrl + "_SP_" + spName).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read SP Info:{}",key);
            SPInfo spMeta = new SPInfo();
            spMeta.dbName = dbUrl;
            spMeta.spName = spName;
            DatabaseMetaData metaData = this.conn.getMetaData();
            ResultSet rs = metaData.getProcedureColumns(null, null, spName, null);
            while (rs.next()) {

                SPParameterInfo parameter = new SPParameterInfo();
                parameter.parameterName = removeMSSQLPre(rs.getString("COLUMN_NAME"));
                parameter.parameterType = ParameterTypeInfo.convert(rs.getInt("COLUMN_TYPE"));
                if(parameter.parameterType ==ParameterTypeInfo.procedureColumnReturn)
                {
                    spMeta.hasReturn = true;
                    //ignore  PostgreSQL SP return parameter, with use functionX
                    if(DBType==DBTypeInfo.PostgreSQL)
                    {
                         continue;
                    }
                }
                parameter.sqlTypes = rs.getInt("DATA_TYPE");
                parameter.sqlTypeName = rs.getString("TYPE_NAME");
                spMeta.parameterList.add(parameter);
            }
            spMeta.sqlCall =  buildSPCallString(spMeta);
            CachedSPMeta.put(key,spMeta);
        }
        return CachedSPMeta.get(key);
    }

    @SuppressWarnings("unused")
    public void printResult(ResultSet rs) throws SQLException {
        ResultSetMetaData meta = rs.getMetaData();
        while (rs.next()) {
            System.out.println();
            for (int i = 1; i <= meta.getColumnCount(); i++) {
                System.out.print(meta.getColumnName(i) + ":" + rs.getObject(i));
                System.out.print("  ");
            }
        }
    }

    private String removeMSSQLPre(String paramName){
        if(DBType==DBTypeInfo.MicrosoftSQLServer && paramName.startsWith("@"))
        {
            return paramName.substring(1);
        }
        return paramName;
    }
    private static String buildParamKey(String paramName){
        if(paramName.startsWith("@") || paramName.startsWith("$"))
        {
            return paramName.substring(1).toLowerCase();
        }
        return paramName.toLowerCase();
    }
    private String buildSPCallString(SPInfo spMeta) throws SQLException {
        String paramTemplete = "";
        if(spMeta.parameterList.size()>0)
        {
            StringBuilder sb = new StringBuilder();
            sb.append("(");
            for (SPParameterInfo ignored :spMeta.parameterList){
                if(!ignored.isIgnore())
                {
                    sb.append("?,");
                }
            }
            //delete last comma ","
            sb.deleteCharAt(sb.length()-1);
            sb.append(")");
            paramTemplete = sb.toString();
        }
        if(this.DBType == DBTypeInfo.PostgreSQL)
        {
            return String.format("SELECT %s%s", spMeta.spName, paramTemplete);
        }
        if(spMeta.hasReturn)
        {
            return String.format("{? = call %s%s}", spMeta.spName, paramTemplete);
        }
        return String.format("{call %s%s}", spMeta.spName, paramTemplete);
    }

    public SPInfo findTSQLInfo(String commandText) throws SQLException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        String key = (dbUrl + "_" + commandText).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read TSQL Info:{}",key);
            SPInfo spMeta = new SPInfo();
            spMeta.dbName = dbUrl;
            spMeta.sqlCall = commandText;
            spMeta.spName = "TSQL";
            Collection<String> list = findTSQLParameters(commandText);
            if(list.size()>0)
            {
                for (String param :list)
                {
                    SPParameterInfo parameter = new SPParameterInfo();
                    String[] paramMeta = param.split("\\|");
                    String paramName = paramMeta[0];
                    String replacement = "?";
                    parameter.parameterName =paramName;
                    parameter.parameterType = ParameterTypeInfo.procedureColumnIn;
                    if(paramMeta.length>1)
                    {
                        String[] parmTypes = paramMeta[1].split("::");
                        if(parmTypes.length>1)
                        {
                            parameter.sqlTypes = Integer.parseInt(parmTypes[0]);
                            parameter.sqlTypeName = parmTypes[1];
                            replacement = "?::" + parameter.sqlTypeName;
                        }
                        else
                        {
                            parameter.sqlTypes = Integer.parseInt(paramMeta[1]);

                        }
                    }
                    else
                    {
                        parameter.sqlTypes = Types.VARCHAR;
                    }
                    spMeta.parameterList.add(parameter);
                    //reset TSQL string, as this.conn.prepareStatement don't understand ":Param"
                    spMeta.sqlCall = spMeta.sqlCall.replace("$"+param,replacement);
                }
            }
            else
            {
                PreparedStatement preparedStatement= this.conn.prepareStatement(spMeta.sqlCall);
                ParameterMetaData metaList = preparedStatement.getParameterMetaData();
                for (int i = 1;i<=metaList.getParameterCount();i++)
                {
                    SPParameterInfo parameter = new SPParameterInfo();
                    parameter.parameterName ="?";
                    parameter.parameterType = ParameterTypeInfo.procedureColumnIn;
                    parameter.sqlTypes = metaList.getParameterType(i);
                    spMeta.parameterList.add(parameter);
                }
            }
            CachedSPMeta.put(key,spMeta);
        }
        return CachedSPMeta.get(key);
    }


    public TableInfo findTypeInfo(String typeName) throws Exception {
        return  findTypeInfo(this.conn,typeName);
    }
    public static TableInfo findTypeInfo(Connection conn, String typeName) throws Exception {
        String key = (DBConnectionInfo.Parser(conn.getMetaData().getURL()).Url + "_TYPE_" + typeName).toLowerCase();
        if(!CachedTableMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read Type Info:{}", key);
            DatabaseMetaData meta = conn.getMetaData();
            String schemaName=null;
            int schemaIndex =typeName.indexOf(".");
            if(schemaIndex>=0)
            {
                schemaName = typeName.substring(0,schemaIndex);
                typeName = typeName.substring(schemaIndex+1);
            }

            ResultSet rsColumn = meta.getColumns(null, schemaName, typeName, null);
            TableInfo output =  new TableInfo();
            output.name = typeName;
            output.columnList = DBBeanReader.read(rsColumn,ColumnInfo.class);
            rsColumn.close();
            CachedTableMeta.put(key,output);
        }
        return CachedTableMeta.get(key);
    }
    public ConcurrentHashMap<String,TableInfo> findAllTableInfo() throws Exception {
        String key = (dbUrl + "_findAllTableInfo_" + this.conn.getCatalog()).toLowerCase();
        if(!CachedAllTableMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read All TABLE Info:{}", key);
            ConcurrentHashMap<String,TableInfo> output = new ConcurrentHashMap<>();
            DatabaseMetaData meta = this.conn.getMetaData();
            String[] types = {"TABLE","VIEW"};
            ResultSet rs = meta.getTables(null, null, "%", types);
            while (rs.next()) {
                String tableName = rs.getString("TABLE_NAME");
                output.put(tableName, findTableInfo(tableName));
            }
            CachedAllTableMeta.put(key,output);
        }
        return CachedAllTableMeta.get(key);
    }
    public ConcurrentHashMap<String,TableInfo> findAllTypeInfo() throws Exception {
        String key = (dbUrl + "_findAllTypeInfo_" + this.conn.getCatalog()).toLowerCase();
        if(!CachedAllTableMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read All Type Info:{}", key);
            ConcurrentHashMap<String,TableInfo> output = new ConcurrentHashMap<>();
            DatabaseMetaData meta = this.conn.getMetaData();
            String[] types = {"TYPE"};
            ResultSet rs = meta.getTables(null, null, "%", types);
            while (rs.next()) {
                String tableName = rs.getString("TABLE_NAME");
                output.put(tableName, findTableInfo(tableName));
            }
            CachedAllTableMeta.put(key,output);
        }
        return CachedAllTableMeta.get(key);
    }
    public TableInfo findTableInfo(String tableName) throws Exception {
        String key = (dbUrl + "_TABLE_" + tableName).toLowerCase();
        if(!CachedTableMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read TABLE Info:{}", key);
            DatabaseMetaData meta = this.conn.getMetaData();
            String schemaName=null;
            int schemaIndex =tableName.indexOf(".");
            if(schemaIndex>=0)
            {
                schemaName = tableName.substring(0,schemaIndex);
                tableName = tableName.substring(schemaIndex+1);
            }
            ResultSet rsColumn = meta.getColumns(null, schemaName, tableName, null);
            //printResult(rsColumn);
            TableInfo output =  new TableInfo();
            output.name = tableName;
            output.columnList = DBBeanReader.read(rsColumn,ColumnInfo.class);
            rsColumn.close();
            ResultSet rsKeys = meta.getPrimaryKeys(null, schemaName, tableName);
            while (rsKeys.next())
            {
                for (ColumnInfo column:output.columnList)
                {
                    if(column.columnName.equalsIgnoreCase(rsKeys.getString(4)))
                    {
                        column.isKey = true;
                        break;
                    }
                }
            }
            rsKeys.close();
            findColumnInfo_ex(output.columnList);
            CachedTableMeta.put(key,output);
        }
        return CachedTableMeta.get(key);
    }
    public HashMap<String,ColumnInfo> getTableColumnList(String tableName) throws Exception {
        TableInfo table = findTableInfo(tableName);
        if(table==null || table.columnList==null || table.columnList.size()==0)
        {
            throw new IllegalArgumentException("Specific table don't have any column:" + tableName);
        }
        HashMap<String,ColumnInfo> columnList = new HashMap<>();
        for (ColumnInfo column:table.columnList)
        {
            columnList.put(column.columnName,column);
        }
        return columnList;
    }
    private void findColumnInfo_ex(List<ColumnInfo> columnList) throws Exception {
        for (ColumnInfo column:columnList)
        {
            if(column.sqlType == Types.STRUCT)
            {
                column.columnList = findTypeInfo(column.sqlTypeName).columnList;
                findColumnInfo_ex(column.columnList);
            }
            if(column.sqlType == Types.ARRAY && column.getArrayElementType()==Types.STRUCT)
            {
                column.columnList = findTypeInfo(StringExtension.TrimStart(column.getArrayElementTypeName(),"_")).columnList;
                findColumnInfo_ex(column.columnList);
            }
        }
    }

    public synchronized static SPInfo findSPMetaInfo(String connectionString,String spName,CommandTypeInfo commandType) throws SQLException, ClassNotFoundException, IOException, NamingException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        //double CachedSPMeta for reduce DB connection
        String key = ("findSPMetaInfo_" + connectionString + "_" + spName).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            try(DBHelper helper = new DBHelper(connectionString)){
                switch (commandType)
                {
                    case TSQL:
                        CachedSPMeta.put(key,helper.findTSQLInfo(spName));
                        break;
                    case StoredProcedure:
                        CachedSPMeta.put(key,helper.findSPInfo(spName));
                        break;
                }

            }
        }
        return  CachedSPMeta.get(key);
    }
    public synchronized static TableInfo findTableInfo(String connectionString,String tableName) throws Exception {
        //double CachedSPMeta for reduce DB connection
        String key = ("findTableInfo_" + connectionString + "_" + tableName).toLowerCase();
        if(!CachedTableMeta.containsKey(key))
        {
            try(DBHelper helper = new DBHelper(connectionString)){
                CachedTableMeta.put(key,helper.findTableInfo(tableName));
            }
        }
        return  CachedTableMeta.get(key);
    }


    /**
     * Search TSQL script find parameter with following format $param, it will find $name and $size
     *     SELECT * from world.city where name=$name limit 0,$size
     * @param content TSQL String
     * @return TSQLParameters
     * @throws NoSuchMethodException
     * @throws IllegalAccessException
     * @throws InvocationTargetException
     */
    public static ArrayList<String> findTSQLParameters(String content) throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        Pattern pattern = Pattern.compile("(\\$[^\",;=\\) ]*)", Pattern.CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(content);
        ArrayList<String> list = new ArrayList<>();
        while (matcher.find()) {
            String param = matcher.group();
            list.add(param.substring(1).trim());
        }
        return list;
    }

    //endregion

    public static TableInfo findResultsColumnInfo(ResultSetMetaData rsMeta) throws SQLException {
        int numberOfColumns = rsMeta.getColumnCount();
        TableInfo tableMeta = new TableInfo();
        for (int i = 1; i <= numberOfColumns; i++) {
            ColumnInfo columnMeta = new ColumnInfo();
            //use label name instead of columnName if specific label.
            String label = rsMeta.getColumnLabel(i);
            columnMeta.columnName = StringExtension.IsNullOrEmpty(label)?rsMeta.getColumnName(i):label;
            columnMeta.columnSize = rsMeta.getColumnDisplaySize(i);
            columnMeta.javaClassName =rsMeta.getColumnClassName(i);
            columnMeta.sqlTypeName = rsMeta.getColumnTypeName(i);
            columnMeta.sqlType = rsMeta.getColumnType(i);
            columnMeta.nullable = rsMeta.isNullable(i)!=ResultSetMetaData.columnNoNulls ;
            columnMeta.autoIncrement = rsMeta.isAutoIncrement(i);
            columnMeta.position = i-1;
            tableMeta.columnList.add(columnMeta);
        }
        return tableMeta;
    }
    public synchronized static HashMap<String,FieldInfo> findClassFieldInfo(Class<?> type) throws Exception {
        String classKey = "$DBHelper$FindClassFieldInfo".concat(type.getName());
        return StaticMemoryCacheHelper.AddNeverExpiredMemoryCache(classKey, new Operation<HashMap<String, FieldInfo>>(type) {
            @Override
            public HashMap<String, FieldInfo> call() throws Exception {
                HashMap<String, FieldInfo> fieldList = new HashMap<>();
                Class<?> runType = (Class<?>) this.params[0];
                Debugger.LogFactory.trace(runType.getName());
                for (Field field : runType.getDeclaredFields()) {
                    //ignore directly;
                    IgnoreDBColumnAttribute ignoreAttribute = field.getAnnotation(IgnoreDBColumnAttribute.class);
                    if (ignoreAttribute != null) {
                        continue;
                    }
                    DBColumnAttribute columnAttribute = field.getAnnotation(DBColumnAttribute.class);
                    FieldInfo fieldMeta = new FieldInfo();
                    fieldMeta.field = field;
                    if (columnAttribute != null) {
                        String columnName = columnAttribute.columnName().length() == 0 ? field.getName() : columnAttribute.columnName();
                        if (columnAttribute.isEntity()) {
                            //find sub field, save into Memory
                            findClassFieldInfo(field.getType());
                        }
                        fieldMeta.columnAttribute = columnAttribute;
                        fieldList.put(columnName.toLowerCase(), fieldMeta);
                    } else {
                        fieldList.put(field.getName().toLowerCase(), fieldMeta);
                    }
                }
                return fieldList;
            }
        });
    }

    public static Map<String,Object>  ReadOutputParameterList(SPInfo spMetaInfo, CallableStatement callableStatement,DBTypeInfo dbType) throws SQLException {
        Map<String,Object> outputParameterList = new HashMap<>();
        if(spMetaInfo.parameterList!=null && spMetaInfo.parameterList.size()>0)
        {
            for (int i=0;i<spMetaInfo.parameterList.size();i++){
                SPParameterInfo param = spMetaInfo.parameterList.get(i);
                if(param.parameterType == ParameterTypeInfo.procedureColumnOut
                        ||   param.parameterType == ParameterTypeInfo.procedureColumnInOut
                        ||   param.parameterType == ParameterTypeInfo.procedureColumnReturn
                        )
                {
                    if(dbType==DBTypeInfo.PostgreSQL)
                    {
                        outputParameterList.put(buildParamKey(param.parameterName), callableStatement.getObject(i + 1));
                    }
                    else
                    {

                        outputParameterList.put(buildParamKey(param.parameterName), callableStatement.getObject(param.parameterName));
                    }
                }
            }
        }
        return outputParameterList;
    }
    public Map<String,Object> getOutputParameterList() throws SQLException {
        return ReadOutputParameterList(this.SPMetaInfo,(CallableStatement)this.statement,this.DBType);
    }

    public static int sqlTypeNameToType(String sqlTypeName) throws IllegalAccessException {
        for (Field field : Types.class.getFields()) {
            if(field.getName().equalsIgnoreCase(sqlTypeName)){
                return (int)field.get(null);
            }
        }
        if(postgreSQLTypeNameToSQLType.containsKey(sqlTypeName))
        {
            return postgreSQLTypeNameToSQLType.get(sqlTypeName);
        }
        return  Types.VARCHAR;
    }

    public static final Map<String,String> postgreSQLTypeNameToJava = new HashMap<String,String>()
    {{
        put("bool", "boolean");
        put("\"char\"", "byte");
        put("int2", "short");
        put("int4", "int");
        put("int8", "long");
        put("float4", "float");
        put("float8", "double");
        put("char", "java.lang.String");
        put("varchar","java.lang.String");
        put("text", "java.lang.String");
        put("name", "java.lang.String");
        put("bytea	", "byte[]");
        put("ate", "java.sql.Date");
        put("time", "java.sql.Time"); // (stored value treated as local time)
        put("timetz", "java.sql.Time");
        put("timestamp", "java.sql.Timestamp"); // (stored value treated as local time)
        put("timestamptz", "java.sql.Timestamp");
    }};
    public static final Map<String,Integer> postgreSQLTypeNameToSQLType = new HashMap<String,Integer>()
    {{
        put("bool", Types.BOOLEAN);
        put("\"char\"", Types.CHAR);
        put("int2", Types.SMALLINT);
        put("int4", Types.INTEGER);
        put("int8", Types.BIGINT);
        put("float4", Types.REAL);
        put("float8", Types.FLOAT);
        put("char", Types.VARCHAR);
        put("varchar", Types.VARCHAR);
        put("text",  Types.LONGVARCHAR);
        put("name", Types.VARCHAR);
        put("bytea	", Types.BINARY);
        put("date", Types.DATE);
        put("time", Types.TIME);
        put("timetz", Types.TIME_WITH_TIMEZONE);
        put("timestamp", Types.TIMESTAMP);
        put("timestamptz", Types.TIMESTAMP_WITH_TIMEZONE);
    }};
    public static final Map<Integer,String> sqlTypeToJava = new HashMap<Integer,String>()
    {{
        put(Types.CHAR, "String");
        put(Types.VARCHAR, "String");
        put(Types.LONGVARCHAR, "String");
        put(Types.NUMERIC, "java.math.BigDecimal");
        put(Types.DECIMAL, "java.math.BigDecimal");
        put(Types.BIT, "Boolean");
        put(Types.TINYINT, "Byte");
        put(Types.SMALLINT, "Short");
        put(Types.INTEGER, "Integer");
        put(Types.BIGINT, "Long");
        put(Types.REAL, "Float");
        put(Types.FLOAT, "Double");
        put(Types.DOUBLE, "Double");
        put(Types.BINARY, "Byte[]");
        put(Types.VARBINARY, "Byte[]");
        put(Types.LONGVARBINARY, "Byte[]");
        put(Types.DATE, "java.util.Date");
        put(Types.TIME, "java.util.Date");
        put(Types.TIMESTAMP, "java.util.Date");
        put(Types.CLOB, "Clob");
        put(Types.BLOB, "Blob");
        put(Types.ARRAY, "Array");
        put(Types.STRUCT, "Struct");
        //put(Types.OTHER, "java.util.UUID");
    }};
}





