package LinkFuture.Core.DBHelper;

import LinkFuture.Core.DBHelper.Model.ColumnInfo;
import LinkFuture.Core.DBHelper.Model.CommandTypeInfo;
import LinkFuture.Core.DBHelper.Model.DBTypeInfo;
import LinkFuture.Init.Config;
import LinkFuture.Init.Extensions.StringExtension;
import LinkFuture.Init.ObjectExtend.NameValuePair;
import org.json.JSONArray;
import org.json.JSONObject;

import javax.naming.NamingException;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Created by Cyokin
 * on 10/28/2015.
 */
@SuppressWarnings("MalformedFormatString")
public class NewGenericDBHelper extends DBHelper {


    //region where key words
    public static final String $WHERE = "$where";

    //region Logical key words
    public static final String $OR = "$or";
    public static final String $IN = "$in";
    public static final String $LIKE = "$like";
    public static final String $SIMILAR = "$similar";
    public static final String $CONTAIN = "$contain";
    public static final String $ANY = "$any";
    public static final String $BETWEEN = "$between";
    //endregion

    //region comparison key words
    public static final String $GT = "$gt";
    public static final String $GTE = "$gte";
    public static final String $LT = "$lt";
    public static final String $LTE = "$lte";
    public static final String $NE = "$ne";
    //endregion
    //endregion

    public static final String $SORT = "$sort";
    public static final String $LIMIT = "$limit";
    public static final String $OFFSET = "$offset";
    public static final String $UPSERT = "$upsert";

    //region aggregate key words
    public static final String $SUM = "$sum";
    public static final String $COUNT = "$count";
    public static final String $MIN = "$min";
    public static final String $MAX = "$max";
    public static final String $AVG = "$avg";
    public static final String $GROUP = "$group";
    public static final List<String> $AGGREGATE_FUNCTIONS =  Arrays.asList($SUM,$COUNT,$MIN,$MAX,$AVG);
    //endregion


    //region arithmetic key words
    public static final String $MULTIPLY = "$multiply";
    public static final String $DIVIDE = "$divide";
    public static final String $PLUS = "$plus";
    public static final String $MINUS = "$minus";
    public static final String $MODULE = "$module";
    public static final List<String> $ARITHMETIC_FUNCTIONS =  Arrays.asList($MULTIPLY,$DIVIDE,$PLUS,$MINUS,$MODULE);
    //endregion


    public NewGenericDBHelper(String connectionString) throws IOException, SQLException, ClassNotFoundException, NamingException {
        super(connectionString);
    }

    private String buildColumnValue(ColumnInfo column,String uniqueColumnName)  {
        if(uniqueColumnName==null) uniqueColumnName = column.columnName;
        if(column.sqlType== Types.STRUCT || column.sqlType== Types.BIT )
        {
            return String.format("$%s|%s::%s", uniqueColumnName, column.sqlType,column.sqlTypeName);
        }
        if(column.sqlType== Types.ARRAY)
        {
            return String.format("$%s|%s::%s[]", uniqueColumnName, column.sqlType,column.getArrayElementTypeName());
        }
        return String.format("$%s|%s", uniqueColumnName, column.sqlType);
    }
    private String buildWhereValue(ColumnInfo column, String subQuery,String uniqueColumnName) throws IllegalAccessException {
        if(column.sqlType==Types.ARRAY)
        {
            return String.format("$%s|%s::%s[] = %s", uniqueColumnName,column.sqlType ,column.getArrayElementTypeName(),column.columnName);
        }
        if(column.sqlType==Types.BIT)
        {
            return String.format("%s = $%s|%s::%s", column.columnName, uniqueColumnName,column.sqlType,column.sqlTypeName);
        }
        if(column.isSqlJsonType())
        {
            return String.format("%s @> $%s|%s::%s", column.columnName,uniqueColumnName,column.sqlType,column.sqlTypeName);
        }
        if(column.sqlType==Types.STRUCT)
        {
            if(StringExtension.IsNullOrEmpty(subQuery))
            {
                return String.format("%s = $%s|%s::%s", column.columnName, uniqueColumnName,column.sqlType,column.sqlTypeName);
            }
            return String.format("(%s).%s = $%s|%s", column.columnName,subQuery, uniqueColumnName,Types.VARCHAR);
        }
        return String.format("%s = $%s|%s", column.columnName, uniqueColumnName,column.sqlType);
    }
    private Object getUpsertValue(Object updatedValue){
        if(updatedValue instanceof JSONObject && ((JSONObject)updatedValue).has($UPSERT))
        {
            return ((JSONObject)updatedValue).get($UPSERT);
        }
        return null;
    }

    private String buildUpdateValue(ColumnInfo column, String subQuery,String uniqueColumnName,boolean jsonUpsertModel ){
        if(column.sqlType==Types.BIT)
        {
            return String.format("%s = $%s|%s::%s", column.columnName, uniqueColumnName,column.sqlType,column.sqlTypeName);
        }
        if(column.isSqlJsonType() && jsonUpsertModel)
        {
            /*
                JSON update
                    --replace
                    SELECT jsonb_merge('{"a":"1","b":"2"}'::JSONB,'{"b":"3"}'::JSONB);
                    --insert
                    SELECT jsonb_merge('{"a":"1","b":"2"}'::JSONB,'{"c":"3"}'::JSONB);

                    CREATE FUNCTION jsonb_merge(JSONB, JSONB)
                      RETURNS JSONB AS $$
                    WITH json_union AS (
                      SELECT * FROM JSONB_EACH($1)
                      UNION ALL
                      SELECT * FROM JSONB_EACH($2)
                    ) SELECT JSON_OBJECT_AGG(key, value)::JSONB
                      FROM json_union
                      WHERE key NOT IN (SELECT key FROM json_union WHERE value ='null');
                    $$ LANGUAGE SQL;
                    * */
            //make sure CREATE FUNCTION jsonb_merge in your DB
            return String.format("%s = jsonb_merge(%s::JSONB, $%s|%s::JSONB)::%s"
                    , column.columnName
                    , column.columnName
                    , uniqueColumnName
                    , column.sqlType
                    ,column.sqlTypeName);
        }
        if(StringExtension.IsNullOrEmpty(subQuery))
        {
            return String.format("%s=%s", column.columnName, buildColumnValue(column,uniqueColumnName));
        }
        else
        {
            return String.format("%s.%s= $%s|%s", column.columnName,subQuery,uniqueColumnName,Types.VARCHAR );
        }
    }


    //region Insert
    public Object insert(String tableName,JSONObject obj) throws Exception {
        JSONArray array = new JSONArray();
        array.put(obj);
        List<Object> list = insert(tableName,array);
        if(list==null || list.size()==0)
        {
            return null;
        }
        return list.get(0);
    }
    public List<Object> insert(String tableName, JSONArray obj) throws Exception {

        return super.insert(buildInsertTSQL(tableName,obj));
    }
    //endregion

    //region Delete
    public int delete(String tableName,NameValuePair...params) throws Exception {
        JSONObject jsonQuery = getQueryJSON(params);
        return delete(tableName,jsonQuery);
    }
    public int delete(String tableName, JSONObject jsonQuery) throws Exception {
        return super.executeSQL(buildDeleteTSQL(tableName,jsonQuery));
    }
    //endregion

    //region update
    public int update(String tableName,NameValuePair...params) throws Exception {
        JSONObject jsonQuery = getQueryJSON(params);
        return update(tableName, jsonQuery);
    }
    public int update(String tableName, JSONObject jsonQuery) throws Exception {
        return super.executeSQL(buildUpdateTSQL(tableName, jsonQuery));
    }
    //endregion

    //region select
    public JSONObject selectToJson(String tableName,NameValuePair...params) throws Exception {
        JSONObject jsonQuery = getQueryJSON(params);
        return selectToJson(tableName, jsonQuery);
    }
    public JSONObject selectToJson(String tableName, JSONObject jsonQuery) throws Exception {
        return super.executeToJson(buildSelectTSQL(tableName, jsonQuery), CommandTypeInfo.TSQL);
    }
    public String executeToXml(String tableName,NameValuePair...params) throws Exception {
        JSONObject jsonQuery = getQueryJSON(params);
        return selectToXml(tableName, jsonQuery);
    }
    public String selectToXml(String tableName, JSONObject jsonQuery) throws Exception {
        return super.executeToXml(buildSelectTSQL(tableName, jsonQuery), CommandTypeInfo.TSQL);
    }
    //endregion

    //region TSQL build
    public String buildDeleteTSQL(String tableName, JSONObject jsonQuery) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        String where = buildWhereTSQL(columnList, jsonQuery);
        if(StringExtension.IsNullOrEmpty(where))
        {
            throw new IllegalArgumentException("Please specific where condition or none of condition is valid for table " + tableName);
        }
        StringBuilder sb = new StringBuilder();
        sb.append("DELETE FROM ");
        sb.append(tableName);
        sb.append(" WHERE ");
        sb.append(where);
        return sb.toString();
    }
    public String buildWhereTSQL(HashMap<String, ColumnInfo> columnList, JSONObject jsonWhere){
        Iterable<String> iterable = () -> jsonWhere.keys();
        StringBuilder sb = new StringBuilder();
        String andString = StreamSupport.stream(iterable.spliterator(), false)
                .filter(key -> {
                List<String> inputColumnName = getInputColumnName(key);
                return columnList.containsKey(inputColumnName.get(0));
            }).map(key -> {
            Object jsonValue = jsonWhere.get(key);
            List<String> columnNameList = getInputColumnName(key);
            ColumnInfo column = columnList.get(columnNameList.get(0));
            String uniqueName = getUniqueSqlParameterName(column.columnName);
            if (jsonValue instanceof JSONObject) {
                JSONObject compare = (JSONObject) jsonValue;
                String operationString = compare.keySet().iterator().next().toString().toLowerCase();
                Object operationValue = compare.get(operationString);
                switch (operationString) {
                    case $GT:
                        if(!comparableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $gt operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s > $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $GTE:
                        if(!comparableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $gte operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s >= $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $LT:
                        if(!comparableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $lt operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s < $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $LTE:
                        if(!comparableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $lte operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s <= $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $NE:
                        if(!comparableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $lte operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s != $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $LIKE:
                        if(!likeableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $like operation");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s LIKE $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $SIMILAR:
                        if(!likeableTypes.contains(column.sqlType))
                        {
                            throw new IllegalArgumentException("Specific type does not support $similar operation");
                        }
                        if(DBType != DBTypeInfo.PostgreSQL)
                        {
                            throw new IllegalArgumentException("$similar operation only support PostgreSQL database");
                        }
                        super.addParameter(uniqueName, operationValue);
                        return String.format("%s SIMILAR TO $%s|%s", column.columnName, uniqueName, column.sqlType);
                    case $IN:
                        JSONArray inList = (JSONArray)operationValue;
                        List<String> inString = new ArrayList<>();
                        for (int i = 0; i < inList.length(); i++) {
                            inString.add("'"+ inList.get(i).toString() + "'");
                        }
                        return String.format("%s in (%s)", column.columnName, StringExtension.Join(inString, ","));
                    case $BETWEEN:
                        JSONArray betweenList = (JSONArray)operationValue;
                        return String.format("%s between %s and %s", column.columnName, betweenList.get(0),betweenList.get(1));
                    case $CONTAIN:
                        if(column.sqlType==Types.ARRAY)
                        {
                            super.addParameter(uniqueName, operationValue);
                            return String.format("%s @> $%s|%s::%s[]", column.columnName,uniqueName,column.sqlType ,column.getArrayElementTypeName());
                        }
                        throw new IllegalArgumentException("Only array type support any operation");
                    case $ANY:
                        if(column.sqlType==Types.ARRAY)
                        {
                            super.addParameter(uniqueName, operationValue);
                            try {
                                return String.format("$%s|%s::%s= ANY(%s)", uniqueName,column.getArrayElementType() ,column.getArrayElementTypeName(),column.columnName);
                            } catch (IllegalAccessException e) {
                                throw new IllegalArgumentException(e.getMessage());
                            }
                        }
                        throw new IllegalArgumentException("Only array type support any operation");
                    default:
                        throw new IllegalArgumentException(String.format("Specific operation %s not support yet",operationString));
                }
            } else {
                try {
                    super.addParameter(uniqueName, jsonValue);
                    return buildWhereValue(column, columnNameList.get(1),uniqueName);
                } catch (IllegalAccessException e) {
                    throw new IllegalArgumentException(e.getMessage());
                }
            }
        }).collect(Collectors.joining(" AND "));
        sb.append(andString);
        if(jsonWhere.has($OR))
        {
            JSONArray orArray = jsonWhere.getJSONArray($OR);
            for (int i = 0; i < orArray.length(); i++) {
                if(i!=0 || andString.length()>0) sb.append(" OR ");
                sb.append("(");
                sb.append(buildWhereTSQL(columnList, orArray.getJSONObject(i)));
                sb.append(")");
            }
        }
        return sb.toString();
    }
    public String buildSortTSQL(HashMap<String, ColumnInfo> columnList, JSONObject jsonSort){
        Iterable<String> iterable = () -> jsonSort.keys();
        String sortQuery = StreamSupport.stream(iterable.spliterator(), false)
                .filter(key -> columnList.containsKey(key))
                .map(key -> {
            String jsonValue = jsonSort.getString(key);
            if (jsonValue.equalsIgnoreCase("DESC")) {
                return String.format("%s DESC", key);
            } else {
                return String.format("%s ASC", key);
            }
        }).collect(Collectors.joining(","));

        return StringExtension.IsNullOrEmpty(sortQuery)?null:" ORDER BY ".concat(sortQuery);
    }
    public String buildInsertTSQL(String tableName, JSONArray obj) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        Supplier<Stream<ColumnInfo>> columnStream = () -> columnList.values().stream();
        StringBuilder sb = new StringBuilder();
        sb.append("INSERT INTO ");
        sb.append(tableName);
        for (int i=0;i<obj.length();i++)
        {
            JSONObject item = obj.getJSONObject(i);
            if(i==0)
            {
                sb.append("(");
                sb.append(columnStream.get()
                        .filter(c -> item.has(c.columnName))
                        .map(c -> c.columnName)
                        .collect(Collectors.joining(",")));
                sb.append(") VALUES");
            }
            sb.append("(");
            sb.append(columnStream.get()
                    .filter(c -> item.has(c.columnName))
                    .map(c ->
                            {
                                String uniqueName = getUniqueSqlParameterName(c.columnName);
                                super.addParameter(uniqueName,item.get(c.columnName));
                                return buildColumnValue(c,uniqueName );
                            }
                    )
                    .collect(Collectors.joining(",")));
            sb.append(")");
            if(i!=obj.length()-1)
            {
                sb.append(",");
            }
        }
        return sb.toString();
    }
    public String buildUpdateTSQL(String tableName, JSONObject jsonQuery) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        if(!jsonQuery.has($WHERE))
        {
            throw new IllegalArgumentException("Please specific where condition or none of condition is valid for table " + tableName);
        }
        String where = buildWhereTSQL(columnList, jsonQuery.getJSONObject($WHERE));
        if(StringExtension.IsNullOrEmpty(where))
        {
            throw new IllegalArgumentException("Please specific where condition or none of condition is valid for table " + tableName);
        }

        Iterable<String> iterable = () -> jsonQuery.keys();
        String setString = StreamSupport.stream(iterable.spliterator(), false).filter(key -> {
            List<String> inputColumnName = getInputColumnName(key);
            return columnList.containsKey(inputColumnName.get(0));
        }).map(key -> {
            List<String> columnNameList = getInputColumnName(key);
            ColumnInfo column = columnList.get(columnNameList.get(0));
            String uniqueName = getUniqueSqlParameterName(column.columnName);
            Object paramValue = jsonQuery.get(key);
            boolean jsonUpsertModel = false;
            if(column.isSqlJsonType())
            {
                Object upsertValue = getUpsertValue(paramValue);
                if(upsertValue!=null)
                {
                    jsonUpsertModel = true;
                    paramValue = upsertValue;
                }
            }
            super.addParameter(uniqueName,paramValue);
            return buildUpdateValue(column,columnNameList.get(1),uniqueName,jsonUpsertModel);
        }).collect(Collectors.joining(","));
        if(StringExtension.IsNullOrEmpty(setString))
        {
            throw new IllegalArgumentException("Please set value for update or none of parameters is valid for table " + tableName);
        }
        StringBuilder sb = new StringBuilder();
        sb.append("UPDATE ");
        sb.append(tableName);
        sb.append(" SET ");
        sb.append(setString);
        sb.append(" WHERE ");
        sb.append(where);
        return sb.toString();
    }

    private String getValidColumn(HashMap<String,ColumnInfo> columnList,JSONArray array,CharSequence delimiter){
        return StreamSupport.stream(array.spliterator(),true)
                .filter(c->columnList.containsKey(c))
                .map(c->(String)c)
                .collect(Collectors.joining(delimiter));
    }
    private String buildArithmeticOperation(HashMap<String,ColumnInfo> columnList, JSONArray array, String delimiter){
        String operationString = StreamSupport.stream(array.spliterator(),true)
                .map(c-> buildOperation(columnList,c)
                )
                .collect(Collectors.joining(delimiter));
        return String.format(" ( %s ) ",operationString);
    }
    private String buildAggregateOperation(HashMap<String,ColumnInfo> columnList,Object aggregateObject,String operation)
    {
        return String.format("%s(%s)",operation, buildOperation(columnList,aggregateObject));
    }
    private String buildOperation(HashMap<String,ColumnInfo> columnList, Object jsonObject){
        if(jsonObject instanceof JSONObject)
        {
            return buildFunction(columnList,(JSONObject)jsonObject);
        }
        if(jsonObject instanceof String)
        {
            if(columnList.containsKey(jsonObject))
            {
                return (String)jsonObject;
            }
            throw new IllegalArgumentException("Invalid column " + jsonObject);
        }
        // only arithmetic support number
        if(jsonObject instanceof Number)
        {
            return jsonObject.toString();
        }
        throw new IllegalArgumentException("Invalid " + jsonObject + " inside multiply function");
    }

    private String buildFieldTSQL(HashMap<String,ColumnInfo> columnList,JSONObject jsonQuery)
    {
        return columnList.values().stream()
                .filter(c -> jsonQuery.has(c.columnName) && jsonQuery.getBoolean(c.columnName))
                .map(c -> c.columnName)
                .collect(Collectors.joining(","));
    }
    private String buildAggregateTSQL(HashMap<String,ColumnInfo> columnList,JSONObject jsonQuery)
    {
        Iterable<String> iterable = () -> jsonQuery.keys();
        return StreamSupport.stream(iterable.spliterator(), false)
                .filter(key -> jsonQuery.get(key) instanceof JSONObject
                        && $AGGREGATE_FUNCTIONS.stream().anyMatch(d->jsonQuery.getJSONObject(key).has(d)))
                .map(fieldName->{
                    JSONObject aggregateJsonObject = jsonQuery.getJSONObject(fieldName);

            return String.format(" %s as %s", buildFunction(columnList, aggregateJsonObject),fieldName);
        }).collect(Collectors.joining(","));
    }
    private String buildArithmeticTSQL(HashMap<String,ColumnInfo> columnList,JSONObject jsonQuery)
    {
        Iterable<String> iterable = () -> jsonQuery.keys();
        return StreamSupport.stream(iterable.spliterator(), false)
                .filter(key -> jsonQuery.get(key) instanceof JSONObject
                        && $ARITHMETIC_FUNCTIONS.stream().anyMatch(d->jsonQuery.getJSONObject(key).has(d)))
                .map(fieldName->{
                    JSONObject aggregateJsonObject = jsonQuery.getJSONObject(fieldName);

                    return String.format(" %s AS %s", buildFunction(columnList, aggregateJsonObject),fieldName);
                }).collect(Collectors.joining(","));
    }
    private String buildFunction(HashMap<String,ColumnInfo> columnList, JSONObject functionJsonObject){
        String functionName = functionJsonObject.keys().next();
        switch (functionName)
        {
            case $MULTIPLY:
                return buildArithmeticOperation(columnList,functionJsonObject.getJSONArray(functionName)," * ");
            case $DIVIDE:
                return buildArithmeticOperation(columnList,functionJsonObject.getJSONArray(functionName)," / ");
            case $PLUS:
                return buildArithmeticOperation(columnList,functionJsonObject.getJSONArray(functionName)," + ");
            case $MINUS:
                return buildArithmeticOperation(columnList,functionJsonObject.getJSONArray(functionName)," - ");
            case $MODULE:
                return buildArithmeticOperation(columnList,functionJsonObject.getJSONArray(functionName)," % ");
            case $SUM:
                return buildAggregateOperation(columnList,functionJsonObject.get(functionName),"SUM");
            case $COUNT:
                return buildAggregateOperation(columnList,functionJsonObject.get(functionName),"COUNT");
            case $MIN:
                return buildAggregateOperation(columnList,functionJsonObject.get(functionName),"MIN");
            case $MAX:
                return buildAggregateOperation(columnList,functionJsonObject.get(functionName),"MAX");
            case $AVG:
                return buildAggregateOperation(columnList,functionJsonObject.get(functionName),"AVG");
        }
        throw new IllegalArgumentException(String.format("specific aggregate function %s does not support yet",functionName));
    }

    public String buildSelectTSQL(String tableName,JSONObject jsonQuery) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        int limit = jsonQuery.has($LIMIT)?jsonQuery.getInt($LIMIT):10;
        int offset =jsonQuery.has($OFFSET)?jsonQuery.getInt($OFFSET):0;
        String groups = jsonQuery.has($GROUP)?getValidColumn(columnList,jsonQuery.getJSONArray($GROUP),","):null;

        StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        sb.append(Config.NewLine);

        //get all column select
        String fieldQuery = buildFieldTSQL(columnList,jsonQuery);
        String aggregateQuery = buildAggregateTSQL(columnList,jsonQuery);
        String arithmeticQuery = buildArithmeticTSQL(columnList,jsonQuery);

        if(StringExtension.IsNullOrEmpty(fieldQuery)
         && StringExtension.IsNullOrEmpty(aggregateQuery)
         && StringExtension.IsNullOrEmpty(arithmeticQuery)
                )
        {
            sb.append("*");
        }
        else
        {
            sb.append(Stream.of(fieldQuery, aggregateQuery, arithmeticQuery)
                    .filter(c->!StringExtension.IsNullOrEmpty(c))
                    .collect(Collectors.joining(",")));
        }
        if(limit>0)
        {
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" ,count(1) OVER() AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" ,FOUND_ROWS() AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
        }
        sb.append(Config.NewLine);
        sb.append(" FROM ");
        sb.append(tableName);
        if(jsonQuery.has($WHERE))
        {
            String where = buildWhereTSQL(columnList, jsonQuery.getJSONObject($WHERE));
            if(!StringExtension.IsNullOrEmpty(where))
            {
                sb.append(Config.NewLine);
                sb.append(" WHERE ");
                sb.append(where);
            }
        }
        if(!StringExtension.IsNullOrEmpty(groups))
        {
            sb.append(Config.NewLine);
            sb.append(String.format(" GROUP BY %s",groups));
        }
        if(jsonQuery.has($SORT))
        {
            sb.append(Config.NewLine);
            sb.append(buildSortTSQL(columnList, jsonQuery.getJSONObject($SORT)));
        }
        if(limit>0)
        {
            //add query pagination support
            sb.append(Config.NewLine);
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" LIMIT ").append(limit);
                sb.append(" OFFSET ").append(offset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" LIMIT ");
                sb.append(offset).append(",");
                sb.append(limit);
            }
        }
        return sb.toString();
    }
    //endregion

    private List<String> getInputColumnName(String name) {
        int subQueryIndex = name.indexOf(".");
        if(subQueryIndex>0)
        {
            return Arrays.asList(name.substring(0, subQueryIndex), name.substring(subQueryIndex + 1));
        }
        return Arrays.asList(name,"");
    }
    private JSONObject getQueryJSON(NameValuePair...params) {
        JSONObject jsonQuery = new JSONObject();
        for (NameValuePair item:params)
        {
            jsonQuery.put(item.id,item.value);
        }
        return jsonQuery;
    }
}
