package LinkFuture.Init.Extensions;

import LinkFuture.Init.Config;

import javax.crypto.*;
import javax.crypto.spec.DESedeKeySpec;
import javax.xml.bind.DatatypeConverter;
import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.spec.*;

/**
 * Created by
 * Cyokin on 5/4/2014.
 */
public class SecurityExtension {
    //region DES
    final static String defaultEncryptionSchema = "DESede";

    public static String DESEncrypt(String input,String encryptionKey)
    {
           return DESEncrypt(defaultEncryptionSchema,input,encryptionKey, Config.DefaultEncoding);
    }

    /**
     *    DESEncrypt
     * @param encryptionSchema  DES  or  DESede
     * @param input  input string
     * @param encryptionKey  key
     * @param encoding   utf-8
     * @return encrypted string
     */
    public static String DESEncrypt(String encryptionSchema,String input,String encryptionKey,String encoding)
    {
        try {
            SecretKeyFactory skf = SecretKeyFactory.getInstance(encryptionSchema);
            byte[]  arrayBytes = encryptionKey.getBytes(encoding);
            KeySpec ks = new DESedeKeySpec(arrayBytes);
            SecretKey key = skf.generateSecret(ks);
            Cipher cipher =Cipher.getInstance(encryptionSchema);
            cipher.init(Cipher.ENCRYPT_MODE, key);
            byte[] plainText = input.getBytes(encoding);
            byte[] encryptedText = cipher.doFinal(plainText);
            return DatatypeConverter.printBase64Binary(encryptedText);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return input;
    }

    public static String DESDecrypt(String input,String encryptionKey){
        return DESDecrypt(defaultEncryptionSchema,input,encryptionKey,Config.DefaultEncoding);
    }
    /**
     *    DESDecrypt
     * @param encryptionSchema  DES  or  DESede
     * @param input  input string
     * @param encryptionKey  key
     * @param encoding   utf-8
     * @return  original string
     */
    public static String DESDecrypt(String encryptionSchema,String input,String encryptionKey,String encoding)
    {
        try {
            SecretKeyFactory skf = SecretKeyFactory.getInstance(encryptionSchema);
            byte[]  arrayBytes = encryptionKey.getBytes(encoding);
            KeySpec ks = new DESedeKeySpec(arrayBytes);
            SecretKey key = skf.generateSecret(ks);

            Cipher cipher =Cipher.getInstance(encryptionSchema);
            cipher.init(Cipher.DECRYPT_MODE, key);
            byte[] encryptedText = DatatypeConverter.parseBase64Binary(input);
            byte[] plainText = cipher.doFinal(encryptedText);
            return new String(plainText,encoding);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return input;
    }
    //endregion

    //region RSA
    public static final String ALGORITHM = "RSA";

    //region GenerateKey
    public static void RSAGenerateKey(int keySize,File privateKeyFile,File publicKeyFile) throws NoSuchAlgorithmException, IOException {
        RSAKey key = RSAGenerateKey(keySize);
        // Saving the Public key in a file
        try(ObjectOutputStream publicKeyOS = new ObjectOutputStream(
                new FileOutputStream(publicKeyFile))){
            publicKeyOS.writeObject(key.getPublicKey());
        };
        // Saving the Private key in a file
        try(ObjectOutputStream privateKeyOS = new ObjectOutputStream(
                new FileOutputStream(privateKeyFile))){
            privateKeyOS.writeObject(key.getPrivateKey());
        }
    }

    /**
     * RSAGenerateKey
     * @param keySize   512 or 1024, depends on your encrypt string length
     * @return
     * @throws NoSuchAlgorithmException
     */
    public static RSAKey RSAGenerateKey(int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator generator = KeyPairGenerator.getInstance(ALGORITHM);
        generator.initialize(keySize);
        KeyPair pair = generator.generateKeyPair();
        return new RSAKey(pair.getPublic(),pair.getPrivate());
    }
    public static class RSAKey{

        private PublicKey publicKey;
        private PrivateKey privateKey;

        public PublicKey getPublicKey() {
            return publicKey;
        }

        public PrivateKey getPrivateKey() {
            return privateKey;
        }

        public RSAKey(PublicKey publicKey,PrivateKey privateKey)
        {
            this.publicKey = publicKey;
            this.privateKey = privateKey;
        }
        public String getBase64PublicKey(){
            return DatatypeConverter.printBase64Binary(this.publicKey.getEncoded());
        }
        public String getBase64PrivateKey(){
            return DatatypeConverter.printBase64Binary(this.privateKey.getEncoded());
        }
        public String getHexPublicKey(){
            return DatatypeConverter.printHexBinary(this.publicKey.getEncoded());
        }
        public String getHexPrivateKey(){
            return DatatypeConverter.printHexBinary(this.privateKey.getEncoded());
        }
        public String getModulusPublicKey() {
            try {
                RSAPublicKeySpec pub = KeyFactory.getInstance(ALGORITHM).getKeySpec(this.getPublicKey(), RSAPublicKeySpec.class);
                return pub.getModulus().toString(16);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public String toString() {
            return "Base64 Public Key:" + this.getBase64PublicKey() + System.lineSeparator()
                 + "Base64 Private Key:" + this.getBase64PrivateKey() + System.lineSeparator()
                 + "Hex Public Key:" + this.getHexPublicKey() + System.lineSeparator()
                 + "Hex Private Key:" + this.getHexPrivateKey() + System.lineSeparator()
                 + "Modulus Public Key:" + this.getModulusPublicKey() + System.lineSeparator()
                    ;
        }
    }
    //endregion

    //region Encrypt
    public static class Ciphertext{
        private byte[] encrypt;

        public byte[] getEncrypt() {
            return encrypt;
        }
        public String getBase64Text() {
            return DatatypeConverter.printBase64Binary(encrypt);
        }
        public String getHexText() {
            return DatatypeConverter.printHexBinary(encrypt);
        }
        public Ciphertext(byte[] encrypt){
            this.encrypt = encrypt;
        }

        @Override
        public String toString() {
            return    "Base64:" + this.getBase64Text() + System.lineSeparator()
                    + "Hex:" + this.getHexText() + System.lineSeparator()
                    ;
        }
    }

    public static PublicKey RSAGetPublicKey(byte[] publicKeyBinary) throws NoSuchAlgorithmException, InvalidKeySpecException {
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(publicKeyBinary);
        KeyFactory keyFact = KeyFactory.getInstance(ALGORITHM);
        return keyFact.generatePublic(x509KeySpec);
    }
    public static PublicKey RSAGetPublicKey(File publicKeyFile) throws NoSuchAlgorithmException, InvalidKeySpecException, IOException, ClassNotFoundException {
        ObjectInputStream inputStream = new ObjectInputStream(new FileInputStream(publicKeyFile));
        return  (PublicKey) inputStream.readObject();
    }

    public static PublicKey getPublicKeyFromBase64(String base64PublicKeyFile) throws IOException, ClassNotFoundException, InvalidKeySpecException, NoSuchAlgorithmException {
        return RSAGetPublicKey(DatatypeConverter.parseBase64Binary(base64PublicKeyFile));
    }
    public static PublicKey getPublicKeyFromHex(String hexPublicKey) throws IOException, ClassNotFoundException, InvalidKeySpecException, NoSuchAlgorithmException {
        return RSAGetPublicKey(DatatypeConverter.parseHexBinary(hexPublicKey));
    }

    public static Ciphertext RSAEncrypt(String text, PublicKey key) throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException, UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException {
        return RSAEncrypt(text.getBytes(Config.DefaultEncoding),key);
    }
    public static Ciphertext RSAEncrypt(byte[] text, PublicKey key) throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException, UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException {
        Cipher cipher = Cipher.getInstance(ALGORITHM);
        // encrypt the plain text using the public key
        cipher.init(Cipher.ENCRYPT_MODE, key);
        return new Ciphertext(cipher.doFinal(text));
    }
    //endregion

    //region Decrypt
    public static PrivateKey RSAGetPrivateKey(File privateKeyFile) throws IOException, ClassNotFoundException {
        ObjectInputStream inputStream = new ObjectInputStream(new FileInputStream(privateKeyFile));
        return (PrivateKey) inputStream.readObject();
    }
    public static PrivateKey RSAGetPrivateKey(byte[] privateKeyBinary) throws IOException, ClassNotFoundException, InvalidKeySpecException, NoSuchAlgorithmException {
        PKCS8EncodedKeySpec x509KeySpec = new PKCS8EncodedKeySpec(privateKeyBinary);
        KeyFactory keyFact = KeyFactory.getInstance(ALGORITHM);
        return keyFact.generatePrivate(x509KeySpec);
    }
    public static PrivateKey RSAGetPrivateKeyFromBase64(String base64PrivateKey) throws IOException, ClassNotFoundException, InvalidKeySpecException, NoSuchAlgorithmException {
        return RSAGetPrivateKey(DatatypeConverter.parseBase64Binary(base64PrivateKey));
    }
    public static PrivateKey RSAGetPrivateKeyFromHex(String hexPrivateKey) throws IOException, ClassNotFoundException, InvalidKeySpecException, NoSuchAlgorithmException {
        return RSAGetPrivateKey(DatatypeConverter.parseHexBinary(hexPrivateKey));
    }

    public static String RSADecryptFromBase64(String base64Text, PrivateKey key) throws UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException, NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        return RSADecrypt(DatatypeConverter.parseBase64Binary(base64Text),key);
    }
    public static String RSADecryptFromHex(String hexText,PrivateKey key) throws UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException, NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        return RSADecrypt(DatatypeConverter.parseHexBinary(hexText), key);
    }
    public static String RSADecrypt(byte[] text, PrivateKey key) throws UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException, NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        final Cipher cipher = Cipher.getInstance(ALGORITHM);
        // decrypt the text using the private key
        cipher.init(Cipher.DECRYPT_MODE, key);
        return  new String(cipher.doFinal(text),Config.DefaultEncoding);
    }

    public static String RSADecrypt(BigInteger text, PrivateKey key) throws UnsupportedEncodingException, BadPaddingException, IllegalBlockSizeException, NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        final Cipher cipher = Cipher.getInstance(ALGORITHM);
        // decrypt the text using the private key
        cipher.init(Cipher.DECRYPT_MODE, key);
        return  new String(cipher.doFinal(text.toByteArray()),Config.DefaultEncoding);
    }
    //endregion

    //endregion
}
