    package LinkFuture.Core.WebClient;


import LinkFuture.Init.Debugger;
import LinkFuture.Init.Config;
import LinkFuture.Init.ConfigurationManager.ConfigurationController;
import LinkFuture.Init.Extensions.StringExtension;
import LinkFuture.Init.ObjectExtend.NameValuePair;
import LinkFuture.Init.Utility;
import sun.net.www.MessageHeader;

import javax.net.ssl.*;
import javax.xml.ws.WebServiceException;
import java.io.*;
import java.lang.reflect.Field;
import java.net.*;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;

/**
 * User: Cyokin Zhang
 * Date: 9/27/13
 * Time: 7:32 PM
 */
public class WebClient {
    //allow https connection without install certification
    private static Field requestField;
    private static Field requestKeysField;
    private static Field requestValuesField;
    private static final String AuthorizationHeader = "Authorization";
    private static final int MaxRedirection = 10;
    private static final GroupAuthenticator GlobalAuthenticator = new GroupAuthenticator();

    static {
        AllowHttpsConnection();
        AppendCookie();
    }
    private synchronized static void AppendCookie(){
        boolean enableCookie = ConfigurationController.AppSettings("EnableCookie").equalsIgnoreCase("true");
        if(enableCookie)
        {
           CookieManager cm = new CookieManager();
           cm.setCookiePolicy(CookiePolicy.ACCEPT_ALL);
           CookieHandler.setDefault(cm);
        }
    }
    private synchronized static void AppendAuthentication(String host){
        if(Authenticator.requestPasswordAuthentication(host, null, 80,  "", "", "",null, Authenticator.RequestorType.SERVER)==null)
        {
            Debugger.LogFactory.trace("Append global authentication");
            Authenticator.setDefault(GlobalAuthenticator);
        }
    }
    private synchronized static void AllowHttpsConnection(){
        try
        {
            TrustManager[] trustAllCerts = new TrustManager[] { new X509TrustManager() {
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                    return null;
                }
                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                }
                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                }
            } };
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
            // Create all-trusting host name verifier
            HostnameVerifier allHostsValid = new HostnameVerifier() {
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            };
            // Install the all-trusting host verifier
            HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
        }
        catch (Exception ex)
        {
            Debugger.LogFactory.warn("Allow HttpsConnection",ex);
        }
    }

    HttpURLConnection con = null;
    public volatile boolean isConnected = false;
    private DataOutputStream outputStream = null;
    private Thread interruptURLThread = null;
    private WebRequestInfo requestMeta;
    private static final String boundary =  "7ddd51836076e";
    private static final String twoHyphens = "--";
    private static final String crlf = "\r\n";
    private static final String fieldSeperated = twoHyphens + boundary + crlf;
    private WebClient(WebRequestInfo requestMeta){
         this.requestMeta = requestMeta;
    }
    private WebClientResultInfo Send(){
        WebClientResultInfo result = new WebClientResultInfo();
        try {
            this.connect(requestMeta.RequestURL);
            result.code = con.getResponseCode();
            result.errorMessage = con.getResponseMessage();
            //redirection
            if(this.requestMeta.AutoRedirect)
            {
                int redirectCount = 0;
                while (result.code == HttpURLConnection.HTTP_MOVED_TEMP
                        || result.code == HttpURLConnection.HTTP_MOVED_PERM
                        || result.code == HttpURLConnection.HTTP_SEE_OTHER )
                {
                    String newUrl = con.getHeaderField("Location");
                    this.close();
                    Debugger.LogFactory.trace("Redirect to {}", newUrl);
                    this.connect(new URL(newUrl));
                    result.code = con.getResponseCode();
                    result.errorMessage = con.getResponseMessage();
                    redirectCount ++;
                    if(redirectCount> MaxRedirection)
                    {
                        throw new WebServiceException("Reached max redirection times," + MaxRedirection);
                    }
                }
            }
            InputStream replyStream = null;
            //  200 <= code < 300
            boolean isSuccess = result.code>=HttpURLConnection.HTTP_OK && result.code < HttpURLConnection.HTTP_MULT_CHOICE;
            if(isSuccess)
            {
                 replyStream = con.getInputStream();
            }
            else
            {
                replyStream = con.getErrorStream();
            }
            final InputStreamReader inputStreamReader;
            final String contentEncodingField = con.getContentEncoding();
            //read header
            result.HeaderFields = new HashMap<>();
            for (Map.Entry<String, List<String>> header : con.getHeaderFields().entrySet()) {
                result.HeaderFields.put(header.getKey(),header.getValue().toString());
            }
            //remove replyStream.available() > 0, as it is an estimated number of bytes, a weak guarantee that it is not very useful in practice, according to API doc.
            if(replyStream!=null)
            {
                String charset = getCharset();
                if (contentEncodingField != null && contentEncodingField.equalsIgnoreCase("gzip")) {
                    GZIPInputStream gzipInputStream =  new GZIPInputStream(replyStream);
                    inputStreamReader = new InputStreamReader(gzipInputStream,charset);
                } else {
                    inputStreamReader = new InputStreamReader(replyStream,charset);
                }
                result.response = Utility.read(inputStreamReader);
            }
            else
            {
                result.response = Config.Empty;
            }
            result.success =  isSuccess;
        }
        catch (Exception e) {
            Debugger.fatal("Web client exception",e);
            if(result.code==0)
            {
                result.code = HttpURLConnection.HTTP_INTERNAL_ERROR;
            }
            if(isConnected)
            {
                result.errorMessage = e.toString();
            }
            else
            {
                result.errorMessage = "java.net.SocketTimeoutException: Timed out";
            }
            if(StringExtension.IsNullOrEmpty(result.errorMessage)){
                String error = Debugger.getDetail(e);
                result.errorMessage = error.substring(0,error.indexOf(Config.NewLine));
            }
        }
        this.close();
        return result;
    }
    private String getCharset(){
        String contentType = con.getContentType();
        if(contentType==null)
        {
               return Config.DefaultEncoding;
        }
        String[] values = contentType.split(";"); //The values.length must be equal to 2...
        String charset = null;
        for (String value : values) {
            value = value.trim();
            if (value.toLowerCase().startsWith("charset=")) {
                charset = value.split("=", 2)[1];
            }
        }
       return StringExtension.IsNullOrEmpty(charset)?Config.DefaultEncoding:charset;
    }
    private void connect(URL url) throws IOException {
        isConnected = true;
        con = (HttpURLConnection) url.openConnection();
        if(requestMeta.ReadTimeout>0)
        {
            con.setReadTimeout(requestMeta.ReadTimeout);
        }
        if(requestMeta.ConnectionTimeout>0)
        {
            con.setConnectTimeout(requestMeta.ConnectionTimeout);
        }
        this.registerAuthentication();
        //append header
        if(this.requestMeta.RequestHeadList!=null)
        {
            for (Map.Entry<String,Object> item:requestMeta.RequestHeadList.entrySet())
            {
                con.setRequestProperty(item.getKey(), item.getValue().toString());
            }
        }
        if(requestMeta.ReadTimeout >0 || requestMeta.ConnectionTimeout>0)
        {
            //use thread to force disconnect, as for somehow default timeout not working on unix.
            interruptURLThread = new Thread(new WebClientInterruptThread(this));
            interruptURLThread.start();
        }
        writeBytes();
    }

    //use saved authorization to avoid round trip, design for digest and basic authorization;
    private void registerAuthentication() {
        if(requestMeta.Credential!=null)
        {
            String host = this.requestMeta.RequestURL.getHost();
            String ticket = GroupAuthenticator.RegisterAuthentication(host,requestMeta.Credential);
            AppendAuthentication(host);
            if(!StringExtension.IsNullOrEmpty(ticket) && !this.requestMeta.RequestHeadList.containsKey(AuthorizationHeader)){
                this.requestMeta.addHead(AuthorizationHeader,ticket);
            }
        }
    }
    //save digest authorization value, for use it next time with some host;
    private void savingAuthorizationHeader() {
        if(requestMeta.Credential!=null && requestMeta.Credential.RequestAuthType == AuthType.Digest )
        {
            try {
                if(requestField==null)
                {
                    try {
                        requestField = con.getClass().getDeclaredField("requests");
                        requestField.setAccessible(true);
                        Class message = MessageHeader.class;
                        requestKeysField = message.getDeclaredField("keys");
                        requestKeysField.setAccessible(true);
                        requestValuesField = message.getDeclaredField("values");
                        requestValuesField.setAccessible(true);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                if(requestField!=null && requestKeysField!=null && requestValuesField!=null)
                {
                    MessageHeader requestHeader = (MessageHeader)requestField.get(con);
                    String[] keys = (String[])requestKeysField.get(requestHeader);
                    String host = this.requestMeta.RequestURL.getHost();
                    for (int i=0;i<keys.length;i++)
                    {
                        if(keys[i]!=null && keys[i].equalsIgnoreCase(AuthorizationHeader))
                        {
                            String[] values = (String[])requestValuesField.get(requestHeader);
                            GroupAuthenticator.SetAuthorizationTicket(host,values[i]);
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void close()  {
        //this.savingAuthorizationHeader();
        if(interruptURLThread!=null)
        {
            interruptURLThread.interrupt();
            interruptURLThread = null;
        }
        if(outputStream !=null)
        {
            try {
                outputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            outputStream = null;
        }
        if(con!=null)
        {
            isConnected = false;
            con.disconnect();
            con = null;
        }
    }

    private String buildFormPostString() throws UnsupportedEncodingException {
        if(requestMeta.PostStringList.size() > 0)
        {
            StringBuilder sb = new StringBuilder();
            for (Map.Entry<String,Object> item:requestMeta.PostStringList.entrySet())
            {
                sb.append(fieldSeperated);
                sb.append("Content-Disposition: form-data; name=\"" + item.getKey() + "\"");
                sb.append(crlf);
                sb.append(crlf);
                sb.append(item.getValue());
                sb.append(crlf);
            }
            return sb.toString();
        }
        return null;
    }
    private String buildUrlEncodingPostString() throws UnsupportedEncodingException {
        if(requestMeta.PostStringList.size() > 0)
        {
            StringBuilder sb = new StringBuilder();
            int i=0;
            for (Map.Entry<String,Object> item:requestMeta.PostStringList.entrySet())
            {
                sb.append(String.format("%s=%s", item.getKey(), URLEncoder.encode(item.getValue().toString(), Config.DefaultEncoding)));
                i++;
                if(i<requestMeta.PostStringList.size())
                {
                    sb.append("&");
                }
            }
            return sb.toString();
        }
        return null;
    }
    private void writeBytes() throws IOException {
        switch (requestMeta.RequestMethod)
        {
            case Get:
            case Delete:
                con.setRequestMethod(requestMeta.RequestMethod.toString().toUpperCase());
                con.setUseCaches(requestMeta.UseCaches);
                Debugger.LogFactory.trace("{} data from {}", requestMeta.RequestMethod,requestMeta.RequestURL);
                break;
            case Post:
            case Put:
                con.setRequestMethod(requestMeta.RequestMethod.toString().toUpperCase());
                con.setUseCaches(false);
                con.setRequestProperty("Cache-Control", "no-cache");
                boolean hasPayload = !StringExtension.IsNullOrEmpty(requestMeta.Payload);
                boolean hasPostData = requestMeta.PostStringList!=null && requestMeta.PostStringList.size()>0 || hasPayload;
                boolean hasPostFile =  requestMeta.PostFileList!=null && requestMeta.PostFileList.size()>0;
                //do we have data to post?
                if(hasPostData || hasPostFile)
                {
                    //open write
                    con.setDoOutput(true);
                    con.setDoInput(true);
                    //we only have string to post, then use application/x-www-form-urlencoded
                    //check http://en.wikipedia.org/wiki/POST_(HTTP)
                    if(hasPostData && !hasPostFile && requestMeta.EncType==FormContentTypes.UrlEncoded)
                    {
                        byte[] postString = null;
                        if(hasPayload)
                        {
                            postString = requestMeta.Payload.getBytes(Config.DefaultEncoding);
                        }
                        else
                        {
                            con.setRequestProperty("Content-Type",FormContentTypes.UrlEncoded.toString());
                            postString = buildUrlEncodingPostString().getBytes(Config.DefaultEncoding);
                        }
                        con.setRequestProperty("Content-Length",String.valueOf(postString.length));
                        outputStream = new DataOutputStream(con.getOutputStream());
                        outputStream.write(postString);
                    }
                    else
                    {
                        //looks we got file to post, we need use multipart/form-data
                        //check http://www.w3.org/TR/html401/interact/forms.html
                        con.setRequestProperty("Content-Type",FormContentTypes.FormData.toString().concat(";boundary="+ boundary));
                        outputStream = new DataOutputStream(con.getOutputStream());
                        if(hasPostData)
                        {
                            outputStream.writeBytes(buildFormPostString());
                        }
                        for (Map.Entry<String,WebRequestFileInfo> item:requestMeta.PostFileList.entrySet())
                        {
                            WebRequestFileInfo fileInfo = item.getValue();
                            outputStream.writeBytes(fieldSeperated);
                            outputStream.writeBytes("Content-Disposition:form-data;name=\"" + fileInfo.Name + "\";filename=\"" + fileInfo.FileName + "\"" + crlf);
                            if(StringExtension.IsNullOrEmpty(fileInfo.ContentType))
                            {
                                outputStream.writeBytes("Content-Type: application/octet-stream");
                            }
                            else
                            {
                                outputStream.writeBytes("Content-Type: " + fileInfo.ContentType);
                            }
                            outputStream.writeBytes(crlf + "Content-Transfer-Encoding: binary");
                            outputStream.writeBytes(crlf);
                            outputStream.writeBytes(crlf);
                            //write stream
                            int nRead;
                            byte[] data = new byte[1024];
                            while ((nRead = fileInfo.FileStream.read(data, 0, data.length)) != -1) {
                                outputStream.write(data,0,nRead);
                            }
                            outputStream.writeBytes(crlf);
                        }
                        outputStream.writeBytes(twoHyphens + boundary + twoHyphens + crlf);
                    }
                    outputStream.flush();
                }
                else
                {
                    throw new IllegalArgumentException("Must append data to post");
                }
                //con.setRequestProperty("Content-Language", "en-US");
                Debugger.LogFactory.trace("{} data to {}", requestMeta.RequestMethod,requestMeta.RequestURL);
                break;

        }

    }


    //region  SendRequest
    public static WebClientResultInfo sendRequest(WebRequestInfo requestMeta){
        WebClient client = new WebClient(requestMeta);
        return client.Send();
    }

    /**
     *   Send Request
     * @param requestMeta
     * @param retryTimes must >=1
     * @return
     */
    public static WebClientResultInfo sendRequest(WebRequestInfo requestMeta,int retryTimes)
    {
        //at least once
        if(retryTimes<1) retryTimes=1;
        WebClientResultInfo result = null;
        for (int i=0;i<retryTimes;i++)
        {
            result = sendRequest(requestMeta);
            if(result.success)
            {
                return result;
            }
        }
        return result;
    }
    public static WebClientResultInfo sendRequest(URL url,HttpMethod method, ArrayList<NameValuePair> postData) throws UnsupportedEncodingException {
        WebRequestInfo requestMeta = new WebRequestInfo();
        requestMeta.RequestURL = url;
        requestMeta.RequestMethod = method;
        if(requestMeta.RequestMethod==HttpMethod.Post && postData !=null)
        {
            for (NameValuePair item:postData)
            {
                requestMeta.addPostData(item.id,item.value);
            }
        }
        return sendRequest(requestMeta);
    }
    public static WebClientResultInfo sendRequest(String url,HttpMethod method, ArrayList<NameValuePair> postData) throws UnsupportedEncodingException, MalformedURLException {
        return sendRequest(BuildUrl(url),method,postData);
    }
    //endregion

    //region Post
    public static WebClientResultInfo post(URL url, ArrayList<NameValuePair> postData) throws UnsupportedEncodingException {
        return sendRequest(url,HttpMethod.Post,postData);
    }
    public static WebClientResultInfo post(String url, ArrayList<NameValuePair> postData) throws UnsupportedEncodingException, MalformedURLException {
        return sendRequest(BuildUrl(url),HttpMethod.Post,postData);
    }
    //endregion

    private static URL BuildUrl(String url) throws MalformedURLException
    {
        //auto append http protocal
        if(url.startsWith("//"))
        {
            url = "http:" + url;
        }
        return new URL(url);
    }

    //region Get
    public static WebClientResultInfo get(String url) throws MalformedURLException {
        return get(BuildUrl(url));
    }
    public static WebClientResultInfo get(URL url)  {
        WebRequestInfo requestMeta = new WebRequestInfo();
        requestMeta.RequestURL = url;
        requestMeta.RequestMethod = HttpMethod.Get;
        return sendRequest(requestMeta);
    }
    //endregion
}
